/* view.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1995 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

void view_expunge_messages();
void view_after_expunge();
void view_bye();
void view_select_messages();
void view_select_new_messages();
void view_unselect_messages();
void view_set_select_messages();
void view_read_message();
void view_read_selected();
void view_destroy();
void view_next();
void view_prev();
void view_context_switch();
void view_close_mailbox();
void view_check_mailbox();
void view_lview_select();
void view_lview_accept();
void view_lview_dismiss();
void view_lview_destroy();
void view_lview_help();
void stuff_lview_list();
void view_lview_window();
void view_copy();
void view_move();
void view_check_flags();
void view_printfull();
void view_printpart();
void view_printnone();
void view_print_dispatch();
void view_set_deleted();
void view_set_seen();
void view_set_answered();
void view_set_flagged();
void view_clear_deleted();
void view_clear_seen();
void view_clear_answered();
void view_clear_flagged();
void view_setflag();
void view_clearflag();
void view_filter();
void view_save_dispatch();
void view_savefull();
void view_savepart();
void view_savenone();
void view_execfull();
void view_execpart();
void view_execnone();
void view_exec_dispatch();
void view_reply_sender();
void view_reply_all();
void view_forward();
void view_forward_attach();
void view_remail();
void view_lview_detach();
void view_lview_save();
void view_lview_hide();
void view_lview_show();
void view_do_actions();
void view_action_help();
void view_window_help();

char * view_fetch();

int message_list_compare();
void update_logical_views();
void do_the_search();
void update_lview_message_list();

char *get_keyword();
void keyword_select();
void keyword_accept();
void keyword_cancel();
void keyword_help();
void view_set_keyword();
void view_unset_keyword();
void keyword_destroy();


Menu view_save_menu[] = {
  { "Full Header", "view_savefull", 'F',
      view_savefull, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Part Header", "view_savepart", 'P',
      view_savepart, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "  No Header", "view_savenone", 'N',
      view_savenone, NULL, 0, NULL, NULL, BTN_NOSELECTION },
};


Menu view_pipe_menu[] = {
  { "Full Header", "view_execfull", 'F',
      view_execfull, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Part Header", "view_execpart", 'P',
      view_execpart, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "  No Header", "view_execnone", 'N',
      view_execnone, NULL, 0, NULL, NULL, BTN_NOSELECTION },


};


Menu view_print_menu[] = {

  { "Full Header", "view_printfull", 'F',
      view_printfull, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Part Header", "view_printpart", 'P',
      view_printpart, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "  No Header", "view_printnone", 'N',
      view_printnone, NULL, 0, NULL, NULL, BTN_NOSELECTION },

};




Menu view_file_menu[] = {

  { "Read Messages", "view_read_selected", 'R', 
      view_read_selected, NULL, 0, NULL, NULL, BTN_NOSELECTION },

  { "Copy to Mailbox", "view_copy", 'C', 
      view_copy, NULL, 0, NULL, NULL, BTN_NOSELECTION|BTN_ISNEWS },

  { "Move to Mailbox", "view_move", 'M', 
      view_move, NULL, 0, NULL, NULL, BTN_NOSELECTION|BTN_ISNEWS },

  { "Print Messages", "view_print", 'P', 
      NULL, view_print_menu, XtNumber(view_print_menu), 
      NULL, NULL, BTN_NOSELECTION },

  { "Save to File", "view_save_menu", 'S',
      NULL, view_save_menu, XtNumber(view_save_menu), 
      NULL, NULL, BTN_NOSELECTION },

  { "Pipe | Command", "view_pipe_menu", '|',
      NULL, view_pipe_menu, XtNumber(view_pipe_menu), 
      NULL, NULL, BTN_NOSELECTION },

  { "Address_Book", "view_address_book", 'A',
      main_address_book, NULL, 0, NULL, NULL, BTN_ON },


  { "Close Window", "view_bye", 'W', 
      view_bye,   NULL, 0, NULL, NULL, BTN_ON },

};


Menu view_setflags_menu[] = {

  { "Deleted", "view_set_deleted", 'D',
      view_set_deleted, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Seen", "view_set_seen", 'S',
      view_set_seen, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Flagged", "view_set_flagged", 'F',
      view_set_flagged, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Answered", "view_set_answered", 'A',
      view_set_answered, NULL, 0, NULL, NULL, BTN_NOSELECTION },

};


Menu view_clearflags_menu[] = {

  { "Deleted", "view_clear_deleted", 'D',
      view_clear_deleted, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Seen", "view_clear_seen", 'S',
      view_clear_seen, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Flagged", "view_clear_flagged", 'F',
      view_clear_flagged, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Answered", "view_clear_answered", 'A',
      view_clear_answered, NULL, 0, NULL, NULL, BTN_NOSELECTION },

};



Menu view_edit_menu[] = {

  { "Select All Messages", "view_select_messages", 'A',
      view_select_messages, NULL, 0, NULL, NULL, BTN_ON },

  { "Select New Messages", "view_select_new_messages", 'N',
      view_select_new_messages, NULL, 0, NULL, NULL, BTN_ON },

  { "Unselect All Messages", "view_unselect_messages", 'U',
      view_unselect_messages, NULL, 0, NULL, NULL, BTN_NOSELECTION },

  /* 
   * Redundant, yes, but let's save a couple keystrokes
   * for this common operation.
   */

  { "Delete", "view_set_deleted", 'D',
      view_set_deleted, NULL, 0, NULL, NULL, BTN_NOSELECTION },

  { "Set Flags", "view_setflags_menu", 'S',
      NULL, view_setflags_menu, XtNumber(view_setflags_menu), 
      NULL, NULL, BTN_NOSELECTION },

  { "Clear Flags", "view_clearflags_menu", 'C',
      NULL, view_clearflags_menu, XtNumber(view_clearflags_menu),
      NULL, NULL, BTN_NOSELECTION },

  { "Set Keyword", "view_set_keyword", 'K',
      view_set_keyword, NULL, 0, NULL, NULL, 
      BTN_NOKEYWORDS|BTN_NOSELECTION },

  { "Unset Keyword", "view_unset_keyword", 'U',
      view_unset_keyword, NULL, 0, 
      NULL, NULL, BTN_NOKEYWORDS|BTN_NOSELECTION },

};


Menu view_options_menu[] = {
  { "Filters", "view_filter", 'F', 
      view_filter, NULL, 0, NULL, NULL, BTN_ON },

};


Menu view_reply_submenu[] = {

  { "Sender", "view_reply_sender", 'S', 
      view_reply_sender, NULL, 0, NULL, NULL, BTN_ON },

  { "All", "view_reply_all", 'A',
      view_reply_all, NULL, 0, NULL, NULL, BTN_ON },
};


Menu view_forward_submenu[] = {
  { "Insert", "view_forward", 'I', 
      view_forward, NULL, 0, NULL, NULL, BTN_ON },

  { "Attach", "view_forward_attach", 'A',
      view_forward_attach, NULL, 0, NULL, NULL, BTN_ON },
};



Menu view_send_menu[] = {

  { "Compose New Message", "VIEW_COMPOSE", 'N', 
      main_compose, NULL, 0, NULL, NULL, BTN_ON },

  { "Reply to ...", "VIEW_REPLY", 'R', 
      NULL, view_reply_submenu, XtNumber(view_reply_submenu), 
      NULL, NULL, BTN_NOSELECTION },

  { "Forward", "VIEW_FORWARD", 'F',
      NULL, view_forward_submenu, XtNumber(view_forward_submenu),
      NULL, NULL, BTN_NOSELECTION },

  { "Remail", "VIEW_REMAIL", 'm',
      view_remail, NULL, 0, NULL, NULL, BTN_NOSELECTION },

};


Menu view_mailbox_menu[] = {

  { "Check for New Mail", "view_check_mailbox", 'N',
     view_check_mailbox, NULL, 0, NULL, NULL, BTN_ON },

  { "Expunge Deleted Messages", "view_expunge_messages", 'E', 
      view_expunge_messages, NULL, 0, NULL, NULL, BTN_ON },

  { "Open Another Mailbox", "view_open_mailbox", 'O',
      main_open, NULL, 0, NULL, NULL, BTN_ON },

  { "Close This Mailbox", "view_close_mailbox", 'C', 
      view_close_mailbox,   NULL, 0, NULL, NULL, BTN_ON },

};


Menu view_help_menu[] = {
  { "Window", "view_window_help", 'W',
      view_window_help, NULL, 0, NULL, NULL, BTN_ON },

  { "Actions", "view_action_help", 'A', 
      view_action_help, NULL, 0, NULL, NULL, BTN_ON },
};

Menu view_menu[] = {

  { "File", "VIEW_FILE_MENU", 'F', 
      NULL, view_file_menu, XtNumber(view_file_menu), 
      NULL, NULL, BTN_ON },

  { "Edit", "VIEW_EDIT_MENU", 'E', 
      NULL, view_edit_menu, XtNumber(view_edit_menu), 
      NULL, NULL, BTN_ON },

  { "Options", "VIEW_OPTIONS_MENU", 'O',
      NULL, view_options_menu, XtNumber(view_options_menu), 
      NULL, NULL, BTN_ON },

  { "Send", "VIEW_SEND_MENU", 'S',
      NULL, view_send_menu, XtNumber(view_send_menu), 
      NULL, NULL, BTN_ON },

  { "Mailbox", "view_mailbox_menu", 'M',
     NULL, view_mailbox_menu, XtNumber(view_mailbox_menu),
     NULL, NULL, BTN_ON },

  { "<", "view_next", '<',
      view_next, NULL, 0, NULL, NULL, BTN_NONEXT },

  { "?", "view_lview_window", '?',
      view_lview_window, NULL, 0, NULL, NULL, BTN_ON },

  { ">", "view_prev", '>',
      view_prev, NULL, 0, NULL, NULL, BTN_NOPREV },


  { "Help", "VIEW_HELP!_MENU", 'H', 
      NULL, view_help_menu, XtNumber(view_help_menu), 
      NULL, NULL, BTN_ON },


};


void create_view_window()
{

  Arg args[ARGLISTSIZE];
  int n = 0;
  View *view;
  Widget menubar, action_button;
  XtTranslations translations;

  if(session.view == NULL) {
    view = new_view();
    view->current = NULL;
    view->lview = NULL;
    session.view = view;
  }
  else
    view = session.view;

  if(view->is_realized == TRUE) {
    XRaiseWindow(display,XtWindow(view->shell));
    return;
  }

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  view->shell = XtCreateWidget("view_shell",
				      topLevelShellWidgetClass,
				      top, args, n ); n = 0;

  view->form = XmCreateForm(view->shell, "form", args, n); n = 0;
  XtManageChild(view->form);
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(view->form,"menubar", args, n); n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar, view_menu, XtNumber(view_menu), 
		      BTN_ON|BTN_NOSELECTION, view);


  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n ++;
  XtSetArg(args[n], XmNtopWidget, menubar);             n ++;

  action_button = XmCreatePushButton(view->form,"Action:", args, n); n = 0;
  XtAddCallback(action_button,XmNactivateCallback,
		(XtCallbackProc) view_do_actions, view);
  XtManageChild(action_button);

  XtSetArg(args[n], XmNtopAttachment,   XmATTACH_WIDGET ); n ++;
  XtSetArg(args[n], XmNtopWidget,       menubar         ); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM   ); n ++;
  XtSetArg(args[n], XmNleftAttachment,  XmATTACH_WIDGET ); n ++;
  XtSetArg(args[n], XmNleftWidget,      action_button   ); n ++;
  XtSetArg(args[n], XmNcolumns,         72              ); n ++;

  view->action = XmCreateTextField(view->form,"action",args,n); n = 0;

  XtManageChild(view->action);

  translations = XtParseTranslationTable(GLOBAL_text_field_translations);
  XtOverrideTranslations(view->action,translations);

  translations = 
    XtParseTranslationTable(GLOBAL_terminal_text_field_translations);
  XtOverrideTranslations(view->action,translations);  
  XtAddCallback(view->action, XmNactivateCallback, 
		(XtCallbackProc) view_do_actions, view);

  XtAddCallback(view->action, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_field_edit, NULL);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, view->action); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n ++;
  XtSetArg(args[n], XmNvisibleItemCount, preferences.view_height ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmEXTENDED_SELECT); n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmCONSTANT); n ++;


  view->list = XmCreateScrolledList(view->form,"list",args,n); n = 0;

  XtAddCallback(view->list, XmNextendedSelectionCallback,
		view_set_select_messages, view);

  XtAddCallback(view->list, XmNdefaultActionCallback,
		view_read_message, view);


  XtAddCallback(view->shell, XmNdestroyCallback,
		view_destroy, view);


  XtManageChild(view->list);
  XtManageChild(view->shell);
  XtRealizeWidget(view->shell);

  view->is_realized = TRUE;

}


Widget new_list_widget(view) 
     View *view;
{

  Arg args[ARGLISTSIZE];
  int n = 0;
  
  Widget list;

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, view->action); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n ++;
  XtSetArg(args[n], XmNvisibleItemCount, preferences.view_height ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmEXTENDED_SELECT); n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmCONSTANT); n ++;


  list = XmCreateScrolledList(view->form,"list",args,n); n = 0;

  XtAddCallback(list, XmNextendedSelectionCallback,
		view_set_select_messages, view);

  XtAddCallback(list, XmNdefaultActionCallback,
		view_read_message, view);

  return(list);

}


void view_action_help(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  help(view->shell,"Actions", "action.help");

}

void view_window_help(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  help(view->shell,"View Window", "viewwin.help");

}


void view_do_actions(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  char **parsed_actions;
  char *raw_action = NULL;
  char *clean_action = NULL;
  int n;
  char * ptr;
  ACTIONREC * a;
  HEADERREC * h;
  Lview *lview;

  char warn[FILEBUFFLEN];

  lview = view->current;

  raw_action = GetTextField(view->action);

  if((lview == NULL) || (raw_action == NULL))
    return;

  if(*raw_action == NUL_TERM) {
    fs_give((void **) &raw_action);
    return;
  }

  clean_action = clean_and_parse_action(raw_action);
  raw_action = NULL;

  parsed_actions = get_actions(clean_action);
  fs_give((void **) &clean_action);

  if(parsed_actions == NULL)
    return;


  for(n = 0; parsed_actions[n]; n ++ ) {

    if(lview->script_arg)
      fs_give((void **) &lview->script_arg);

    lview->script_arg = NULL;

    a = find_action(parsed_actions[n]);

    if(a == NULL) {
      sprintf(warn,"No such action as %s\n",parsed_actions[n]);
      mm_log(warn,WARN);
      break;
    }

    ptr = &parsed_actions[n][strlen(a->name) + 1];
    if(a->option == TRUE) {
      h = get_hdrlist(ptr);
      if(h != NULL) {
	lview->script_header = h->hdrtype;
	ptr += strlen(h->name) + 1;
      }
    }

    if(a->arg == TRUE)
      lview->script_arg = cpystr(ptr);

    lview->scripting = TRUE;

    switch(a->action) {

    case DELETE_ACT:
      view_set_deleted(w,view,xp);
      break;
    case EXPUNGE_ACT:
      view_expunge_messages(w,view,xp);
      break;
    case COPY_ACT:
      view_copy(w,view,xp);
      break;
    case MOVE_ACT:
      view_move(w,view,xp);
      break;
    case SAVE_ACT:
      view_save_dispatch(w,view,HEADER_UNKNOWN);
      break;
    case PRINT_ACT:
      view_print_dispatch(w,view,HEADER_UNKNOWN);
      break;
    case SHELL_ACT:
      view_exec_dispatch(w,view,HEADER_UNKNOWN);
      break;
    case SELECT_ACT:
      view_select_messages(w,view,xp);
      break;
    case NEW_ACT:
      view_select_new_messages(w,view,xp);
      break;
    case UNSELECT_ACT:
      view_unselect_messages(w,view,xp);
      break;
    case FLAG_ACT:
      view_set_flagged(w,view,xp);
      break;
    case READ_ACT:
      view_read_selected(w,view,xp);
      break;
      
    default:
      sprintf(warn,"Unimplemented action: %s\n",a->name);
      mm_log(warn,WARN);

      break;

    }

    lview->scripting = FALSE;
    continue;
  }

  free_actions(parsed_actions);

}

void view_set_select_messages(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{

  unsigned long n = 0;
  Message_List *message_list;

  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    if((is_selected(xp, message_list->number)) == TRUE)
      message_list->selected = TRUE;
    else
      message_list->selected = FALSE;
    if(message_list->selected == TRUE)
      n ++;
  }

  view->current->num_selected = n;
  if(n) 
    view->current->button_state &= ~(BTN_NOSELECTION);
  else
    view->current->button_state |= BTN_NOSELECTION;

  check_buttons(view_menu,XtNumber(view_menu), view->current->button_state);

}

void view_check_mailbox(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{
  Mailbox *mailbox;

  for(mailbox = session.mailboxes; mailbox; mailbox = mailbox->next) {
    if(mailbox->mailstream == view->current->mailstream) {
      check_mailbox(mailbox);
      break;
    }
  }
}

void view_read_message(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{

  Message_List *message_list;

  view->current->num_selected = 0L;
  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    message_list->selected = FALSE;
    if(message_list->number == xp->item_position) {
      set_watch_cursors();
      message_list->selected = TRUE;
      view->current->num_selected = 1L;
      view->current->button_state &= ~(BTN_NOSELECTION);
      check_buttons(view_menu, XtNumber(view_menu), 
		    view->current->button_state);
      view_read_selected(w,view,xp);
      break;
    }
  }
}

void view_read_selected(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{

  if(view->current->num_selected) {
    set_watch_cursors();
    read_messagelist(view->current->message_list);
  }
}


void view_next(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{

  if(view->current->next)
    view_context_switch(view,view->current,view->current->next);

}


void view_prev(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{

  if(view->current->prev)
    view_context_switch(view,view->current,view->current->prev);

}


void view_expunge_messages(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  set_watch_cursors();
  mail_expunge(view->current->mailstream);
  view_after_expunge();
  reset_cursors();

}


void view_select_messages(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Message_List *message_list;
  unsigned long selected = 0L;


  view_unselect_messages(w,view,xp);

  set_watch_cursors();

  XtSetArg(args[n], XmNselectionPolicy, XmMULTIPLE_SELECT);    n ++;
  XtSetValues(view->list,args,n );                             n = 0;

  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    XmListSelectPos(view->list,message_list->number,FALSE);
    message_list->selected = TRUE;
    selected ++;
  }

  XtSetArg(args[n], XmNselectionPolicy, XmEXTENDED_SELECT);    n ++;
  XtSetValues(view->list,args,n );                             n = 0;

  view->current->num_selected = selected;
  if(selected > 0) 
    view->current->button_state &= ~(BTN_NOSELECTION);
  else
    view->current->button_state |= BTN_NOSELECTION;

  check_buttons(view_menu,XtNumber(view_menu), view->current->button_state);
  reset_cursors();

}

void view_unselect_messages(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Message_List *message_list;

  set_watch_cursors();

  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    XmListDeselectPos(view->list,message_list->number);
    message_list->selected = FALSE;
  }

  view->current->num_selected = 0L;
  view->current->button_state |= BTN_NOSELECTION;

  check_buttons(view_menu,XtNumber(view_menu), view->current->button_state);
  reset_cursors();

}


void view_select_new_messages(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  
  Arg args[ARGLISTSIZE];
  int n = 0;
  Message_List *message_list;
  MESSAGECACHE *elt;
  unsigned long selected = 0L;


  view_unselect_messages(w,view,xp);

  set_watch_cursors();

  XtSetArg(args[n], XmNselectionPolicy, XmMULTIPLE_SELECT);     n ++;
  XtSetValues(view->list,args,n );                              n = 0;

  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    if((!message_list->message) || (!message_list->message->longcache))
      continue;
    elt = &(message_list->message->longcache->elt);
    if((elt) && (elt->recent) && (!elt->seen)) {
      XmListSelectPos(view->list,message_list->number,FALSE);
      message_list->selected = TRUE;
      selected ++;
    }
  }

  XtSetArg(args[n], XmNselectionPolicy, XmEXTENDED_SELECT);     n ++;
  XtSetValues(view->list,args,n );                              n = 0;

  view->current->num_selected = selected;
  if(selected) 
    view->current->button_state &= ~(BTN_NOSELECTION);
  else
    view->current->button_state |= BTN_NOSELECTION;

  check_buttons(view_menu,XtNumber(view_menu), view->current->button_state);
  reset_cursors();

}

void view_bye(w,view,e)
     Widget w;
     View *view;
     XEvent *e;
{

  view->current = NULL;
  XtDestroyWidget(view->shell);
  if(view->list_is_realized == TRUE) {
    XtDestroyWidget(view->list_shell);
    view->list_is_realized = FALSE;
  }

  if(session.edfilter && session.edfilter->is_realized == TRUE) {
    XtDestroyWidget(session.edfilter->shell);
    session.edfilter->is_realized = FALSE;
  }

  view->is_realized = FALSE;

}


void view_destroy(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view->current = NULL;

  if(view->list_is_realized) {
    XtDestroyWidget(view->list_shell);
    view->list_is_realized = FALSE;
  }

  if(session.edfilter && session.edfilter->is_realized == TRUE) {
    XtDestroyWidget(session.edfilter->shell);
    session.edfilter->is_realized = FALSE;
  }

  view->is_realized = FALSE;
}


void view_copy(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  char *sequence = NULL;
  char *mailbox_name = NULL;
  int result = NIL;


  if(view->current->scripting && view->current->script_arg)
    mailbox_name = fix_mailboxpath(view->current->script_arg);

  if(mailbox_name == NULL)
    mailbox_name = get_mailbox_name(view->shell, NULL, TRUE,
				    view->current->mailbox->host, 
				    preferences.default_savebox);
  if(mailbox_name == NULL)
    return;

  set_watch_cursors();

  sequence = generate_sequence(view->current);

  if(sequence && *sequence && *mailbox_name) 
    result = mail_copy(view->current->mailstream, sequence, mailbox_name);
  if(result == T)
    mm_log("Copy successful.", NIL);

  if(sequence)
    fs_give((void **) &sequence);
  fs_give((void **) &mailbox_name);

  reset_cursors();
}


void view_move(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  char *sequence = NULL;
  char *mailbox_name = NULL;
  int result = NIL;

  if(view->current->scripting && view->current->script_arg)
    mailbox_name = fix_mailboxpath(view->current->script_arg);

  if(mailbox_name == NULL)
    mailbox_name = get_mailbox_name(view->shell, NULL, TRUE,
				    view->current->mailbox->host, 
				    preferences.default_savebox);
  if(mailbox_name == NULL)
    return;
  
  set_watch_cursors();
  
  sequence = generate_sequence(view->current);
  
  if(sequence && *sequence && *mailbox_name) 
    result = mail_move(view->current->mailstream, sequence, mailbox_name);
  if(result == T)
    mm_log("Copy successful.", NIL);

  if(sequence)
    fs_give((void **) &sequence);
  fs_give((void **) &mailbox_name);

  reset_cursors();
}




/*
 * To avoid excessive scrolling and flicker when we change context,
 * we grab a new list widget, load it up, and then change the old
 * one out from under the user. There's still a screen flash, but
 * that's life...
 */


void view_context_switch(view,old,new)
     View *view;
     Lview *old;
     Lview *new;
{

  Arg args[ARGLISTSIZE];
  int n = 0;
  Dimension width, height;

  char msg[512];
  Widget list, oldlist;

  unsigned long total = 0L;
  unsigned long highlighted = 0L;
  Message_List *message_list;

  if(! view || view->is_realized == FALSE)
    return;


  XtSetArg(args[n], XmNwidth, &width);       n ++;
  XtSetArg(args[n], XmNheight, &height);     n ++;
  XtGetValues(view->shell, args, n );         n = 0;


  view->current = new;

  list = new_list_widget(view);

  if(new->title) {
    XtSetArg(args[n], XmNtitle, new->title); n ++;
    XtSetValues(view->shell, args, n); n = 0;
  }

  if((new->filter) && (new->filter->action))
    XmTextSetString(view->action,new->filter->action);
  else
    XmTextSetString(view->action,preferences.default_action);

  XtSetArg(args[n], XmNselectionPolicy, XmMULTIPLE_SELECT);     n ++;
  XtSetValues(list,args,n );                                    n = 0;


  for(message_list = new->message_list;
      message_list; message_list = message_list->next) {
    if(message_list->message) {
      XmListAddItemUnselected(list,message_list->message->viewline,0);
      message_list->number = (++total);
      if(message_list->selected) {
	XmListSelectPos(list,message_list->number, FALSE);
	highlighted ++;
      }
    }

  }


  XtSetArg(args[n], XmNselectionPolicy, XmEXTENDED_SELECT);     n ++;
  XtSetValues(list,args,n );                                    n = 0;

  new->num_selected = highlighted;

  XmListSetBottomPos(list,new->count);

  oldlist = view->list;


  XtUnmanageChild(oldlist);

  view->list = list;
  XtManageChild(list);
  XtSetArg(args[n], XmNwidth, width );                          n ++;
  XtSetArg(args[n], XmNheight, height );                        n ++;
  XtSetValues(view->shell, args, n );                           n = 0;

  XtDestroyWidget(oldlist);

  XFlush(display);


  if(highlighted != 0L)
    new->button_state &= ~(BTN_NOSELECTION);
  else
    new->button_state |= BTN_NOSELECTION;
  check_buttons(view_menu,XtNumber(view_menu), new->button_state);

}

/* 
 * Called when we are going to close a mailbox, and we need to 
 * destroy the current lview. Since any other views of this mailbox
 * are going away, we can't use them. This finds a view unrelated
 * to this mailbox or else NULL. It's safe to context switch to
 * another mailbox.
 */

Lview *find_lview_another_mailbox(mailbox)
     Mailbox *mailbox;
{
  Lview *lview = NULL;
  
  for(lview = session.view->lview; lview; lview = lview->prev) {
    if(lview->mailbox != mailbox)
      return(lview);
  }
  return(NULL);

}

void view_unlink_destroy_lview(view,lview,closing)
     View *view;
     Lview *lview;
     Boolean closing;
{

  Lview *next_view;

  if(view->lview == lview)
    view->lview = lview->prev;

  if(lview->prev) {
    lview->prev->next = lview->next;
    if(lview->next)
      lview->prev->button_state &= ~(BTN_NONEXT);
    else
      lview->prev->button_state |= BTN_NONEXT;
  }

  if(lview->next) {
    lview->next->prev = lview->prev;
    if(lview->prev)
      lview->next->button_state &= ~(BTN_NOPREV);
    else
      lview->next->button_state |= BTN_NOPREV;
  }

  if(lview->filter && lview->filter->filter_type == FILTER_TYPE_ALL) 
    free_message_list(lview->message_list,TRUE,TRUE);
  else
    free_message_list(lview->message_list,TRUE,FALSE);

  free_filter(lview->filter);

  if(view->is_realized) {
    if(lview == view->current) {

      if(closing) {
	next_view = find_lview_another_mailbox(lview->mailbox);
      }
      else {
	if(lview->prev)
	  next_view = lview->prev;
	else
	  if(lview->next)
	    next_view = lview->next;
      }
      if(next_view)
	view_context_switch(view,NULL,next_view);
      else 
	view_bye(view->shell,view,NULL);
    }
  }

  stuff_lview_list(view,TRUE);

  free_lview(lview);
}

/* Returns the "ALL" lview (main mailbox) for a particular mailstream */

Lview *get_all_lview(mailstream)
     MAILSTREAM *mailstream;
{

  Lview *lview;

  if(session.view && session.view->lview)
    for(lview = session.view->lview; lview ; lview = lview->prev) 
      if((lview->mailstream == mailstream) && (lview->filter) 
	 && (lview->filter->filter_type == FILTER_TYPE_ALL))
	return(lview);

  return(NULL);

}

/* Returns the "UNFILTERED" lview for a particular mailstream */

Lview *get_bucket_lview(mailstream)
     MAILSTREAM *mailstream;
{

  Lview *lview;

  if(session.view && session.view->lview)
    for(lview = session.view->lview; lview ; lview = lview->prev) 
      if((lview->mailstream == mailstream) && (lview->filter) 
	 && (lview->filter->filter_type == FILTER_TYPE_BUCKET))
	return(lview);

  return(NULL);

}



void load_default_filters(lview,filter_map)
     Lview *lview;
     Filter_Map *filter_map;
{
  FILE *fp;
  char buf[FILEBUFFLEN];
  char *name;
  int level;
  char *parent_name;
  char *ptr;
  Lview *our_lview;
  Lview *parent_lview;
  Filter *our_filter;
  Filter *curr_filter;
  parse_errors errors;
  char *temp_txt;
  char errmsg[FILEBUFFLEN];

  mm_log("Loading default filters.", NIL);

  if((fp = fopen(filter_map->filename,"r")) == NULL) {
    return;
  }

  /* 
   * parse the filter map file. 
   * filter_name [SP] level [SP] parent_name [LF]
   */

  while(fgets(buf,sizeof(buf),fp)) {
    buf[strlen(buf) - 1] = NUL_TERM;
    name = buf;
    ptr = strchr(buf,SPACECHAR);
    if(ptr == NULL)
      continue;
    *ptr = NUL_TERM;
    ptr ++;
    level = atoi(ptr);
    if(level == 0)
      continue;
    parent_name = strchr(ptr,SPACECHAR);
    if(parent_name == NULL)
      continue;
    parent_name ++;

    for(curr_filter = session.filters; 
	curr_filter; curr_filter = curr_filter->next) {

      /* find requested filter */

      if(strcmp(curr_filter->name,name) == STRMATCH)
	break;
    }

    if(curr_filter == NULL)
      continue;
    
    /* find the (active) parent */

    for(parent_lview = session.view->lview; 
	parent_lview; parent_lview = parent_lview->prev) {

      if(parent_lview->mailstream != lview->mailstream)
	continue;

      if(parent_lview->level != (level - 1))
	 continue;

      if((parent_lview->filter == NULL) 
	 || ((strcmp(parent_lview->filter->name,parent_name)) == STRMATCH))
	break;

    }

    /* no parent found */

    if(parent_lview == NULL)
      continue;


    /* we have a parent -- now see if the filter parses */

    our_filter = new_filter();
    
    our_filter->filter_type = FILTER_TYPE_USER;

    our_filter->name = cpystr(curr_filter->name);
    temp_txt = cpystr(curr_filter->text);
    our_filter->action = cpystr(curr_filter->action);

    errors = lv_parse_filter(our_filter->name,temp_txt,
			     (LEAF **) &our_filter->search_tree,
			     &our_filter->text,
			     &our_filter->search_types,
			     errmsg);

    if(errors != parse_success) {
      free_filter(our_filter);
      fs_give((void **) &temp_txt);
      mm_log(errmsg,WARN);
      continue;
    }

    our_lview = new_lview();
    our_lview->filter = our_filter;
    our_lview->parent = parent_lview;
    our_lview->level = level;
    create_lview(our_lview,TRUE);

  }
  fclose(fp);

  update_logical_views(lview->mailbox,FALSE);

  stuff_lview_list(session.view,TRUE);


  check_buttons(view_menu,XtNumber(view_menu), 
		session.view->current->button_state);

}


void create_lview(lview, batch)
     Lview *lview;
     Boolean batch;
{

  char buffer[FILEBUFFLEN];
  char *fixedname;
  Mailbox *mailbox;
  Lview *parent;
  Lview *bucket;
  Filter *filter;

  set_watch_cursors();

  create_view_window();

  if(! session.view->current)
    return;

  mailbox = session.view->current->mailbox;

  lview->mailstream   = mailbox->mailstream;
  lview->mailbox      = mailbox;

  if(((bucket = get_bucket_lview(mailbox->mailstream)) == NULL)
     && lview->filter && lview->filter->filter_type == FILTER_TYPE_USER) {

    /* create the "UNFILTERED" view */

    filter = new_filter();
    filter->name = cpystr("UNFILTERED");
    filter->action = cpystr(preferences.default_action);
    filter->filter_type = FILTER_TYPE_BUCKET;
    
    bucket = new_lview();
    bucket->parent = get_all_lview(mailbox->mailstream);
    bucket->level = 1;
    bucket->mailstream = mailbox->mailstream;
    bucket->mailbox = mailbox;
    bucket->filter = filter;
    filter->mailboxname = cpystr(mailbox->imapname);
    fixedname = unfix_mailboxpath(mailbox->mailboxname);

    sprintf(buffer,"[%s] %s%s%s",
	  filter->name,
	    (mailbox->host) ? mailbox->host : EMPTYSTR,
	    (mailbox->host) ? ":" : EMPTYSTR,
	    fixedname);

    fs_give((void **) &fixedname);

    bucket->title = cpystr(buffer);

    if((mailbox->mailstream != NULL) 
       && (*mailbox->mailstream->user_flags == NULL))
      bucket->button_state |= BTN_NOKEYWORDS;

    if(mailbox->type == MAILBOX_TYPE_NEWS)
      bucket->button_state |= BTN_ISNEWS;


    for(parent = session.view->lview; parent; parent = parent->prev) {
      if(parent == bucket->parent)
	break;
    }

    if(parent) {
    bucket->next = parent;
    bucket->prev = parent->prev;
    parent->prev = bucket;
    parent->button_state &= ~(BTN_NOPREV);
    if(bucket->prev) {
      bucket->button_state &= ~(BTN_NOPREV);
      bucket->prev->next = bucket;
      bucket->prev->button_state &= ~(BTN_NONEXT);
    }
    else
      bucket->button_state |= BTN_NOPREV;

    bucket->button_state &= ~(BTN_NONEXT);
  }

    else {
      /* this is pretty serious, but we should never see it */
      mm_log("Internal error. Cannot locate parent view.", ERROR);
    }
  }




  lview->filter->mailboxname = cpystr(mailbox->imapname);

  fixedname = unfix_mailboxpath(mailbox->mailboxname);

  sprintf(buffer,"[%s] %s%s%s",
	  lview->filter->name,
	  (mailbox->host) ? mailbox->host : EMPTYSTR,
	  (mailbox->host) ? ":" : EMPTYSTR,
	  fixedname);

  fs_give((void **) &fixedname);

  lview->title = cpystr(buffer);


  /* 
   *  Link into our structure beneath the parent. We should always
   *  have a parent. session.view->lview is ordered in reverse.
   *  (Everything is "prev" to the root node).
   */

  if(mailbox->type == MAILBOX_TYPE_NEWS)
    lview->button_state |= BTN_ISNEWS;


  if((mailbox->mailstream != NULL) 
     && (*mailbox->mailstream->user_flags == NULL))
    lview->button_state |= BTN_NOKEYWORDS;


  for(parent = session.view->lview; parent; parent = parent->prev) {
    if(parent == lview->parent)
      break;
  }

  if(parent) {
    lview->next = parent;
    lview->prev = parent->prev;
    parent->prev = lview;
    parent->button_state &= ~(BTN_NOPREV);
    if(lview->prev) {
      lview->button_state &= ~(BTN_NOPREV);
      lview->prev->next = lview;
      lview->prev->button_state &= ~(BTN_NONEXT);
    }
    else
      lview->button_state |= BTN_NOPREV;

    lview->button_state &= ~(BTN_NONEXT);
  }

  else {
    /* this is pretty serious, but we should never see it */
    mm_log("Internal error. Cannot locate parent view.", ERROR);
  }

  if(batch == FALSE) {
    update_logical_views(mailbox,TRUE);
    view_context_switch(session.view, session.view->current, lview);
    reset_cursors();
  }
}

/* Creates a main mailbox ("ALL" lview) and loads its message info */

void create_all_lview(mailbox)
     Mailbox *mailbox;
{
  Lview  *lview;
  Filter *filter;
  Filter_Map *filter_map;
  Message_List *message_list;
  char buffer[FILEBUFFLEN];
  char *fixedname;

  set_watch_cursors();

  create_view_window();

  lview = new_lview();

  lview->mailstream   = mailbox->mailstream;
  lview->mailbox      = mailbox;
  lview->count        = mailbox->nmsgs;

  filter = new_filter();
  
  filter->mailboxname = cpystr(mailbox->imapname);
  filter->name = cpystr("ALL");
  filter->filter_type = FILTER_TYPE_ALL;

  lview->filter = filter;

  fixedname = unfix_mailboxpath(mailbox->mailboxname);

  sprintf(buffer,"[%s] %s%s%s",
	  filter->name,
	  (mailbox->host) ? mailbox->host : EMPTYSTR,
	  (mailbox->host) ? ":" : EMPTYSTR,
	  fixedname);

  fs_give((void **) &fixedname);

  lview->title = cpystr(buffer);
  lview->next = NULL;
  lview->button_state |= BTN_NONEXT;

  if(session.view->lview) {
    lview->prev = session.view->lview;
    lview->button_state &= ~(BTN_NOPREV);
    session.view->lview->next = lview;
    session.view->lview->button_state &= ~(BTN_NONEXT);
  }
  else { 
    lview->button_state |= BTN_NOPREV;
    lview->prev = NULL;
  }

  if(mailbox->type == MAILBOX_TYPE_NEWS)
    lview->button_state |= BTN_ISNEWS;
  

  if((mailbox->mailstream != NULL) 
     && (*mailbox->mailstream->user_flags == NULL))
    lview->button_state |= BTN_NOKEYWORDS;

  session.view->lview = lview;

  get_message_info(mailbox,lview);

  lview->has_new = has_new_messages(lview,TRUE);

  stuff_lview_list(session.view,TRUE);

  view_context_switch(session.view, session.view->current, lview);

  if((filter_map = get_filter_map(lview)) != NULL)
    load_default_filters(lview,filter_map);

  reset_cursors();

}

/*
 * Called from the c-client when a message is expunged.
 * We go through this in distinct steps. First, remove the message
 * from the "ALL" view (main mailbox), but !!DON'T!! yet free the message.
 * Then decrement subsequent message numbers.
 * Once that's accomplished, we go through all our logical views for
 * this mailbox and remove any message_list structures which contain
 * the expunged message. Again, we decrement higher numbers, but only in 
 * the lview structure, since our individual message numbers have already
 * been decremented in the ALL lview. This time, we can free the 
 * message_list, but we still don't free the message itself. We also make 
 * sure our lview structure is up to date, especially if it's the current 
 * view. The final act is to destroy the message once all of its references 
 * have been fixed up.
 */




void view_expunge(mailstream,msgno)
     MAILSTREAM *mailstream;
     unsigned long msgno;
{

  Boolean found_it = FALSE;
  Lview *lview;

  Message_List *message_list;
  Message_List *dead_message_list = NULL;
  Boolean was_selected;

  if(session.view == NULL)
    return;

  lview = get_all_lview(mailstream);

  if(! lview)
    return;

  found_it = FALSE;

  for(message_list = lview->message_list; 
      message_list; message_list = message_list->next) {

    if(message_list->message == NULL)
      continue;

    if(message_list->message->msgno == msgno) {
      found_it = TRUE;
      dead_message_list = message_list;

      if(lview->count)
	lview->count --;

      if((message_list->selected == TRUE) && (lview->num_selected > 0L)) {
	lview->num_selected --;
	if(lview->num_selected == 0L)
	  lview->button_state |= BTN_NOSELECTION;
      }

      if(message_list->next)
	message_list->next->prev = message_list->prev;
      if(message_list->prev) 
	message_list->prev->next = message_list->next;
      else
	lview->message_list = message_list->next;
    }  

    if(message_list->message->msgno > msgno) {
      message_list->message->msgno --;
      message_list->number --;
    }
  }
    
  lview->has_new = has_new_messages(lview,FALSE);
    
  
  if(found_it == FALSE)       /* shouldn't happen */
    return;

  for(lview = session.view->lview; lview; lview = lview->prev) {
    if(lview->mailstream != mailstream)
      continue;
    if(lview->filter && lview->filter->filter_type == FILTER_TYPE_ALL)
      continue;  /* already got this guy */

    found_it = FALSE;

    for(message_list = lview->message_list; 
	message_list; message_list = message_list->next) {
      
      if(message_list->message == NULL)
	continue;

      if(message_list->message == dead_message_list->message) {
	found_it = TRUE;
	
	if(lview->count)
	  lview->count --;
	if((message_list->selected == TRUE) && (lview->num_selected > 0L)) {
	  lview->num_selected --;
	  if(lview->num_selected == 0L)
	    lview->button_state |= BTN_NOSELECTION;
	}
	
	if(message_list->next)
	  message_list->next->prev = message_list->prev;
	if(message_list->prev) 
	  message_list->prev->next = message_list->next;
	else
	  lview->message_list = message_list->next;
	
	free_message_list(message_list,FALSE,FALSE);
      }  

      if(found_it == TRUE)
	message_list->number --;
    }
    
    lview->has_new = has_new_messages(lview,FALSE);
  }
  
  free_message_list(dead_message_list,FALSE,TRUE);
}



void view_after_expunge()
{
  if(! session.view->is_realized)
    return;

  /* 
   * Just make sure the current view is really current. This saves us 
   * from flashing every message change made during the expunge.
   */

  view_context_switch(session.view,
		      session.view->current,session.view->current);

  stuff_lview_list(session.view,TRUE);
}





void view_close_mailbox(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Mailbox *mailbox;
  Lview *lview;
  Read_Info *read_info;
  Boolean found = FALSE;
  char log[FILEBUFFLEN];

  for(mailbox = session.mailboxes; mailbox ; mailbox = mailbox->next) {
    if(mailbox->mailstream != view->current->mailstream)
      continue;

    sprintf(log,"Closing mailbox '%s'.",mailbox->imapname);
    mm_log(log,NIL);

    XtRemoveTimeOut(mailbox->timer);

    /* 
     * Why? Well, if the mailbox is still there, we need to
     *  checkpoint any recent flag changes before closing.
     *  That's why.
     */

    if(mail_ping(mailbox->mailstream)) {
      mail_check(mailbox->mailstream);
      mail_close(mailbox->mailstream);
    }

    if(session.view) {
      do {
	for(lview = session.view->lview; lview ; lview = lview->prev)
	  if(lview->mailstream == mailbox->mailstream) {
	    found = TRUE;
	    view_unlink_destroy_lview(session.view,lview,TRUE);
	  }
	  else
	    found = FALSE;
      } while((lview) && (found == TRUE));
    }

    if(session.read)
      for(read_info = session.read->read_info; read_info; 
	  read_info = read_info->prev)
	if(read_info->mailstream == mailbox->mailstream) {
	  if((session.read->is_realized) 
	     && (read_info == session.read->current))
	    read_close_kill_current(session.read->shell,session.read,NULL);
	  else
	    kill_read_info(read_info);
	}

    compose_mailbox_close(mailbox->mailstream);

    if(mailbox == session.mailboxes)
      session.mailboxes = mailbox->next;

    if(mailbox->prev)
      mailbox->prev->next = mailbox->next;
    if(mailbox->next)
      mailbox->next->prev = mailbox->prev;

    break;
  }

  free_mailbox(mailbox);

  if(session.mailboxes == NULL)
    session.button_state |= (BTN_NOMAILBOXESOPEN);

  update_main_buttons();


}

Menu view_lview_file_menu[] = {

  { "Jump to Selected", "view_lview_accept", 'J',
      view_lview_accept, NULL, 0, NULL, NULL, BTN_ON },

  { "Detach Selected", "view_lview_detach", 'D',
      view_lview_detach, NULL, 0, NULL, NULL, BTN_ON },

  { "Save Layout", "view_lview_save", 'S',
      view_lview_save, NULL, 0, NULL, NULL, BTN_ON },

 { "Close Window", "view_lview_dismiss", 'C',
      view_lview_dismiss, NULL, 0, NULL, NULL, BTN_ON },

};

Menu view_lview_option_menu[] = {

  { "Hide Empty Views", "view_lview_hide", 'H',
      view_lview_hide, NULL, 0, NULL, NULL, BTN_ON },

  { "Show All Views", "view_lview_show", 'A',
      view_lview_show, NULL, 0, NULL, NULL, BTN_ON },

};


Menu view_lview_menu[] = {

  { "File", "view_lview_file_menu", 'F',
      NULL, view_lview_file_menu, XtNumber(view_lview_file_menu),
      NULL, NULL, BTN_ON },
 
  { "Option", "view_lview_option_menu", 'O',
      NULL, view_lview_option_menu, XtNumber(view_lview_option_menu),
      NULL, NULL, BTN_ON },

  { "Help", "lview_list_HELP!", 'H',
      view_lview_help, NULL, 0, NULL, NULL, BTN_ON },



};



void view_lview_window(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{


  Arg args[ARGLISTSIZE];
  int n = 0;
  Position x, y;
  XtTranslations translations;
  Widget form, menubar, dummy;

  if(view->list_is_realized == TRUE) {
    XRaiseWindow(display,XtWindow(view->list_shell));
    return;
  }

  get_pointer_position(w, &x, &y);

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  view->list_shell = XtCreateWidget("Logical Views",
					 topLevelShellWidgetClass,
					 top, args, n ); n = 0;

  form = XmCreateForm(view->list_shell, "form", args, n); n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form,"menubar", args, n); n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar,view_lview_menu, 
		      XtNumber(view_lview_menu), BTN_ON, (XtPointer) view);

  /* An invisible text field. This is used only to set the width. */

  XtSetArg(args[n], XmNcolumns, COMPOSEWIDTH - 20); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  dummy = XmCreateTextField(form,"dummy", args, n); n = 0;
  XtManageChild(dummy);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, menubar); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n ++;
  
  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmCONSTANT); n ++;
  XtSetArg(args[n], XmNvisibleItemCount, 10 ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmSINGLE_SELECT); n ++;
  view->list_list = 
    XmCreateScrolledList(form,"list",args,n); n = 0;
  XtAddCallback(view->list_list,
		XmNsingleSelectionCallback, view_lview_select, view);
  XtAddCallback(view->list_list,
		XmNdefaultActionCallback, view_lview_accept, view);

  XtAddCallback(view->list_shell,
		XmNdestroyCallback, view_lview_destroy, view);

  stuff_lview_list(view,FALSE);

  XtManageChild(view->list_list); 
  XtManageChild(form);
  XtManageChild(view->list_shell);
  XtRealizeWidget(view->list_shell);
  view->list_is_realized = TRUE;

}

void view_lview_save(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{
  FILE *fp;
  Lview *lview;
  Lview *current;
  Filter_Map *filter_map;
  char log[MAXPATHLEN + 128];

  for(current = view->lview; current; current = current->prev) {
    if((current->filter == NULL) 
       || (current->filter->filter_type != FILTER_TYPE_ALL))
      continue;
    filter_map = get_filter_map(current);
    if(filter_map == NULL)
      filter_map = generate_filter_map(current);
    
    set_watch_cursors();

    if((fp = fopen(filter_map->filename,"w")) == NULL) {
      sprintf(log,"Cannot open '%s' for writing.", filter_map->filename);
      mm_log(log,WARN);
      reset_cursors();
      continue;
    }
    (void) chmod(filter_map->filename,S_IRWXU);

    for(lview = current->prev; lview; lview = lview->prev) {
      if(lview->filter == NULL)
	continue;
      if(lview->filter->filter_type == FILTER_TYPE_ALL) {
	fclose(fp);
	break;
      }
      if(lview->filter->filter_type != FILTER_TYPE_USER)
	continue;
      if((lview->parent == NULL) 
	 || (lview->parent->filter == NULL)
	 || (lview->parent->filter->name == NULL))
	continue;
      fprintf(fp,"%s %d %s\n",
	      lview->filter->name,
	      lview->level,
	      lview->parent->filter->name);
    }
    fclose(fp);
  }
  reset_cursors();
  return;
}

void view_lview_select(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{
  Lview *lview;

  for(lview = view->lview; lview; lview = lview->prev) {
    if(lview->number == xp->item_position) {
      if(lview->selected == FALSE)
	lview->selected = TRUE;
      else
	lview->selected = FALSE;
    }
    else
      lview->selected = FALSE;
  }
}


void view_lview_accept(w,view,xp)
     Widget w;
     View *view;
     XmListCallbackStruct *xp;
{
  Lview *lview;

  for(lview = view->lview; lview; lview = lview->prev) {
    if(lview->selected == TRUE) {
      view_context_switch(view,view->current,lview);
      XRaiseWindow(display,XtWindow(view->shell));
      break;
    }
  }
}


void view_lview_hide(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  view->hide_empty = TRUE;
  stuff_lview_list(view,TRUE);

}


void view_lview_show(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  view->hide_empty = FALSE;
  stuff_lview_list(view,TRUE);

}


void view_lview_detach(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  Lview *lview;

  for(lview = view->lview; lview; lview = lview->prev) {
    if(lview->selected == TRUE) {

      /* 
       * Annoying, but we have to make the lview we want
       * to detach the ``current'' lview. We might get some update 
       * flicker as it gets switched in, and then immediately
       * switched out again. That's life. "view_close_mailbox()"
       * and "view_unlink_destroy_lview()" operate on the assumption
       * that the current lview is the target.
       */

      view_context_switch(view,view->current,lview);
      break;
    }
  }

  if(! lview)
    return;

  if(lview->filter && lview->filter->filter_type == FILTER_TYPE_ALL)
    view_close_mailbox(w,view,xp);
  else  
    view_unlink_destroy_lview(view,lview,FALSE);

}


void view_lview_dismiss(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Lview *lview;

  for(lview = view->lview; lview; lview = lview->prev)
    if(lview->selected == TRUE)
      lview->selected = FALSE;

  XtDestroyWidget(view->list_shell);
  view->list_is_realized = FALSE;
}

void view_lview_destroy(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Lview *lview;

  for(lview = view->lview; lview; lview = lview->prev)
    if(lview->selected == TRUE)
      lview->selected = FALSE;

  view->list_is_realized = FALSE;
}


void view_lview_help(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  help(view->list_shell,"Logical Views","lviewlist.help");

}


void stuff_lview_list(view, update)
     View *view;
     Boolean update;
{
  Lview *lview;
  int n = 1;
  int level = 0;
  char buffer[FILEBUFFLEN];
  char tmp[128];
  int spaces;
  char *fixed_name = NULL;
  XmString xstr;

  if(view == NULL) 
    return;

  if(update) {
    if(view->list_is_realized == FALSE)
      return;
    XmListDeselectAllItems(view->list_list);
    XmListDeleteAllItems(view->list_list);
  }

  for(lview = view->lview; lview; lview = lview->prev) {

    if((view->hide_empty == TRUE) && (lview->count == 0L)) {
      lview->number = 0;
      lview->selected = FALSE;
      continue;
    }

    lview->number = n;
    lview->selected = FALSE;
    buffer[0] = NUL_TERM;


    sprintf(buffer,"%6lu",lview->count);

    if((lview->filter) 
       && (lview->filter->filter_type == FILTER_TYPE_ONESHOT))
      strcat(buffer,"-");
    else
      strcat(buffer," ");

    if(lview->has_new)
      strcat(buffer,"* ");
    else
      strcat(buffer,"  ");

    for(level = 0; level < lview->level; level ++)
      strcat(buffer,">");
    if(lview->level)
      strcat(buffer," ");
    strcat(buffer,lview->filter->name);
    for(spaces = strlen(buffer); spaces < 30; spaces ++)
      strcat(buffer,SPACESTR);

    if(lview->mailbox && lview->mailbox->mailboxname)
      fixed_name = unfix_mailboxpath(lview->mailbox->mailboxname);
    if(fixed_name) {
      strcat(buffer,fixed_name);
      fs_give((void **) &fixed_name);
    }
    else
      strcat(buffer,EMPTYSTR);

    if(lview->mailbox && lview->mailbox->host) {
      strcat(buffer,"(");
      strcat(buffer,lview->mailbox->host);
      strcat(buffer,")");
    }
      
    xstr = XmStringCreateSimple(buffer);
    XmListAddItemUnselected(view->list_list,xstr,0);
    XmStringFree(xstr);
    n ++;

  }
}



char *view_fetch(message,header_mode)
     Message *message;
     Header_Mode header_mode;
{

  char *mail_header = NULL;
  char *header = NULL;
  char *mail_text = NULL;
  char *everything;

  if((message == NULL) 
     || (message->mailstream == NULL)
     || (message->msgno == 0L))
    return(NULL);

  mail_header = cpystr(mail_fetchheader(message->mailstream,message->msgno));
  stripcr(mail_header);
  if(mail_header[strlen(mail_header) - 1] == LFCHAR)
    mail_header[strlen(mail_header) - 1] = NUL_TERM;

  if(header_mode == HEADER_FULL) 
    header = cpystr(mail_header);

  if(header_mode == HEADER_PART)
    header = get_short_header(mail_header);

  if(header == NULL)
    header = cpystr(EMPTYSTR);

  mail_text = cpystr(mail_fetchtext(message->mailstream,message->msgno));
  stripcr(mail_text);

  everything = (char *) fs_get(strlen(header) + strlen(mail_text) + 8);
  strcpy(everything,header);
  if(*header != NUL_TERM)
    strcat(everything,LFSTR);
  strcat(everything,mail_text);

  fs_give((void **) &mail_header);
  fs_give((void **) &header);
  fs_give((void **) &mail_text);

  return(everything);

}


void view_printfull(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_print_dispatch(w,view,HEADER_FULL);
}


void view_printpart(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_print_dispatch(w,view,HEADER_PART);
}

void view_printnone(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_print_dispatch(w,view,HEADER_NONE);
}


void view_print_dispatch(w,view,header_mode)
     Widget w;
     View *view;
     Header_Mode header_mode;
{
  Message_List *message_list;
  char *contents;
  int errors = 0;

  if(*preferences.print_command == NUL_TERM) {
    mm_log("No Print command defined.", WARN);
    return;
  }

  set_watch_cursors();



  for(message_list = view->current->message_list; message_list;
      message_list = message_list->next ) {

    if(message_list->selected) {
      contents = view_fetch(message_list->message,
			    (view->current->scripting)
			    ? view->current->script_header : header_mode);
      if(! contents)
	continue;

      
      if(write_to_pipe(preferences.print_command,
		    NULL, contents, strlen(contents)))
	errors ++;
      fs_give((void **) &contents);
    }
  }
  if(errors)
    mm_log("Print Failed.", WARN);
  else
    mm_log("Print successful.", NIL);

  reset_cursors();

}

void view_execfull(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_exec_dispatch(w,view,HEADER_FULL);
}


void view_execpart(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_exec_dispatch(w,view,HEADER_PART);
}

void view_execnone(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_exec_dispatch(w,view,HEADER_NONE);
}


void view_exec_dispatch(w,view,header_mode)
     Widget w;
     View *view;
     Header_Mode header_mode;
{
  Message_List *message_list;
  char *contents;
  int errors = 0;
  char *command = NULL;

  if(view->current->scripting && view->current->script_arg)
    command = cpystr(view->current->script_arg);

  if(command == NULL)
    command = input_string(view->shell,"Shell Command", NULL, NULL);
  if(command == NULL)
    return;

  if(*command == NUL_TERM) {
    fs_give((void **) &command);
    return;
  }

  set_watch_cursors();

  for(message_list = view->current->message_list; message_list;
      message_list = message_list->next ) {

    if(message_list->selected) {
      contents = view_fetch(message_list->message,header_mode);
      if(! contents)
	continue;

      
      if(write_to_pipe(command,
		    NULL, contents, strlen(contents)))
	errors ++;
      fs_give((void **) &contents);
    }
  }
  if(errors)
    mm_log("Shell command failed.", WARN);
  else
    mm_log("Shell command successful.", NIL);

  fs_give((void **) &command);
  reset_cursors();

}



void view_savefull(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_save_dispatch(w,view,HEADER_FULL);
}


void view_savepart(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_save_dispatch(w,view,HEADER_PART);
}

void view_savenone(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_save_dispatch(w,view,HEADER_NONE);
}



void view_save_dispatch(w,view,header_mode)
     Widget w;
     View *view;
     Header_Mode header_mode;
{
  FILE *fp;
  char *filename = NULL;
  Message_List *message_list;
  char *contents;
  int errors = 0;
  Boolean append_it;


  if(view->current->scripting && view->current->script_arg) {
    filename = cpystr(view->current->script_arg);
    append_it = TRUE;
  }

  if(! filename)
    filename = file_select(view->shell,NULL, NULL, NULL, TRUE, &append_it);
  if(filename == NULL)
    return;

  set_watch_cursors();

  if((fp = fopen(filename,(append_it == TRUE) ? "a" : "w")) != NULL) {
    (void) chmod(filename,S_IRWXU);

    for(message_list = view->current->message_list; message_list;
	message_list = message_list->next ) {

      if(message_list->selected) {
	contents = view_fetch(message_list->message,
			      (view->current->scripting) 
			      ? view->current->script_header : header_mode);
	if(! contents)
	  continue;
	if((fwrite(contents,strlen(contents),1,fp)) != 1)
	  errors ++;
	fs_give((void **) &contents);
      }
      if(errors)
	break;
    }
    if((fclose(fp)) != SYSCALL_SUCCESS)
      errors ++;
  }

  if(errors)
    mm_log("Save failed.", WARN);
  else
    mm_log("Save successful.", NIL);

  reset_cursors();

}




void view_set_deleted(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_setflag(w,view,DELETED_FLAG);
}

void view_set_seen(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_setflag(w,view,SEEN_FLAG);
}

void view_set_flagged(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_setflag(w,view,FLAGGED_FLAG);
}

void view_set_answered(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_setflag(w,view,ANSWERED_FLAG);
}

void view_clear_deleted(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_clearflag(w,view,DELETED_FLAG);
}

void view_clear_seen(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_clearflag(w,view,SEEN_FLAG);
}

void view_clear_flagged(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_clearflag(w,view,FLAGGED_FLAG);
}

void view_clear_answered(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  view_clearflag(w,view,ANSWERED_FLAG);
}




void view_setflag(w,view,flag)
     Widget w;
     View *view;
     char *flag;
{
  char *sequence = NULL;

  sequence = generate_sequence(view->current);
  if(sequence == NULL)
    return;
  set_watch_cursors();

  mail_setflag(view->current->mailstream,sequence,flag);

  fs_give((void **) &sequence);
  reset_cursors();
  return;

}


void view_clearflag(w,view,flag)
     Widget w;
     View *view;
     char *flag;
{
  char *sequence = NULL;

  sequence = generate_sequence(view->current);
  if(sequence == NULL)
    return;
  set_watch_cursors();

  mail_clearflag(view->current->mailstream,sequence,flag);

  fs_give((void **) &sequence);
  reset_cursors();
  return;

}

void view_filter(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  create_edfilter_window(view->shell);
}


void view_reply_sender(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Message_List *message_list;
  
  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    if(message_list->selected)
      compose("Reply to Sender", MAIL_BEHAVIOUR, COMPOSE_REPLY,
	      NULL, NULL, message_list->message);
  }

}

void view_reply_all(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Message_List *message_list;
  
  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    if(message_list->selected)
      compose("Reply to All", MAIL_BEHAVIOUR, COMPOSE_REPLYALL,
	      NULL, NULL, message_list->message);
  }

}

void view_remail(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Message_List *message_list;
  
  for(message_list = view->current->message_list;
      message_list; message_list = message_list->next) {
    if(message_list->selected)
      compose("Remail Message", MAIL_BEHAVIOUR, COMPOSE_REMAIL,
	      NULL, NULL, message_list->message);
  }

}

void view_forward(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  (void) compose("Forward Message(s)", MAIL_BEHAVIOUR, COMPOSE_FORWARD,
		 NULL, view->current, NULL);

}
void view_forward_attach(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{

  (void) compose("Forward Attached Message(s)", MAIL_BEHAVIOUR, 
		 COMPOSE_FORWARDATTACH,
		 NULL, view->current, NULL);

}



int message_list_compare(list1,list2)
     Message_List *list1;
     Message_List *list2;
{

  Message_List *ptr1 = list1;
  Message_List *ptr2 = list2;


  for( ; ; ) {
    if((ptr1 != NULL) && (ptr2 != NULL)) {
      if(ptr1->message != ptr2->message)
	return(LISTNOMATCH);
      else {
	ptr1 = ptr1->next;
	ptr2 = ptr2->next;
	continue;
      }
    }
    else {
      if(ptr1 != ptr2)
	return(LISTNOMATCH);
      else
	return(LISTMATCH);
    }
  }
}


void update_current_view(mailbox)
     Mailbox *mailbox;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Lview *lview;
  Message_List *message_list;
  char ** strtab;


  if(session.view == NULL || session.view->is_realized == FALSE)
    return;
  if(session.view->current == NULL)
    return;

  lview = session.view->current;
  if(lview->count == 0L)
    return;

  XmListDeselectAllItems(session.view->list);

  strtab = (char **) fs_get(lview->count * (sizeof(char *)));
  for(message_list = lview->message_list, n = 0 ; message_list;
      message_list = message_list->next) {
    strtab[n] = (char *) message_list->message->viewline;
    n ++;
  }
  XmListReplaceItemsPos(session.view->list,
			(XmStringTable) strtab, lview->count,1);
  free(strtab);
  n = 0;

  XtSetArg(args[n], XmNselectionPolicy, XmMULTIPLE_SELECT);   n ++;
  XtSetValues(session.view->list, args, n);                   n = 0;

  for(message_list = lview->message_list; message_list;
      message_list = message_list->next) {
	if(message_list->selected == TRUE) 
	  XmListSelectPos(session.view->list, message_list->number, FALSE);
      }

  XtSetArg(args[n], XmNselectionPolicy, XmEXTENDED_SELECT);   n ++;
  XtSetValues(session.view->list, args, n);                   n = 0;
  
}

void update_bucket(mailbox)
     Mailbox *mailbox;
{
  Lview *all;
  Lview *bucket;
  Message_List *message_list;
  Message_List *old;
  Message_List *new;

  all = get_all_lview(mailbox->mailstream);
  bucket = get_bucket_lview(mailbox->mailstream);

  if(all == NULL || bucket == NULL)
    return;

  free_message_list(mailbox->found,TRUE,FALSE);
  mailbox->found = NULL;

  for(message_list = all->message_list; message_list; 
      message_list = message_list->next) {
    if((message_list->message) && (message_list->message->in_view == FALSE))
      mm_searched(mailbox->mailstream,message_list->message->msgno);
  }

  new = mailbox->found;
  mailbox->found = NULL;
  old = bucket->message_list;

  if((message_list_compare(old,new)) == LISTMATCH)
    free_message_list(new,TRUE,FALSE);
  else {
    update_lview_message_list(bucket,old,new);
    free_message_list(old,TRUE,FALSE);
  }

}

void set_found_flags(message_list)
     Message_List *message_list;
{
  Message_List *curr;

  for(curr = message_list; curr; curr = curr->next)
    if(curr->message)
      curr->message->in_view = TRUE;

}

/* 
 *  The initialize flag says to update all views.
 *  We set it to true when creating a filter, and after an expunge
 *  to catch the one-shot filters.
 */

void update_logical_views(mailbox,initialize)
     Mailbox *mailbox;
     Boolean initialize;
{
  Lview *lview;
  Message_List *old;
  Message_List *new;

  if(session.view == NULL)
    return;
  
  for(lview = session.view->lview; lview ; lview = lview->prev) {
    if(lview->filter == NULL || lview->mailbox != mailbox)
      continue;

    if(lview->filter->filter_type == FILTER_TYPE_ALL) {
      lview->has_new = has_new_messages(lview,TRUE); /* and wipe "in_view" */
      continue;
    }

    if(lview->filter->filter_type == FILTER_TYPE_BUCKET)
      continue;

    if((lview->filter->filter_type == FILTER_TYPE_ONESHOT)
       && (initialize == FALSE))
      continue;


    new = do_imap_search(lview->mailbox,lview->parent,
			 lview->filter->search_tree);

    old = lview->message_list;

    if((message_list_compare(old,new)) == LISTMATCH) {
      free_message_list(new,TRUE,FALSE);
      set_found_flags(old);
      continue;
    }
    
    update_lview_message_list(lview,old,new);
    free_message_list(old,TRUE,FALSE);
  }

  update_bucket(mailbox);

  stuff_lview_list(session.view,TRUE);
  mailbox->update_needed  = 0;
}


/* 
 *  This function has been overloaded a bit to save running through the list
 * a second time to clear the "in_view" flag.
 */

Boolean has_new_messages(lview,wipe)
     Lview *lview;
     Boolean wipe;
{
  Message_List *message_list;
  MESSAGECACHE *elt;
  Boolean has_new = FALSE;

  for(message_list = lview->message_list; message_list;
      message_list = message_list->next) {

    if(! message_list->message)
      continue;

    if(wipe)
      message_list->message->in_view = FALSE;

    if(message_list->message->longcache) {
      elt = &(message_list->message->longcache->elt);
      if((! elt->seen) && elt->recent) {
	has_new = TRUE;
	if(wipe == FALSE)
	  return(has_new);
      }
    }
  }
  return(has_new);
}


void update_lview_message_list(lview,old,new)
  Lview *lview;
  Message_List *old;
  Message_List *new;
{

  int n = 0;
  Message_List *list1, *list2;

  for(list1 = old; list1; list1 = list1->next) {
    for(list2 = new; list2; list2 = list2->next) {
      if(list1->message != list2->message)
	continue;
      if(list1->selected == TRUE)
	list2->selected = TRUE;
    }
  }

  for(list2 = new; list2; list2 = list2->next) {
    n ++;
    if(list2->message)
      list2->message->in_view = TRUE;
  }

  lview->count = n;
  lview->message_list = new;

  lview->has_new = has_new_messages(lview,FALSE);

  if(lview == session.view->current)
    view_context_switch(session.view,session.view->current,lview);

}



void view_set_keyword(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Message_List *message_list;
  char *keyword = get_keyword(view->shell,view->current->mailstream);
  
  if(keyword) {
    view_setflag(w,view,keyword);
    fs_give((void **) &keyword);
    for(message_list = view->current->message_list; message_list;
	message_list = message_list->next) {
      if((message_list->selected == TRUE) &&
	 (message_list->message != NULL)) {
	update_view_line_stream_msgno(message_list->message->mailstream,
				      message_list->message->msgno);
      }
    }
  }
}


void view_unset_keyword(w,view,xp)
     Widget w;
     View *view;
     XtPointer xp;
{
  Message_List *message_list;
  char *keyword = get_keyword(view->shell,view->current->mailstream);
  
  if(keyword) {
    view_clearflag(w,view,keyword);
    fs_give((void **) &keyword);
    for(message_list = view->current->message_list; message_list;
	message_list = message_list->next) {
      if((message_list->selected == TRUE) &&
	 (message_list->message != NULL)) {
	update_view_line_stream_msgno(message_list->message->mailstream,
				      message_list->message->msgno);
      }
    }
  }
}



Menu keyword_menu[] = {
  { "Accept", "keyword_accept", 'A',
      keyword_accept, NULL, 0, NULL, NULL, BTN_ON },
  { "Cancel", "keyword_cancel", 'C',
      keyword_cancel, NULL, 0, NULL, NULL, BTN_ON },

  { "Help", "keyword_HELP!", 'H',
      keyword_help, NULL, 0, NULL, NULL, BTN_ON },

};




char *get_keyword(w,mailstream)
     Widget w;
     MAILSTREAM *mailstream;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Position x, y;
  Widget form, menubar;
  char *keyword = NULL;
  char *tmp = NULL;
  int num = 0;
  Keyword_Win *keyword_win = NULL;
  XmString xstr;


  if((mailstream == NULL) || (*mailstream->user_flags == NULL))
    return;

  set_pirate_cursors();

  keyword_win = (Keyword_Win *) fs_get(sizeof(Keyword_Win));
  keyword_win->done = 0;
  keyword_win->keyword = NULL;


  get_pointer_position(w,&x,&y);
  

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  keyword_win->shell = XtCreatePopupShell("Keywords",
			 transientShellWidgetClass, w,
			 args, n); n = 0;

  XmAddTabGroup(keyword_win->shell);

  form = XmCreateForm(keyword_win->shell, "form", args, n ); n = 0;

  XtSetArg(args[n], XmNtopAttachment,   XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment,  XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form, "menubar", args, n); n = 0;
  XtManageChild(menubar);

  create_menu_buttons(NULL, menubar, keyword_menu, 
		      XtNumber(keyword_menu),
		      BTN_ON, (XtPointer) keyword_win);


  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC);             n ++;
  XtSetArg(args[n], XmNlistSizePolicy,         XmCONSTANT);           n ++;
  XtSetArg(args[n], XmNvisibleItemCount,       5 );                  n ++; 
  XtSetArg(args[n], XmNselectionPolicy,        XmSINGLE_SELECT);      n ++;
  keyword_win->list = XmCreateScrolledList(form,"list",args,n);      n = 0;

  XtAddCallback(keyword_win->list, XmNsingleSelectionCallback, 
		keyword_select, keyword_win);

  XtAddCallback(keyword_win->list, XmNdefaultActionCallback, 
		keyword_accept, keyword_win);

  XtSetArg(args[n], XmNtopAttachment,   XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget,       menubar); n ++;
  XtSetArg(args[n], XmNleftAttachment,  XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n ++;

  XtSetValues(XtParent(keyword_win->list), args, n); n = 0;

  for(num = 0; mailstream->user_flags[num] != NULL ; num ++) {
    xstr = XmStringCreateSimple(mailstream->user_flags[num]);
    XmListAddItemUnselected(keyword_win->list,xstr,0);
    XmStringFree(xstr);
  }

  XtAddCallback(keyword_win->shell, XmNdestroyCallback,
		keyword_destroy,keyword_win);

  XtManageChild(keyword_win->list);
  XtManageChild(form);
  XtManageChild(keyword_win->shell);

  XtPopup(keyword_win->shell, XtGrabExclusive);
  modal_main_loop(&keyword_win->done);

  XtPopdown(keyword_win->shell);
  XtDestroyWidget(keyword_win->shell);
  keyword = keyword_win->keyword;
  fs_give((void **) &keyword_win);

  reset_cursors();
  return(keyword);
}


void keyword_accept(w,keyword_win,xp)
     Widget w;
     Keyword_Win *keyword_win;
     XtPointer xp;
{

  keyword_win->done = 1;

}

void keyword_destroy(w,keyword_win,xp)
     Widget w;
     Keyword_Win *keyword_win;
     XtPointer xp;
{

  if(keyword_win->keyword)
    fs_give((void **) &keyword_win->keyword);
  fs_give((void **) &keyword_win);
  reset_cursors();

}



void keyword_select(w,keyword_win,xp)
     Widget w;
     Keyword_Win *keyword_win;
     XmListCallbackStruct *xp;
{
  char *str;
  XmStringGetLtoR(xp->item,XmSTRING_DEFAULT_CHARSET, &str);
  if(keyword_win->keyword) {
    if((strcmp(keyword_win->keyword,str)) == STRMATCH) {
      fs_give((void **) &keyword_win->keyword);
      fs_give((void **) &str);
      return;
    }
    else
      fs_give((void **) &keyword_win->keyword);
  }
  keyword_win->keyword = str;

}


void keyword_cancel(w,keyword_win,xp)
     Widget w;
     Keyword_Win *keyword_win;
     XtPointer xp;
{
  if(keyword_win->keyword)
    fs_give((void **) &keyword_win->keyword);
  keyword_win->keyword = NULL;
  keyword_win->done = 1;

}


void keyword_help(w,keyword_win,xp)
     Widget w;
     Keyword_Win *keyword_win;
     XtPointer xp;
{

  help(keyword_win->shell, "Keywords", "keyword.help");

}


