/* system.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1994 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */


#include "ml.h"

#include <pwd.h>
#include <netdb.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <signal.h>


/* Returns the local hostname, qualified, and lowercased */

#ifdef __STDC__
char *get_qualified_local_host_name(void)
#else
char *get_qualified_local_host_name()
#endif
{
  char host_buffer[MAXHOSTNAMELEN + 1];
  struct hostent *hent;
  char *output = NULL;
  
  host_buffer[0] = NUL_TERM;

  if((gethostname(host_buffer,MAXHOSTNAMELEN + 1)) != SYSCALL_SUCCESS)
      return((char *) NULL);

  if((hent = gethostbyname(host_buffer)) == (struct hostent *) NULL)
    output = cpystr(host_buffer);
  else
    output = cpystr(hent->h_name);

  (void) lcase(output);
  return((char *) output);
}

#ifdef __STDC__
char *get_qualified_host_name(char *host)
#else
char *get_qualified_host_name(host)
     char *host;
#endif
{
  struct hostent *hent;
  char *ret = NULL;
  
  if((hent = gethostbyname(host)) == NULL)
    ret = cpystr(host);
  else
    ret = cpystr(hent->h_name);

  (void) lcase(ret);
  return((char *) ret);
}

/*
 * Fill in the local_auth global structure.
 */

#ifdef __STDC__
int get_user(void)
#else
int get_user()
#endif
{
  char *ptr;
  struct passwd *pw = getpwuid(getuid());
  if(pw == NULL)
    return(SYSCALL_FAILURE);

  if((local_auth.hostname = get_qualified_local_host_name()) == NULL)
    local_auth.hostname = cpystr(EMPTYSTR);
  local_auth.username = cpystr((pw->pw_name) ? pw->pw_name : EMPTYSTR);
  local_auth.homedir  = cpystr((pw->pw_dir) ? pw->pw_dir : EMPTYSTR);
  local_auth.shell    = cpystr((pw->pw_shell) ? pw->pw_shell : EMPTYSTR);
  local_auth.fullname = cpystr((pw->pw_gecos) ? pw->pw_gecos : EMPTYSTR);

  /* strip off office and phone info */

  if((ptr = strchr(local_auth.fullname,',')) != NULL)
    *ptr = NUL_TERM;

  return(SYSCALL_SUCCESS);

}

int kid_is_dead;


void dead_kid()
{

  kid_is_dead = 1;

}

#ifdef __STDC__
int write_to_pipe(char * command, char *intro, char *str, 
	       unsigned long length)
#else
int write_to_pipe(command, intro, str, length)
     char *command;
     char *intro;
     char *str;
     unsigned long length;
#endif
{
  FILE *pp = NULL;
  FILE *tmpfp = NULL;
  char filebuff[FILEBUFFLEN];
  char logbuff[FILEBUFFLEN];
  Boolean to_stdout = TRUE;
  char *tmp_filename = NULL;
  char *tmp_command = NULL;

  unsigned long bytes_written = 0L;
  int errors = 0;
  int pid;

  kid_is_dead = 0;


  /* Nothing to do. Not an error. */

  if(! command)
    return(SYSCALL_SUCCESS);

  (void) signal(SIGCHLD, dead_kid);

  if((strstr(command,"%s")) == NULL) {
    to_stdout = TRUE ;
  }
  else {
    to_stdout = FALSE;
    tmp_filename = tmpnam(NULL);
  }

  if(to_stdout) {
    tmp_command = (char *) fs_get(strlen(command) + 32);
    strcpy(tmp_command,command);
    sprintf(logbuff,"Executing: %s",tmp_command);
    mm_log(logbuff,NIL);
    pid = fork();

    if(pid == (-1)) {
      mm_log("Sub-process failed.", WARN);
      return(SYSCALL_FAILURE);
    }

    
    if(pid == 0) {
      
      pp = popen(tmp_command,"w");
      if(pp) {
	if(intro) {
	  if((fwrite(intro,strlen(intro),1,pp)) != 1) 
	    errors ++;
	  if((fputs(LFSTR,pp)) == EOF) 
	    errors ++;
	}
	if(str) 
	  bytes_written = fwrite(str,1,length,pp);
	if(pclose(pp))
	  errors ++;
      }
      else
	errors ++;
      
      if((str) && (bytes_written != length)) 
	errors ++;
      _exit();
    }

    if(pid != 0) {

      modal_main_loop(&kid_is_dead);
      (void) signal(SIGCHLD, SIG_DFL);
      
      fs_give((void **) &tmp_command);
      return((errors) ? SYSCALL_FAILURE : SYSCALL_SUCCESS);
      
    }

    
  }

  /*
   * If we've gotten to here, we're writing to a temp file
   */

  if((tmpfp = fopen(tmp_filename,"w")) != NULL) {
    (void) chmod(tmp_filename,S_IRWXU);  /* for privacy, ignore failure */
    if(intro) {
      if((fwrite(intro,strlen(intro),1,tmpfp)) != 1)
	errors ++;
      if((fputs(LFSTR,tmpfp)) == EOF)
	errors ++;
    }
    
    if(str)
      bytes_written = fwrite(str,1,length,tmpfp);
    if(fclose(tmpfp))
      errors ++;
    if((str) && (bytes_written != length))
      errors ++;
  }
  else
    errors ++;
  
  if(! errors) {
    tmp_command = 
      (char *) fs_get(strlen(command) + strlen(tmp_filename) + 32); 
    /* 
     *  Caller beware! Only one %s format arg used, and most 
     *  likely the user supplied this command string!
     */
    sprintf(tmp_command,command,tmp_filename);
    sprintf(logbuff,"Executing: %s",tmp_command);
    mm_log(logbuff,NIL);
    
    pid = fork();
    
    if(pid == (-1)) {
      mm_log("Sub-process failed.", WARN);
      return(SYSCALL_FAILURE);
    }
    
    
    if(pid == 0) {
      
      pp = popen(tmp_command,"w");
      if(pp != NULL) {
	if(pclose(pp))
	  errors ++;
      }
      else
	errors ++;
      _exit();
    }
      
    if(pid) {
      
      modal_main_loop(&kid_is_dead);
      
      (void) signal(SIGCHLD, SIG_DFL);
      
      if(tmp_filename) 
	unlink(tmp_filename);
      
      if(tmp_command)
	fs_give((void **) &tmp_command);
      
      return((errors) ? SYSCALL_FAILURE : SYSCALL_SUCCESS);
      
    }
    
  }
}



#ifdef __STDC__
Binary_Buffer *read_from_pipe(char *command)
#else
Binary_Buffer *read_from_pipe(command)
     char *command;
#endif
{

  Binary_Buffer *binary_buffer = NULL;

  FILE *pp = NULL;
  FILE *tmpfp = NULL;
  char filebuff[FILEBUFFLEN];
  char logbuff[FILEBUFFLEN];
  Boolean from_stdin = TRUE;
  char *tmp_filename = NULL;
  char *tmp_command = NULL;

  unsigned long bytes_read = 0L;
  int errors = 0;


  if(! command)
    return(NULL);

  if((strstr(command,"%s")) == NULL) {
    from_stdin = TRUE ;
  }
  else {
    from_stdin = FALSE;
    tmp_filename = tmpnam(NULL);
  }

  if(! from_stdin) {

    tmp_command  = 
      (char *) fs_get(strlen(command) + strlen(tmp_filename) + 32);

    sprintf(tmp_command,command,tmp_filename);

    sprintf(logbuff,"Executing: %s",tmp_command);
    mm_log(logbuff,NIL);

    if((pp = popen(tmp_command,"r")) != NULL) {
      (void) chmod(tmp_filename,S_IRWXU);  /* for privacy, ignore failure */
      if(pclose(pp))
	errors ++;
    }
    else
      errors ++;

    if(! errors) 
      binary_buffer = load_binary_file(tmp_filename);
  
    if(tmp_filename) 
      unlink(tmp_filename);

    if(tmp_command)
      fs_give((void **) &tmp_command);

  }

  return(binary_buffer);

}

