/* read.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1994 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

void read_message_part();

void read_bye();
void read_kill_current();
void read_trash_current();
void kill_read_info();
void read_context_switch();
void read_fetch();
void read_destroy();
void read_select_attach();
void read_exec_attach();
void read_new_link();
void read_add_attach_strings();
void read_make_attachment_list();
void read_add_this_part();
void read_full_header();
void read_short_header();
void read_print_dispatch();
void read_printfull();
void read_printpart();
void read_printnone();
void read_reply_sender();
void read_reply_all();
void read_copy();
void read_help();
void read_exec_dispatch();
void read_execfull();
void read_execpart();
void read_execnone();
void read_save_dispatch();
void read_savefull();
void read_savepart();
void read_savenone();
void read_move();
void read_next();
void read_prev();
void read_choosewin();
void read_list_select();
void read_list_cancel();
void read_list_help();
void stuff_read_window_list();
void read_forward();
void read_forward_attach();
void read_remail();
void read_set_deleted();
void read_set_seen();
void read_set_flagged();
void read_set_answered();
void read_clear_deleted();
void read_clear_seen();
void read_clear_flagged();
void read_clear_answered();
void read_setflag();
void read_clearflag();
void read_set_keyword();
void read_unset_keyword();
void pgp_verify();
void pgp_decode();
void pgp_addkey();
void pgp_decode_dispatch();
void read_browse_url();

Menu read_save_menu[] = {

  { "Full Header", "read_savefull", 'F', 
      read_savefull, NULL, 0, NULL, NULL, BTN_ON },
  { "Part Header", "read_savepart", 'P', 
      read_savepart, NULL, 0, NULL, NULL, BTN_ON },
  { "  No Header", "read_savenone", 'N', 
      read_savenone, NULL, 0, NULL, NULL, BTN_ON },

};


Menu read_pipe_menu[] = {

  { "Full Header", "read_execfull", 'F', 
      read_execfull, NULL, 0, NULL, NULL, BTN_ON },
  { "Part Header", "read_execpart", 'P', 
      read_execpart, NULL, 0, NULL, NULL, BTN_ON },
  { "  No Header", "read_execnone", 'N', 
      read_execnone, NULL, 0, NULL, NULL, BTN_ON },

};

Menu read_print_menu[] = {

  { "Full Header", "read_printfull", 'F', 
      read_printfull, NULL, 0, NULL, NULL, BTN_ON },
  { "Part Header", "read_printpart", 'P', 
      read_printpart, NULL, 0, NULL, NULL, BTN_ON },
  { "  No Header", "read_printnone", 'N', 
      read_printnone, NULL, 0, NULL, NULL, BTN_ON },

};


Menu read_file_menu[] = {

  { "Copy to Mailbox", "read_copy", 'C', 
      read_copy, NULL, 0, NULL, NULL, BTN_ISNEWS },

  { "Move to Mailbox", "read_move", 'M', 
      read_move, NULL, 0, NULL, NULL, BTN_ISNEWS },

  { "Print", "read_print", 'P', 
      NULL, read_print_menu, XtNumber(read_print_menu), NULL, NULL, BTN_ON },

  { "Save to File", "read_save_menu", 'S',
      NULL, read_save_menu, XtNumber(read_save_menu), NULL, NULL, BTN_ON },

  { "Pipe | Command", "read_pipe_menu", '|',
      NULL, read_pipe_menu, XtNumber(read_pipe_menu), NULL, NULL, BTN_ON },


  { "Follow Web Link", "read_browse_url", 'W',
      read_browse_url, NULL, 0, NULL, NULL, BTN_ON },

  { "Address_Book", "read_address_book", 'A',
      main_address_book, NULL, 0, NULL, NULL, BTN_ON },


  { "Close Window", "read_bye", 'W', 
      read_bye,   NULL, 0, NULL, NULL, BTN_ON },

};



Menu read_setflags_menu[] = {

  { "Deleted", "read_set_deleted", 'D',
      read_set_deleted, NULL, 0, NULL, NULL, BTN_ON },
  { "Seen", "read_set_seen", 'S',
      read_set_seen, NULL, 0, NULL, NULL, BTN_ON },
  { "Flagged", "read_set_flagged", 'F',
      read_set_flagged, NULL, 0, NULL, NULL, BTN_ON },
  { "Answered", "read_set_answered", 'A',
      read_set_answered, NULL, 0, NULL, NULL, BTN_ON },

};


Menu read_clearflags_menu[] = {

  { "Deleted", "read_clear_deleted", 'D',
      read_clear_deleted, NULL, 0, NULL, NULL, BTN_ON },
  { "Seen", "read_clear_seen", 'S',
      read_clear_seen, NULL, 0, NULL, NULL, BTN_ON },
  { "Flagged", "read_clear_flagged", 'F',
      read_clear_flagged, NULL, 0, NULL, NULL, BTN_ON },
  { "Answered", "read_clear_answered", 'A',
      read_clear_answered, NULL, 0, NULL, NULL, BTN_ON },

};




Menu read_edit_menu[] = {

  { "Set Flags", "read_setflags_menu", 'S',
      NULL, read_setflags_menu, XtNumber(read_setflags_menu), 
      NULL, NULL, BTN_ON },

  { "Clear Flags", "read_clearflags_menu", 'C',
      NULL, read_clearflags_menu, XtNumber(read_clearflags_menu),
      NULL, NULL, BTN_ON },

  { "Set Keyword", "read_set_keyword", 'K',
      read_set_keyword, NULL, 0, NULL, NULL, BTN_NOKEYWORDS },

  { "Unset Keyword", "read_unset_keyword", 'U',
      read_unset_keyword, NULL, 0, NULL, NULL, BTN_NOKEYWORDS },


};

Menu read_key_menu[] = {
  { "Add To Keyring", "pgp_addkey", 'A',
      pgp_addkey, NULL, 0, NULL, NULL, BTN_NOPGP },

  { "Verify Signature", "pgp_verify", 'V',
      pgp_verify, NULL, 0, NULL, NULL, BTN_NOPGP },
  { "Decode Message", "pgp_decode", 'D',
      pgp_decode, NULL, 0, NULL, NULL, BTN_NOPGP },

};

Menu read_option_menu[] = {

  { "Full Header", "read_full_header", 'F',
      read_full_header, NULL, 0, NULL, NULL, BTN_SHORT_HEADER },
      
  { "Short Header", "read_short_header", 'S',
      read_short_header, NULL, 0, NULL, NULL, BTN_LONG_HEADER },

#ifdef PGP

  { "Public Key", "read_key_menu", 'K',
      NULL, read_key_menu, XtNumber(read_key_menu), NULL, NULL, BTN_NOPGP },

#endif /* PGP */

};


Menu read_reply_submenu[] = {

  { "Sender", "read_reply_sender", 'S', 
      read_reply_sender, NULL, 0, NULL, NULL, BTN_ON },

  { "All", "read_reply_all", 'A',
      read_reply_all, NULL, 0, NULL, NULL, BTN_ON },
};


Menu read_forward_submenu[] = {

  { "Insert", "read_forward", 'I', 
      read_forward, NULL, 0, NULL, NULL, BTN_ON },

  { "Attach", "read_forward_attach", 'A',
      read_forward_attach, NULL, 0, NULL, NULL, BTN_ON },
};



Menu read_send_menu[] = {

  { "Compose New Message", "read_compose", 'N', 
      main_compose, NULL, 0, NULL, NULL, BTN_ON },

  { "Reply to ...", "read_reply", 'R', 
      NULL, read_reply_submenu, XtNumber(read_reply_submenu), 
      NULL, NULL, BTN_ON },

  { "Forward", "read_forward", 'F',
      NULL, read_forward_submenu, XtNumber(read_forward_submenu),
      NULL, NULL, BTN_ON },

  { "Remail", "read_remail", 'm',
      read_remail, NULL, 0, NULL, NULL, BTN_ON },

};


Menu read_menu[] = {
  { "File", "read_file_menu", 'F', 
      NULL, read_file_menu, XtNumber(read_file_menu), 
      NULL, NULL, BTN_ON },

  { "Edit", "read_edit_menu", 'E', 
      NULL, read_edit_menu, XtNumber(read_edit_menu), 
      NULL, NULL, BTN_ON },

  { "Options", "read_option_menu", 'O',
      NULL, read_option_menu, XtNumber(read_option_menu), 
      NULL, NULL, BTN_ON },

  { "Send", "read_send_menu", 'S',
      NULL, read_send_menu, XtNumber(read_send_menu), 
      NULL, NULL, BTN_ON },

  { "Dismiss", "read_kill_current", 'D',
      read_kill_current, 0, NULL, NULL, BTN_ON },

  { "Trash", "read_trash_current", 'T',
      read_trash_current, 0, NULL, NULL, BTN_ON },

  { "<", "read_next", '<',
      read_next, NULL, 0, NULL, NULL, BTN_NONEXT },

  { "?", "read_choosewin", '?',
      read_choosewin, NULL, 0, NULL, NULL, BTN_ON },

  { ">", "read_prev", '>',
      read_prev, NULL, 0, NULL, NULL, BTN_NOPREV },

  { "Help", "read_HELP!", 'H', 
      read_help, NULL, 0,
      NULL, NULL, BTN_ON },

};



#ifdef __STDC__
void create_read_window(void)
#else
void create_read_window()
#endif
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Widget form, menubar, pane;
  XtTranslations translations;
  Read *read;

  if(session.read == NULL) {
    read = new_read();
    session.read = read;
  }
  else
    read = session.read;

  if(read->is_realized == TRUE) {
    XRaiseWindow(display,XtWindow(read->shell));
    return;
  }

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  read->shell = XtCreateWidget("read_shell",
			       topLevelShellWidgetClass,
			       top, args, n ); n = 0;

  form = XmCreateForm(read->shell, "form", args, n);            n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM);           n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);          n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);         n ++;
  menubar = XmCreateMenuBar(form,"menubar", args, n);           n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar,read_menu, XtNumber(read_menu), 
		      0, read);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET );         n ++;
  XtSetArg(args[n], XmNtopWidget, menubar);                      n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);           n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);          n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNsashHeight, 6);                           n ++;
  XtSetArg(args[n], XmNsashWidth, 12);                           n ++;
  XtSetArg(args[n], XmNspacing, 4);                              n ++;

  pane = XmCreatePanedWindow(form,"pane", args, n);              n = 0;


  XtSetArg(args[n], XmNpaneMinimum, 30);                         n ++;
  XtSetArg(args[n], XmNeditable, FALSE);                         n ++;
  XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT);             n ++;
  XtSetArg(args[n], XmNwordWrap, TRUE);                          n ++;
  XtSetArg(args[n], XmNrows,     4);                             n ++;
  XtSetArg(args[n], XmNcolumns,  80);                            n ++;
  XtSetArg(args[n], XmNscrollVertical, TRUE);                    n ++;
  XtSetArg(args[n], XmNscrollHorizontal, FALSE );                n ++;
  read->read_header = 
    XmCreateScrolledText(pane, "read_header", args, n);          n = 0;
  XtManageChild(read->read_header);

  translations = XtParseTranslationTable(GLOBAL_text_translations);
  XtOverrideTranslations(read->read_header, translations);

  XtSetArg(args[n], XmNpaneMinimum, 45);                         n ++;
  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmCONSTANT); n ++;
  XtSetArg(args[n], XmNvisibleItemCount, VISIBLE_ATTACHMENTS ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmSINGLE_SELECT); n ++;
  read->read_attach = XmCreateScrolledList(pane,"list",args,n); n = 0;
  XtAddCallback(read->read_attach,
		XmNsingleSelectionCallback, read_select_attach, read);
  XtAddCallback(read->read_attach,
		XmNdefaultActionCallback, read_exec_attach, read);

  XtManageChild(read->read_attach); 

  XtSetArg(args[n], XmNpaneMinimum, 30);                         n ++;
  XtSetArg(args[n], XmNeditable, FALSE);                         n ++;
  XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT);             n ++;
  XtSetArg(args[n], XmNwordWrap, TRUE);                          n ++;
  XtSetArg(args[n], XmNrows,     preferences.read_height);       n ++;
  XtSetArg(args[n], XmNcolumns,  80);                            n ++;
  XtSetArg(args[n], XmNscrollVertical, TRUE);                    n ++;
  XtSetArg(args[n], XmNscrollHorizontal, FALSE );                n ++;
  read->read_text = 
    XmCreateScrolledText(pane, "read_text", args, n);            n = 0;

  XtManageChild(read->read_text);

  translations = XtParseTranslationTable(GLOBAL_text_translations);
  XtOverrideTranslations(read->read_text, translations);

  XtAddCallback(read->shell, XmNdestroyCallback, read_destroy, read);

  XtManageChild(pane);
  XtManageChild(form);
  XtManageChild(read->shell);

  XmProcessTraversal(read->shell, XmTRAVERSE_NEXT_TAB_GROUP);
  XmProcessTraversal(read->shell, XmTRAVERSE_NEXT_TAB_GROUP);

  XtRealizeWidget(read->shell);
  read->is_realized = TRUE;

}



void read_bye(w,read,e)
     Widget w;
     Read *read;
     XEvent *e;
{

  XtDestroyWidget(read->shell);
  read->is_realized = FALSE;

}

void read_help(w,read,e)
     Widget w;
     Read *read;
     XEvent *e;
{
  help(read->shell,"Read Window", "readwin.help");
}

void read_next(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  if(read->current->next)
    read_context_switch(read,read->current,read->current->next);

}




void read_prev(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  if(read->current->prev)
    read_context_switch(read,read->current,read->current->prev);

}



void read_destroy(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  read->is_realized = FALSE;

}


void read_copy(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  char *mailboxname;
  char sequence[32];
  int result;

  if(read->current->mailstream && read->current->msgno) {
    mailboxname = get_mailbox_name(read->shell, NULL, TRUE,
				   read->current->server,
				   preferences.default_savebox);
    if(mailboxname == NULL)
      return;
    if(*mailboxname == NUL_TERM) {
      fs_give((void **) &mailboxname);
      return;
    }

    set_watch_cursors();
    sprintf(sequence,"%d",read->current->msgno);
    result = mail_copy(read->current->mailstream, sequence, mailboxname);
    if(result == T)
      mm_log("Copy successful.", NIL);

    fs_give((void **) &mailboxname);
    reset_cursors();
  }

}

void read_move(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  char *mailboxname;
  char sequence[32];
  int result;

  if(read->current->mailstream && read->current->msgno) {
    mailboxname = get_mailbox_name(read->shell, NULL, TRUE,
				   read->current->server,
				   preferences.default_savebox);
    if(mailboxname == NULL)
      return;
    if(*mailboxname == NUL_TERM) {
      fs_give((void **) &mailboxname);
      return;
    }

    set_watch_cursors();
    sprintf(sequence,"%d",read->current->msgno);
    result = mail_move(read->current->mailstream, sequence, mailboxname);

    if(result == T)
      mm_log("Move successful.", NIL);

    fs_give((void **) &mailboxname);
    reset_cursors();
  }

}







void read_full_header(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  read->current->show_long_header = TRUE;
  read->current->button_state |=  BTN_SHORT_HEADER;
  read->current->button_state &= ~(BTN_LONG_HEADER);
  read_context_switch(read,NULL, read->current);

}


void read_short_header(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  read->current->show_long_header = FALSE;
  read->current->button_state |=  BTN_LONG_HEADER;
  read->current->button_state &= ~(BTN_SHORT_HEADER);
  read_context_switch(read,NULL, read->current);

}



void read_savefull(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  read_save_dispatch(read,HEADER_FULL);

}

void read_savepart(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  read_save_dispatch(read,HEADER_PART);

}

void read_savenone(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  read_save_dispatch(read,HEADER_NONE);

}



void read_save_dispatch(read,header_mode)
     Read *read;
     Header_Mode header_mode;
{
  FILE *fp;
  char *filename;
  int errors = 0;
  char *header_text = NULL;
  char *body_text;
  Boolean append_it;
  filename = file_select(read->shell, NULL, NULL, NULL, TRUE, &append_it);
  if(filename == NULL)
    return;

  set_watch_cursors();

  if(header_mode == HEADER_FULL)
    header_text = read->current->header;
  if(header_mode == HEADER_PART)
    header_text = read->current->short_header;

  if(read->current->alt_text)
    body_text = read->current->alt_text;
  else
    body_text = read->current->current_text;


  if((fp = fopen(filename,(append_it == TRUE) ? "a" : "w")) != NULL) {
    (void) chmod(filename,S_IRWXU);
    if((header_mode != HEADER_NONE) && (header_text != NULL)) {
      if((fwrite(header_text,strlen(header_text),1,fp)) != 1)
	errors ++;
      if((fwrite(LFSTR,1,1,fp)) != 1)
	errors ++;
    }
    if((fwrite(body_text,strlen(body_text),1,fp)) != 1)
      errors ++;
    if((fclose(fp)) != SYSCALL_SUCCESS)
      errors ++;
  }
  else
    errors ++;

  fs_give((void **) &filename);

  if(errors)
    mm_log("Save failed.",WARN);
  else
    mm_log("Save successful.", NIL);

  reset_cursors();
}


void read_printfull(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_print_dispatch(w,read,HEADER_FULL);
}

void read_printpart(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_print_dispatch(w,read,HEADER_PART);
}
void read_printnone(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_print_dispatch(w,read,HEADER_NONE);
}




void read_print_dispatch(w,read,header_mode)
     Widget w;
     Read *read;
     Header_Mode header_mode;
{

  char *header = NULL;

  if(*preferences.print_command == NUL_TERM) {
    mm_log("No Print command defined.", WARN);
    return;
  }

  set_watch_cursors();
  
  if(header_mode == HEADER_FULL)
    header = read->current->header;
  if(header_mode == HEADER_PART)
    header = read->current->short_header;


  if(write_to_pipe(preferences.print_command, header,
		(read->current->alt_text)
		? read->current->alt_text : read->current->current_text,
		(read->current->alt_text)
		? strlen(read->current->alt_text) 
		: strlen(read->current->current_text)))

    mm_log("Print Failed.", WARN);
  else
    mm_log("Print successful.", NIL);

  reset_cursors();

}



char *find_url(s)
     char *s;
{
  char *ptr;
  char *tmp;
  char *ret;
  int n = 0;

  if((s == NULL) || (ptr = strstr(s,"://")) == NULL)
    return(NULL);

  if(ptr > s)
    ptr --;
  while(ptr >= s && isalpha(*ptr))
	   ptr --;
  ptr ++;
  tmp = ptr;

  while((*ptr) && ((!isspace(*ptr)) && (*ptr != '\"') 
	&& (*ptr != '}') && (*ptr != ')') && (*ptr != '>'))) {
    ptr ++;
    n ++;
  }

  ret = (char *) fs_get((unsigned) n + 16);
  strncpy(ret,tmp,n);
  ret[n] = 0;

  return(ret);
}



void read_browse_url(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  XmTextPosition left, right;
  char command[FILEBUFFLEN];
  char log[FILEBUFFLEN];
  char *temp_url = NULL;
  char *url = NULL;

  if(*preferences.url_command == NUL_TERM) {
    mm_log("No URL command defined.", WARN);
    return;
  }

  if((strstr(preferences.url_command,"%s")) == NULL) {
    mm_log("unsupported URL command (needs '%s')", WARN);
    return;
  }

  XmTextGetSelectionPosition(read->read_text, &left, &right);
  if(left != right) {
    temp_url = XmTextGetSelection(read->read_text);
  }
  else {
    temp_url = find_url((read->current->alt_text) 
		   ? read->current->alt_text : read->current->current_text);
  }

  if(temp_url == NULL)
    return;

  url = input_string(read->shell, "Show URL:", temp_url, NULL);
  fs_give((void **) &temp_url);

  if(url == NULL)
    return;

  if(*url == NUL_TERM) {
    fs_give((void **) &url);
    return;
  }

  set_watch_cursors();
  sprintf(command,preferences.url_command,url);
  sprintf(log,"Executing URL command '%s'", command);
  mm_log(log,NIL);

  system(command);

  fs_give((void **) &url);

  reset_cursors();
}

void read_execfull(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_exec_dispatch(w,read,HEADER_FULL);
}

void read_execpart(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_exec_dispatch(w,read,HEADER_PART);
}

void read_execnone(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_exec_dispatch(w,read,HEADER_NONE);
}



void read_exec_dispatch(w,read,header_mode)
     Widget w;
     Read *read;
     Header_Mode header_mode;
{

  char *command;
  char *header = NULL;

  command = input_string(read->shell,"Shell Command", NULL, NULL);

  if(command == NULL)
    return;

  if(*command == NUL_TERM) {
    fs_give((void **) &command);
    return;
  }


  set_watch_cursors();

  if(header_mode == HEADER_FULL)
    header = read->current->header;
  if(header_mode == HEADER_PART)
    header = read->current->short_header;

  if(write_to_pipe(command, header,
		(read->current->alt_text)
		? read->current->alt_text : read->current->current_text,
		(read->current->alt_text)
		? strlen(read->current->alt_text) 
		: strlen(read->current->current_text)))

    mm_log("Shell command failed.", WARN);
  else
    mm_log("Shell command successful.", NIL);

  fs_give((void **) &command);

  reset_cursors();
}





void read_select_attach(w,read,xp)
     Widget w;
     Read *read;
     XmListCallbackStruct *xp;
{

  Part_List *part_list;
  BODY *body = NULL;
  
  if(xp->item_position == read->current->selection) {
    read->current->selection = 0;
    return;
  }

  read->current->selection = xp->item_position;

  for(part_list = read->current->part_list;
      ((part_list) && part_list->partnumber != read->current->selection);
      part_list = part_list->next);

  if(part_list) {
    if(part_list->part) 
      body = &(part_list->part->body);

    read_message_part(read->current,part_list->partstr,body);
  }

}


void read_exec_attach(w,read,xp)
     Widget w;
     Read *read;
     XmListCallbackStruct *xp;
{

  Part_List *part_list;
  BODY *body = NULL;
  int result;

  read->current->selection = xp->item_position;

  if(read->current->selection) {
    for(part_list = read->current->part_list;
	((part_list) && part_list->partnumber != read->current->selection);
	part_list = part_list->next);

    if(part_list) {
      if(part_list->part) 
	body = &(part_list->part->body);

      result = show_mime(read->shell,body);
    }
  }

  XmListSelectPos(read->read_attach,read->current->selection,FALSE);
  XmListSetPos(read->read_attach,read->current->selection);

}

void read_kill_current(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  Read_Info *this_one = NULL;
  Read_Info *next_one = NULL;;

  this_one = read->current;

  if(this_one == NULL)
    return;

  if(this_one->next)
    next_one = this_one->next;
  else
    if(this_one->prev)
      next_one = this_one->prev;

  if(next_one) {
    read_context_switch(read,this_one,next_one);
    check_buttons(read_menu,XtNumber(read_menu),read->current->button_state);
  }
  else
    read_bye(w,read,xp);

  kill_read_info(this_one);
}


/* 
 * kill this message, and do a context switch to any message
 * which isn't in this mailbox; which is being closed.
 * If there aren't any, close the window.
 */



void read_close_kill_current(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  Read_Info *this_one = NULL;
  Read_Info *next_one = NULL;
  Read_Info *current  = NULL;

  this_one = read->current;

  if(this_one == NULL)
    return;

  for(current = read->read_info; current ; current = current->prev)
    if(current->mailstream != this_one->mailstream) {
      next_one = current;
      break;
    }
  
  if(next_one) {
    read_context_switch(read,this_one,next_one);
    check_buttons(read_menu,XtNumber(read_menu),read->current->button_state);
  }
  else
    read_bye(w,read,xp);

  kill_read_info(this_one);
}




void read_trash_current(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  Read_Info *this_one = NULL;
  Read_Info *next_one = NULL;
  char sequence[32];

  this_one = read->current;

  if(this_one == NULL)
    return;

  if(this_one->next)
    next_one = this_one->next;
  else
    if(this_one->prev)
      next_one = this_one->prev;

  if(read->current->mailstream && read->current->msgno) {
    sprintf(sequence,"%d",read->current->msgno);
    set_watch_cursors();
    mail_setflag(read->current->mailstream, sequence, DELETED_FLAG);
    reset_cursors();
  }

  if(next_one) {
    read_context_switch(read,this_one,next_one);
    check_buttons(read_menu,XtNumber(read_menu),read->current->button_state);
  }
  else
    read_bye(w,read,xp);

  kill_read_info(this_one);
}






void kill_read_info(read_info)
     Read_Info *read_info;
{

  if(session.read->read_info == read_info) 
    session.read->read_info = read_info->prev;

  if(read_info->prev) {
    read_info->prev->next = read_info->next;
    if(read_info->next == NULL) 
      read_info->prev->button_state |= BTN_NONEXT;
    else
      read_info->prev->button_state &= ~(BTN_NONEXT);
  }
  if(read_info->next) {
    read_info->next->prev = read_info->prev;
    if(read_info->prev == NULL) 
      read_info->next->button_state |= BTN_NONEXT;
    else
      read_info->next->button_state &= ~(BTN_NONEXT);
  }

  free_read_info(read_info);
  return;

}


void read_messagelist(message_list)
     Message_List *message_list;
{
  
  Message_List *select_list;
  Read_Info *read_info;
  Read_Info *first_message = NULL;
  Message *message;

  create_read_window();

  for(select_list = message_list; select_list;
      select_list = select_list->next) {
    if(select_list->selected == TRUE) {
      message = select_list->message;
      read_info = new_read_info();

      if(first_message == NULL)
	first_message = read_info;

      read_info->mailstream = message->mailstream;
      read_info->msgno = message->msgno;
      read_info->server = cpystr(message->mailbox->host);
      read_info->current_part_str = cpystr(PART_ONE_STR);
      read_info->mailbox_type = message->mailbox->type;

      if(read_info->mailbox_type == MAILBOX_TYPE_NEWS)
	read_info->button_state |= BTN_ISNEWS;

      if((read_info->mailstream != NULL) 
	 && (*read_info->mailstream->user_flags == NULL))
	read_info->button_state |= BTN_NOKEYWORDS;

      if(preferences.pgp_pathname && *preferences.pgp_pathname) {
	if((access(preferences.pgp_pathname,X_OK)) == SYSCALL_FAILURE)
	  read_info->button_state |= BTN_NOPGP;
      }
      else
	read_info->button_state |= BTN_NOPGP;

      read_info->envelope = message->envelope;
      read_info->body = message->body;

      if((read_info->body) && (read_info->body->type == TYPEMULTIPART))
	read_info->attachments = read_info->body->contents.part;

      read_new_link(read_info);
    }
  }

  read_fetch(first_message, first_message->body);
  read_context_switch(session.read,NULL,first_message);
}


void read_rfc822(parent,partstr,body)
     Read_Info *parent;
     char *partstr;
     BODY *body;
{
  
  Read_Info *read_info;
  char *basetxt = NULL;
  unsigned long length;
  char *ptr = NULL;

  create_read_window();

  read_info = new_read_info();

  parent->selection = 0L;

  read_info->mailstream = parent->mailstream;
  read_info->msgno = parent->msgno;
  read_info->server = cpystr(parent->server);
  read_info->mailbox_type = parent->mailbox_type;
  read_info->envelope = parent->envelope;
  read_info->base_part_str = cpystr(partstr);
  read_info->message_type = MESSAGE_PART;
  read_info->fetched = TRUE;
  if(read_info->mailbox_type == MAILBOX_TYPE_NEWS)
    read_info->button_state |= BTN_ISNEWS;

  if((read_info->mailstream != NULL) 
     && (*read_info->mailstream->user_flags == NULL))
    read_info->button_state |= BTN_NOKEYWORDS;

  if(preferences.pgp_pathname && *preferences.pgp_pathname) {
    if((access(preferences.pgp_pathname,X_OK)) == SYSCALL_FAILURE)
      read_info->button_state |= BTN_NOPGP;
  }
  else
    read_info->button_state |= BTN_NOPGP;

  basetxt = cpystr(mail_fetchbody(read_info->mailstream,
				  read_info->msgno,
				  partstr, &length));
  if(basetxt == NULL)
    basetxt = cpystr(EMPTYSTR);

  stripcr(basetxt);

  ptr = strstr(basetxt,"\n\n");

  if(ptr != NULL) {
    ptr ++;
    *ptr = NUL_TERM;
    ptr ++;
    read_info->header = cpystr(basetxt);
  }
  else
    read_info->header = cpystr(EMPTYSTR);

  read_info->short_header = get_short_header(read_info->header);
  if(read_info->short_header == NULL)
    read_info->short_header = cpystr(EMPTYSTR);


  read_new_link(read_info);
  
  if(body->contents.msg.body->type == TYPEMULTIPART) {
    read_info->body = body->contents.msg.body;
    read_info->current_part_str = (char *) fs_get(strlen(partstr) + 2);
    sprintf(read_info->current_part_str,"%s.1",partstr);
    read_info->attachments = read_info->body->contents.part;
    read_fetch(read_info,body);
  }
  else {
    read_info->body = body;
    read_info->current_part_str = cpystr(partstr);
    read_info->current_text = cpystr(ptr);
    read_info->rfc_msg_leaf = TRUE;
  }

  fs_give((void **) &basetxt);

  read_context_switch(session.read,session.read->current,read_info);
}


void read_message_part(read_info,partstr,body)
     Read_Info *read_info;
     char *partstr;
     BODY *body;
{

  /* Don't recurse again... */

  if(read_info->rfc_msg_leaf == TRUE)
    return;

  if(read_info->current_part_str)
    fs_give((void **) &read_info->current_part_str);
  read_info->current_part_str = cpystr(partstr);
  read_info->message_type = MESSAGE_PART;
  read_info->rfc_msg_leaf = FALSE;

  if((body != NULL) && (body->type == TYPEMESSAGE) && (body->subtype != NULL)
     && ((strcasecmp(body->subtype,"rfc822")) == STRMATCH)) {
    read_rfc822(read_info,partstr,body);
    return;
  }

  read_fetch(read_info,body);

  read_context_switch(session.read,NULL,read_info);

}


void read_new_link(read_info)
     Read_Info *read_info;
{

  if(session.read->read_info) {
    session.read->read_info->next = read_info;
    session.read->read_info->button_state &= ~(BTN_NONEXT);
    read_info->button_state &= ~(BTN_NOPREV);
  }
  else
    read_info->button_state |= BTN_NOPREV;

  read_info->button_state |= BTN_NONEXT;
  read_info->prev = session.read->read_info;
  session.read->read_info = read_info;

}




void read_fetch(read_info,body)
     Read_Info *read_info;
     BODY *body;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  unsigned long length = 0L;
  unsigned long newlength = 0L;
  unsigned long foo;
  BODY *local_body = body;
  char buffer[FILEBUFFLEN];
  char *subject = NULL;

  if((local_body) && (local_body->type == TYPEMULTIPART))
    local_body = &(body->contents.part->body);

  set_watch_cursors();

  switch(read_info->message_type) {
  case MESSAGE_NORMAL:

    if(read_info->fetched == TRUE)
      break;

    if(read_info->header == NULL) {
      read_info->header = cpystr(mail_fetchheader(read_info->mailstream,
						  read_info->msgno));
      stripcr(read_info->header);
      if(read_info->header[strlen(read_info->header) - 1] == LFCHAR)
	read_info->header[strlen(read_info->header) - 1] = NUL_TERM;
    }

    read_info->short_header = get_short_header(read_info->header);
    if(read_info->short_header == NULL)
      read_info->short_header = cpystr(EMPTYSTR);

    read_info->current_text = 
      cpystr(mail_fetchbody(read_info->mailstream,
			    read_info->msgno,
			    read_info->current_part_str, &length));

    if(read_info->current_text == NULL)
      read_info->current_text = cpystr(EMPTYSTR);
    
    stripcr(read_info->current_text);
    length = strlen(read_info->current_text);
    break;

  case MESSAGE_PART:
    if(read_info->current_text)
      fs_give((void **) &read_info->current_text);
    read_info->current_text = 
      cpystr(mail_fetchbody(read_info->mailstream,
			    read_info->msgno,
			    read_info->current_part_str, &length));

    if(read_info->current_text == NULL)
      read_info->current_text = cpystr(EMPTYSTR);
    
    stripcr(read_info->current_text);

    length = strlen(read_info->current_text);


    break;

  case MESSAGE_RAWPART:
  case MESSAGE_IN_MEMORY:
  case MESSAGE_FROM_FILE:
    
  default:
    break;
  }

  if(local_body) {

    if(read_info->alt_text)
      fs_give((void **) &read_info->alt_text);


    if(read_info->envelope) 
      subject = read_info->envelope->subject;
    if(subject == NULL)
      subject = "No subject";
      
    sprintf(buffer,"%-64.64s (Part %s%s)",
	    subject,read_info->current_part_str,
	    (read_info->base_part_str) ? " [wrapped]" : EMPTYSTR);
    XtSetArg(args[n], XmNtitle, buffer); n ++;
    XtSetValues(session.read->shell, args, n); n = 0;

    switch(local_body->encoding) {
      
    case ENCBASE64:
      read_info->alt_text = 
	(char *) rfc822_base64(read_info->current_text,length,&newlength);
      if(read_info->alt_text)
	read_info->alt_text[newlength] = NUL_TERM;
      break;
    case ENCQUOTEDPRINTABLE:
      read_info->alt_text =
	(char *) rfc822_qprint(read_info->current_text,length,&newlength);
      if(read_info->alt_text)
	read_info->alt_text[newlength] = NUL_TERM;
      break;
    case ENCOTHER:
    case ENC7BIT:
    case ENC8BIT:
    case ENCBINARY:
    default:
      break;
    }
    if((read_info->alt_text) 
       && ((foo = strlen(read_info->alt_text)) < (newlength)))
      fs_give((void **) &read_info->alt_text);
  }

  read_info->fetched = TRUE;
  reset_cursors();

}




void read_context_switch(read,old,new) 
  Read *read;
  Read_Info *old, *new;
{

  char buffer[FILEBUFFLEN];
  char *subject;
  Arg args[ARGLISTSIZE];
  int n = 0;

  unsigned long selection;

  if((read == NULL) || (new == NULL))
    return;

  /*
   * old isn't used here. It is only in the arg list to make
   * this function syntactically compatible with other context
   * switch functions in the program. The difference in the read 
   * window is that you can't type into it, so there's no old 
   * context to save. Instead we use this function whenever we 
   * need to update the contents of the read window, either to change
   * messages, or to navigate a multipart, or just to toggle the
   * header mode.
   */

  read->current = new;
  selection = read->current->selection;

  /* zap! */


  XmTextSetString(read->read_header, EMPTYSTR);
  XmTextSetString(read->read_text,EMPTYSTR);
  XmListDeselectAllItems(read->read_attach);
  XmListDeleteAllItems(read->read_attach); 

  if(new->envelope)
    subject = new->envelope->subject;
  if(subject == NULL)
    subject = "No subject";
  
  sprintf(buffer,"%-64.64s (Part %s%s)",
	  subject,new->current_part_str, 
	  (new->base_part_str) ? " [wrapped]" : EMPTYSTR);
  XtSetArg(args[n], XmNtitle, buffer); n ++;
  XtSetValues(read->shell, args, n); n = 0;

  switch(new->message_type) {
  case MESSAGE_NORMAL:
  case MESSAGE_PART:
    if(new->fetched == FALSE)
      read_fetch(new,new->body);

    if((new->show_long_header == TRUE) && (new->header))
      XmTextSetString(read->read_header,new->header);
    if((new->show_long_header == FALSE) && (new->short_header))
      XmTextSetString(read->read_header,new->short_header);

    if(new->alt_text)
      XmTextSetString(read->read_text,new->alt_text);
    else
      if(new->current_text)
	XmTextSetString(read->read_text,new->current_text);

    read_make_attachment_list(read);

    if(selection) {
      read->current->selection = selection;
      XmListSelectPos(read->read_attach,selection,FALSE);
      XmListSetPos(read->read_attach,selection);
    }

    break;
  case MESSAGE_RAWPART:
  case MESSAGE_IN_MEMORY:
  case MESSAGE_FROM_FILE:
  default:
    break;
  }

  check_buttons(read_menu,XtNumber(read_menu),new->button_state);

}

void read_make_attachment_list(read)
     Read *read;
{

  int nitems = 0;
  int i;
  XmString xstr;

  /* Erase current list contents and start over */

  XmListDeselectAllItems(read->read_attach);
  XmListDeleteAllItems(read->read_attach); 

  read->current->selection = 0;

  if(read->current->part_list) 
    destroy_part_list(read->current->part_list);

  read->current->part_list = NULL;

  if(read->current->body == NULL) {
    check_buttons(read_menu,XtNumber(read_menu),
		  read->current->button_state);
    return;
  }

  if(read->current->body->type != TYPEMULTIPART)
    read_add_this_part(read->read_attach,read);

  read_add_attach_strings(read->read_attach, read, 
			  read->current->attachments,0, 
			  (read->current->base_part_str) 
			  ? read->current->base_part_str : EMPTYSTR);

  read->current->button_state &= ~(BTN_NOATTACH);
  check_buttons(read_menu,XtNumber(read_menu),
		read->current->button_state);
  return;

}


void read_add_this_part(w,read)
     Widget w;
     Read *read;
{

  char buffer[FILEBUFFLEN];
  char partbuff[FILEBUFFLEN];
  Part_List *part_list;
  
  XmString xstr;
  int len;
  struct mail_body_parameter *params;

  strcpy(buffer,type_to_name(read->current->body->type));
  strcat(buffer,TYPE_SEPARATOR_STR);
  strcat(buffer,(read->current->body->subtype) 
	 ? read->current->body->subtype : NOSUBTYPE );

  if(read->current->rfc_msg_leaf == TRUE)
    strcpy(partbuff,read->current->base_part_str);
  else {
    if(read->current->base_part_str)
      sprintf(partbuff,"%s.1", read->current->base_part_str);
    else
      sprintf(partbuff,"1");
  }

  strcat(buffer," (Part ");
  strcat(buffer,partbuff);
  strcat(buffer,") ");

  for(len = strlen(buffer); len < (COMPOSEWIDTH/2); len ++)
    strcat(buffer,SPACESTR);
  if((read->current->body->description) && 
     (strlen(buffer) + strlen(read->current->body->description) < FILEBUFFLEN))
    strcat(buffer, read->current->body->description); 
  for(params = read->current->body->parameter; 
      params; params = params->next) {
    if((strlen(buffer) + strlen(params->attribute) 
	+ strlen(params->value) + 2) >= FILEBUFFLEN)
      break;
    strcat(buffer,SPACESTR);
    strcat(buffer,params->attribute);
    strcat(buffer,"=");
    strcat(buffer,params->value);
  }
  
  xstr = XmStringCreateSimple(buffer);
  XmListAddItemUnselected(w,xstr,0);
  XmStringFree(xstr);
  

  part_list = (Part_List *) fs_get(sizeof(Part_List));

  part_list->partstr = cpystr("1");
  part_list->partnumber = 1;

  part_list->next = NULL;
  part_list->prev = NULL;
  part_list->part = NULL;

  read->current->part_list = part_list;

}

void read_add_attach_strings(w,read,part,level, partstr)
     Widget w;
     Read *read;
     PART *part;
     int level;
     char *partstr;

{
  int len;
  PART *curr;
  char buffer[FILEBUFFLEN];
  char partbuff[FILEBUFFLEN];

  struct mail_body_parameter *params;
  XmString xstr;
  Part_List *part_list;
  Part_List *curr_part_list;
  int subpart_cnt = 0;

  if(part == NULL)
    return;

  strcpy(partbuff,EMPTYSTR);

  for(curr = part; curr; curr = curr->next) {
    subpart_cnt ++;

    if(*partstr == NUL_TERM)
      sprintf(partbuff,"%d",subpart_cnt);
    else
      sprintf(partbuff,"%s.%d",partstr,subpart_cnt);

    strcpy(buffer,EMPTYSTR);
    for(len = 0; len < level; len ++)
      strcat(buffer,SPACESTR);
    strcat(buffer,type_to_name(curr->body.type));
    strcat(buffer,TYPE_SEPARATOR_STR);
    strcat(buffer,(curr->body.subtype) ? curr->body.subtype : NOSUBTYPE );

    strcat(buffer," (Part ");
    strcat(buffer,partbuff);
    strcat(buffer,") ");

    for(len = strlen(buffer); len < (COMPOSEWIDTH/2); len ++)
      strcat(buffer,SPACESTR);
    if((curr->body.description) 
       && (strlen(buffer) + strlen(curr->body.description) < FILEBUFFLEN)) {
      strcat(buffer,curr->body.description);
      strcat(buffer,SPACESTR);
    }
    for(params = curr->body.parameter; params; params = params->next) {
      if((strlen(buffer) + strlen(params->attribute) 
	  + strlen(params->value) + 2) >= FILEBUFFLEN)
	break;
      strcat(buffer,params->attribute);
      strcat(buffer,"=");
      strcat(buffer,params->value);
      strcat(buffer,SPACESTR);
    }

    xstr = XmStringCreateSimple(buffer);
    XmListAddItemUnselected(w,xstr,0);
    XmStringFree(xstr);
    
    part_list = (Part_List *) fs_get(sizeof(Part_List));

    part_list->partstr = cpystr(partbuff);
    part_list->next = NULL;
    part_list->part = curr;
    
    if(read->current->part_list == NULL) {
      read->current->part_list = part_list;
      part_list->partnumber = 1;
      part_list->prev = NULL;
    }
    else {
      for(curr_part_list = read->current->part_list; curr_part_list->next;
	curr_part_list = curr_part_list->next);

      curr_part_list->next = part_list;
      part_list->prev = curr_part_list;
      part_list->partnumber = curr_part_list->partnumber + 1;
    }

    if(curr->body.type == TYPEMULTIPART)
      read_add_attach_strings(w, read, curr->body.contents.part,
			      level+1, partbuff);

  }

}



void read_reply_sender(w,read,e)
     Widget w;
     Read *read;
     XEvent *e;
{
  char buffer[FILEBUFFLEN];

  sprintf(buffer,"Reply to Sender");
  compose(buffer, MAIL_BEHAVIOUR, COMPOSE_REPLY, read->current, NULL, NULL);
  return;
}


void read_reply_all(w,read,e)
     Widget w;
     Read *read;
     XEvent *e;
{
  char buffer[FILEBUFFLEN];

  sprintf(buffer,"Reply to All");
  compose(buffer, MAIL_BEHAVIOUR, COMPOSE_REPLYALL, read->current, NULL, NULL);
  return;
}

void read_forward(w,read,e)
     Widget w;
     Read *read;
     XEvent *e;
{
  char buffer[FILEBUFFLEN];

  sprintf(buffer,"Forward Message");
  compose(buffer, MAIL_BEHAVIOUR, COMPOSE_FORWARD, read->current, NULL, NULL);
  return;
}

void read_forward_attach(w,read,e)
     Widget w;
     Read *read;
     XEvent *e;
{
  char buffer[FILEBUFFLEN];

  sprintf(buffer,"Forward Attached Message");
  compose(buffer, MAIL_BEHAVIOUR, COMPOSE_FORWARDATTACH, 
	  read->current, NULL, NULL);
  return;
}

void read_remail(w,read,e)
     Widget w;
     Read *read;
     XEvent *e;
{
  char buffer[FILEBUFFLEN];

  sprintf(buffer,"Remail Message");
  compose(buffer, MAIL_BEHAVIOUR, COMPOSE_REMAIL, read->current, NULL, NULL);
  return;
}

Menu read_list_menu[] = {
  { "Cancel", "read_list_cancel", 'C',
      read_list_cancel, NULL, 0, NULL, NULL, BTN_ON },
  { "Help", "read_list_HELP!", 'H',
      read_list_help, NULL, 0, NULL, NULL, BTN_ON },


};


void read_choosewin(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Position x, y;
  XtTranslations translations;
  Widget form, menubar, dummy;

  read->list_done = 0;
  set_pirate_cursors();

  get_pointer_position(w, &x, &y);

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  read->list_shell = XtCreatePopupShell("Selected Messages",
					 transientShellWidgetClass,
					 read->shell, args, n ); n = 0;

  form = XmCreateForm(read->list_shell, "form", args, n); n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form,"menubar", args, n); n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar,read_list_menu, 
		      XtNumber(read_list_menu), 0, (XtPointer) read);

  /* An invisible text field. This is used only to set the width. */

  XtSetArg(args[n], XmNcolumns, COMPOSEWIDTH - 20); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  dummy = XmCreateTextField(form,"dummy", args, n); n = 0;
  XtManageChild(dummy);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, menubar); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n ++;
  
  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmCONSTANT); n ++;
  XtSetArg(args[n], XmNvisibleItemCount, 10 ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmSINGLE_SELECT); n ++;



  read->list_list = 
    XmCreateScrolledList(form,"list",args,n); n = 0;
  XtAddCallback(read->list_list,
		XmNsingleSelectionCallback, read_list_select, read);
  XtAddCallback(read->list_list,
		XmNdefaultActionCallback, read_list_select, read);

  translations = 
    XtParseTranslationTable(GLOBAL_modal_list_translations);
  XtOverrideTranslations(read->list_list,translations);
  stuff_read_window_list(read);

  XtManageChild(read->list_list); 
  XtManageChild(form);
  XtPopup(read->list_shell, XtGrabExclusive);
  
  modal_main_loop(&read->list_done);


  XtPopdown(read->list_shell);
  XtDestroyWidget(read->list_shell);
  reset_cursors();

}

void read_list_select(w,read,xp)
     Widget w;
     Read *read;
     XmListCallbackStruct *xp;
{

  Read_Info *read_info;

  for(read_info = read->read_info;
      read_info->number != xp->item_position;
      read_info = read_info->prev);

  read_context_switch(read,read->current,read_info);
  read->list_done = 1;

}


void read_list_cancel(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{

  read->list_done = 1;

}

void read_list_help(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  help(read->list_shell,"Read Window Selections","readlist.help");

}

void stuff_read_window_list(read)
     Read *read;
{
  Read_Info *read_info;
  unsigned long n = 1L;
  char buffer[FILEBUFFLEN];
  XmString xstr;
  char from[40];
  char *subject;

  for(read_info = read->read_info;
      read_info; read_info = read_info->prev) {

    read_info->number = n;

    if(read_info->mailstream && read_info->envelope) {
      mail_fetchfrom(from,read_info->mailstream,read_info->msgno,36);
      subject = read_info->envelope->subject;
      if(subject == NULL)
	subject = EMPTYSTR;
      sprintf(buffer,"[%s] %-32.32s [%s] %-32.32s",
	      SUBJECT_TEXT,
	      (*subject) ? subject : "<unset>",
	      FROM_TEXT,
	      (*from) ? from : "<unset>");
    }
    else
      sprintf(buffer,"External or non-mailbox message");
    
    xstr = XmStringCreateSimple(buffer);
    XmListAddItemUnselected(read->list_list,xstr,0);
    XmStringFree(xstr);
    n ++;

  }
}


/* called from mm_expunged() */

void read_expunge(mailstream,msgno)
     MAILSTREAM *mailstream;
     unsigned long msgno;
{

  Read_Info *read_info;

  if((session.read == NULL) || (session.read->read_info == NULL))
    return;

  for(read_info = session.read->read_info; read_info;
      read_info = read_info->prev) {
    if(read_info->mailstream != mailstream)
      continue;
      
    if(read_info->msgno == msgno) {
      if(read_info == session.read->current) 
	read_kill_current(session.read->shell,session.read,NULL);
      else
	kill_read_info(read_info);
    }
    if(read_info->msgno > msgno)
      read_info->msgno --;
  }

}


void read_set_deleted(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_setflag(w,read,DELETED_FLAG);
}

void read_set_seen(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_setflag(w,read,SEEN_FLAG);
}

void read_set_flagged(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_setflag(w,read,FLAGGED_FLAG);
}

void read_set_answered(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_setflag(w,read,ANSWERED_FLAG);
}

void read_clear_deleted(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_clearflag(w,read,DELETED_FLAG);
}

void read_clear_seen(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_clearflag(w,read,SEEN_FLAG);
}

void read_clear_flagged(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_clearflag(w,read,FLAGGED_FLAG);
}

void read_clear_answered(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  read_clearflag(w,read,ANSWERED_FLAG);
}



void read_set_keyword(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  char *keyword = get_keyword(read->shell,read->current->mailstream);
  if(keyword) {
    read_setflag(w,read,keyword);
    fs_give((void **) &keyword);
    update_view_line_stream_msgno(read->current->mailstream,
				  read->current->msgno);
  }
}

void read_unset_keyword(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  char *keyword = get_keyword(read->shell,read->current->mailstream);
  if(keyword) {
    read_clearflag(w,read,keyword);
    fs_give((void **) &keyword);
    update_view_line_stream_msgno(read->current->mailstream,
				  read->current->msgno);
  }
}






void read_setflag(w,read,flag)
     Widget w;
     Read *read;
     char *flag;
{
  char sequence[32];

  if(read->current->mailstream && read->current->msgno) {
    sprintf(sequence,"%d",read->current->msgno);
    set_watch_cursors();
    mail_setflag(read->current->mailstream, sequence, flag);
    reset_cursors();
  }

  return;
}


void read_clearflag(w,read,flag)
     Widget w;
     Read *read;
     char *flag;
{
  char sequence[32];

  if(read->current->mailstream && read->current->msgno) {
    sprintf(sequence,"%d",read->current->msgno);
    set_watch_cursors();
    mail_clearflag(read->current->mailstream, sequence, flag);
    reset_cursors();
  }

  return;
}

void pgp_verify(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  pgp_decode_dispatch(w,read,PGP_VRFY);

}

void pgp_decode(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  pgp_decode_dispatch(w,read,PGP_DECR);

}


void pgp_addkey(w,read,xp)
     Widget w;
     Read *read;
     XtPointer xp;
{
  pgp_decode_dispatch(w,read,PGP_KADD);

}





void pgp_decode_dispatch(w,read,method)
     Widget w;
     Read *read;
     int method;
{

  FILE *fp;
  char *pw;
  Remote_Auth *remote_auth;
  char *text;
  char *response = NULL;
  char *infilename = NULL;
  char *outfilename = NULL;
  char *stdoutname = NULL;
  char innamebuff[MAXPATHLEN];
  char outnamebuff[MAXPATHLEN];
  char stdoutbuff[MAXPATHLEN];
  char buffer[MAXPATHLEN];
  Binary_Buffer *binary_buffer;
  Boolean is_pgp = FALSE;
  remote_auth = login(read->shell, "PGP", "PGP",
		      preferences.pgp_username, NULL);

  if(remote_auth == NULL)
    return;


  if(remote_auth->username == NULL || remote_auth->password == NULL) {
    free_Remote_Auth(remote_auth);
    return;
  }

  text = XmTextGetString(read->read_text);

  infilename = tmpnam(innamebuff);
  outfilename = tmpnam(outnamebuff);
  stdoutname = tmpnam(stdoutbuff);
  
  set_watch_cursors();


  switch(method) {
    
  case PGP_VRFY:
    is_pgp = ((strstr(text,PGP_SIGNATU_STR)) != NULL) ? TRUE : FALSE;
    if(is_pgp == FALSE)
      mm_log("No PGP signature found.", WARN);
    sprintf(buffer,PGP_VRFY_CMD,infilename,outfilename);
    break;
  case PGP_DECR:
    is_pgp = ((strstr(text,PGP_MESSAGE_STR)) != NULL) ? TRUE : FALSE;
    if(is_pgp == FALSE)
      mm_log("No PGP encoded message found.", WARN);
    sprintf(buffer,PGP_DECR_CMD,infilename,outfilename,stdoutname);
    break;
  case PGP_KADD:
    is_pgp = ((strstr(text,PGP_PUBLICK_STR)) != NULL) ? TRUE : FALSE;
    if(is_pgp == FALSE)
      mm_log("No PGP public key found.", WARN);
    sprintf(buffer,PGP_KADD_CMD, stdoutname);
    break;
  default:
    break;
    
  }

  if(is_pgp == TRUE) {
    
    mm_log("Executing PGP command:", NIL);
    mm_log(buffer,NIL);

    if(method != PGP_KADD) {
      if((fp = fopen(infilename,"w")) != NULL) {
	(void) chmod(infilename,S_IRWXU);
	fwrite(text,strlen(text),1,fp);
	(void) fclose(fp);
      }
    }
        
    pw = scramble(remote_auth->password);
    if((write_to_pipe(buffer, pw, 
		   (method == PGP_KADD) ?	 text : NULL, 
		   (method == PGP_KADD) ? strlen(text) : 0L)
	) == SYSCALL_SUCCESS) {
      wipeout(pw);
      binary_buffer = load_binary_file(outfilename);
      if(binary_buffer) {
	if(binary_buffer->data) {
	  if(strlen(binary_buffer->data) == binary_buffer->length) {
	    fs_give((void **) &read->current->alt_text);
	    read->current->alt_text = cpystr(binary_buffer->data);
	    XmTextSetString(read->read_text,binary_buffer->data);
	  }
	  else 
	    ask_save(read->shell,binary_buffer);
	}
	free_binary_buffer(binary_buffer);
      }
    }
    else {
      mm_log("PGP command failed.",WARN);
      wipeout(pw);
    }
    fs_give((void **) &pw);
    
    if((fp = fopen(stdoutname,"r")) != NULL) {
      while(fgets(buffer,sizeof(buffer),fp) != NULL)
	mm_log(buffer,MLNOTIFY);
      fclose(fp);
    }
  }

  fs_give((void **) &text);

  unlink(infilename);
  unlink(outfilename);  
  unlink(stdoutname);
  free_Remote_Auth(remote_auth);
  reset_cursors();

}


