/* mailboxes.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1994 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

void mblist_accept();
void mblist_cancel();
void mblist_help();
void mblist_create();
void mblist_delete();
void mblist_rename();
void mblist_setmail();
void mblist_setnews();
void mblist_select();
void sort_mailbox_list();
char *get_server_string_from_window();
char *get_mailbox_string_from_window();
void mblist_set_server();


Mailbox_Win *mailbox_win;



Menu mailboxes_option_menu[] = {
  { "Show Mailboxes", "mailboxes_mail", 'M',
      mblist_setmail, NULL, 0, NULL, NULL, BTN_MAILBOX_DOMAIL },

  { "Show Newsgroups", "mailboxes_news", 'N',
      mblist_setnews, NULL, 0, NULL, NULL, BTN_MAILBOX_DONEWS },

  { "Create/Subscribe Mailbox", "mailboxes_create", 'C',
      mblist_create, NULL, 0, NULL, NULL, BTN_ON },

  { "Delete/Unsubscribe Mailbox", "mailboxes_delete", 'D',
      mblist_delete, NULL, 0, NULL, NULL, BTN_ON },

  { "Rename Mailbox", "mailboxes_rename", 'R',
      mblist_rename, NULL, 0, NULL, NULL, BTN_MAILBOX_DONEWS },

};



Menu mailboxes_menu[] = {
  { "Accept", "mailboxes_accept", 'A',
      mblist_accept, NULL, 0, NULL, NULL, BTN_ON },

  { "Cancel", "mailboxes_create", 'C',
      mblist_cancel, NULL, 0, NULL, NULL, BTN_ON },

  { "Option", "mailboxes_option_menu", 'O',
      NULL, mailboxes_option_menu, XtNumber(mailboxes_option_menu),
      NULL, NULL, BTN_ON },

  { "Help", "mailboxes_HELP!", 'H',
      mblist_help, NULL, 0, NULL, NULL, BTN_ON },

};






void close_it(mailbox)
     Mailbox *mailbox;
{

  Lview *lview;
  Read_Info *read_info;
  Boolean found = FALSE;
  char log[FILEBUFFLEN];

  sprintf(log,"Closing mailbox '%s'.",mailbox->imapname);
  mm_log(log,NIL);

  XtRemoveTimeOut(mailbox->timer);

  /* 
   * Why? Well, if the mailbox is still there, we need to
   *  checkpoint any recent flag changes before closing.
   *  That's why.
   */
  
  if(mail_ping(mailbox->mailstream)) {
    mail_check(mailbox->mailstream);
    mail_close(mailbox->mailstream);
  }
  
  if(session.view) {
    do {
      for(lview = session.view->lview; lview ; lview = lview->prev)
	if(lview->mailstream == mailbox->mailstream) {
	  found = TRUE;
	  view_unlink_destroy_lview(session.view,lview,TRUE);
	}
	else
	  found = FALSE;
    } while((lview) && (found == TRUE));
  }

  if(session.read)
    for(read_info = session.read->read_info; read_info; 
	read_info = read_info->prev)
      if(read_info->mailstream == mailbox->mailstream) {
	if((session.read->is_realized) 
	   && (read_info == session.read->current))
	  read_close_kill_current(session.read->shell,session.read,NULL);
	else
	  kill_read_info(read_info);
      }
  
  compose_mailbox_close(mailbox->mailstream);

  if(mailbox == session.mailboxes)
    session.mailboxes = mailbox->next;
  
  if(mailbox->prev)
    mailbox->prev->next = mailbox->next;
  if(mailbox->next)
    mailbox->next->prev = mailbox->prev;
  
  free_mailbox(mailbox);

  if(session.mailboxes == NULL)
    session.button_state |= (BTN_NOMAILBOXESOPEN);
  
  update_main_buttons();
}



void close_invalid_mailbox(str)
     char *str;
{
  Mailbox *mailbox;
  Boolean found;

  do {
    found = FALSE;
    for(mailbox = session.mailboxes; mailbox; mailbox = mailbox->next) {
      if(strcmp(str,mailbox->imapname) == STRMATCH) {
	found = TRUE;
	close_it(mailbox);
      }
      if(found == TRUE)
	break;
    }
  } while(found == TRUE);
}



void mblist_create(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  char *server;
  char *server_str;
  char *mailbox_str;
  MAILSTREAM *mailstream = NULL;

  mailbox_str = get_mailbox_string_from_window(mailbox_win);
  if(mailbox_str == NULL)
    return;

  XDefineCursor(display, XtWindow(mailbox_win->shell), clock_cursor);

  server_str = get_server_string_from_window(mailbox_win);
  server = GetTextField(mailbox_win->server);

  if(mailbox_win->type == MAILBOX_TYPE_MAIL) {
    mailstream = mail_open(NIL,server_str,OP_HALFOPEN);
    mail_create(mailstream,mailbox_str);
    mail_subscribe(mailstream,mailbox_str);
    mail_close(mailstream);
  }
  else
    mail_subscribe_bboard(mailstream,mailbox_str + 1);


  stuff_mailbox_list(mailbox_win,server, TRUE);

  fs_give((void **) &server);
  fs_give((void **) &server_str);
  fs_give((void **) &mailbox_str);

  XUndefineCursor(display, XtWindow(mailbox_win->shell));

}


void mblist_delete(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  char *server;
  char *server_str;
  char *mailbox_str;
  MAILSTREAM *mailstream = NULL;

  mailbox_str = get_mailbox_string_from_window(mailbox_win);
  if(mailbox_str == NULL)
    return;

  close_invalid_mailbox(mailbox_str);

  XDefineCursor(display, XtWindow(mailbox_win->shell), clock_cursor);

  server_str = get_server_string_from_window(mailbox_win);
  server = GetTextField(mailbox_win->server);

  if(mailbox_win->type == MAILBOX_TYPE_MAIL) {
    mailstream = mail_open(NIL,server_str,OP_HALFOPEN);
    mail_unsubscribe(mailstream,mailbox_str);
    mail_delete(mailstream,mailbox_str);
    mail_close(mailstream);
  }
  else
    mail_unsubscribe_bboard(mailstream,mailbox_str + 1);

  stuff_mailbox_list(mailbox_win,server, TRUE);
  XmTextSetString(mailbox_win->mailbox,EMPTYSTR);

  fs_give((void **) &server);
  fs_give((void **) &server_str);
  fs_give((void **) &mailbox_str);

  XUndefineCursor(display, XtWindow(mailbox_win->shell));

}


void mblist_rename(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  char *server;
  char *server_str;
  char *mailbox_str;
  char *newname;
  char *fixednewname;
  char *formattednewname;

  MAILSTREAM *mailstream = NULL;
  
  if(mailbox_win->type == MAILBOX_TYPE_NEWS)
    return;


  XDefineCursor(display, XtWindow(mailbox_win->shell), pirate_cursor);

  

  newname = input_string(mailbox_win->shell,"New mailbox name", NULL, NULL);
  
  XUndefineCursor(display, XtWindow(mailbox_win->shell));
  
  if(newname == NULL) 
    return;

  if(*newname == NUL_TERM) {
    fs_give((void **) &newname);
    return;
  }


  mailbox_str = get_mailbox_string_from_window(mailbox_win);
  if(mailbox_str == NULL) {
    fs_give((void **) &newname);
    return;
  }

  close_invalid_mailbox(mailbox_str);


  XDefineCursor(display, XtWindow(mailbox_win->shell), clock_cursor);

  server_str = get_server_string_from_window(mailbox_win);
  server = GetTextField(mailbox_win->server);

  fixednewname = fix_mailboxpath(newname);
  fs_give((void **) &newname);
  formattednewname = 
    format_mailbox_name(server,fixednewname,mailbox_win->type);
  fs_give((void **) &fixednewname);

  if(strcmp(mailbox_str,formattednewname) != STRMATCH) {
    mailstream = mail_open(NIL,server_str,OP_HALFOPEN);
    mail_unsubscribe(mailstream,mailbox_str);
    mail_rename(mailstream,mailbox_str,formattednewname);
    mail_subscribe(mailstream,formattednewname);
    mail_close(mailstream);

    stuff_mailbox_list(mailbox_win,server, TRUE);
  }

  XmTextSetString(mailbox_win->mailbox,EMPTYSTR);


  fs_give((void **) &server);
  fs_give((void **) &server_str);
  fs_give((void **) &mailbox_str);
  fs_give((void **) &formattednewname);

  XUndefineCursor(display, XtWindow(mailbox_win->shell));

}

void mblist_setmail(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  char *fixedserver = get_qualified_host_name(preferences.imap_server);

  if(fixedserver) {
    XmTextSetString(mailbox_win->server,fixedserver);
    XmTextSetString(mailbox_win->mailbox, EMPTYSTR);

    mailbox_win->button_state |= BTN_MAILBOX_DOMAIL;
    mailbox_win->button_state &= ~(BTN_MAILBOX_DONEWS);
    
    mailbox_win->type = MAILBOX_TYPE_MAIL;
    stuff_mailbox_list(mailbox_win, fixedserver, FALSE);
    fs_give((void **) &fixedserver);
    check_buttons(mailboxes_menu,XtNumber(mailboxes_menu),
		  mailbox_win->button_state);

  }
}

void mblist_setnews(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  char *fixedserver = get_qualified_host_name(preferences.nntp_server);

  if(fixedserver) {
    XmTextSetString(mailbox_win->server,fixedserver);
    XmTextSetString(mailbox_win->mailbox, EMPTYSTR);

    mailbox_win->button_state |= BTN_MAILBOX_DONEWS;
    mailbox_win->button_state &= ~(BTN_MAILBOX_DOMAIL);

    mailbox_win->type = MAILBOX_TYPE_NEWS;
    stuff_mailbox_list(mailbox_win, fixedserver, FALSE);
    fs_give((void **) &fixedserver);

    check_buttons(mailboxes_menu,XtNumber(mailboxes_menu),
		  mailbox_win->button_state);
  }
}



void mblist_server(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  char *server = GetTextField(mailbox_win->server);
  char *fixedserver = get_qualified_host_name(server);

  if(fixedserver) {
    XmTextSetString(mailbox_win->server,fixedserver);
    XmTextSetString(mailbox_win->mailbox, EMPTYSTR);
    mailbox_win->type = MAILBOX_TYPE_MAIL;
    stuff_mailbox_list(mailbox_win, fixedserver, FALSE);
    fs_give((void **) &fixedserver);
  }
  XmProcessTraversal(mailbox_win->shell, XmTRAVERSE_NEXT_TAB_GROUP);
  fs_give((void **) &server);
}



void mblist_accept(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  mailbox_win->done = 1;
}


void mblist_cancel(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XtPointer xp;
{
  XmTextSetString(mailbox_win->mailbox,EMPTYSTR);
  mailbox_win->done = 1;
}

void mblist_help()
{

  help(mailbox_win->shell,"Mailbox Selection", "mailboxes.help");

}




void mblist_select(w,mailbox_win,xp)
     Widget w;
     Mailbox_Win *mailbox_win;
     XmListCallbackStruct *xp;
{
  char *str;

  XmStringGetLtoR(xp->item, XmSTRING_DEFAULT_CHARSET, &str);
  XmTextSetString(mailbox_win->mailbox,str);
  fs_give((void **) &str);

}


char *fix_mailboxpath(str)
     char * str;
{
  char *ret = NULL;

  if(str == NULL || *str == NUL_TERM)
    return(cpystr(""));

  if((strcasecmp(str,"INBOX")) == 0)
    ucase(str);

  if(*str != '+')
    return(cpystr(str));

  str ++;
  if(*str == '+')
    return(cpystr(str));

  if((preferences.folder_path == NULL) 
     || (*preferences.folder_path == NUL_TERM))
    return(cpystr(str));

  ret = (char *) fs_get(strlen(str) + strlen(preferences.folder_path) + 2);

  strcpy(ret,preferences.folder_path);

  if(ret[strlen(ret) - 1] != '/')
    strcat(ret,"/");
  strcat(ret,str);
  return(ret);
}


char *unfix_mailboxpath(str)
     char *str;
{

  char *buf = NULL;
  char *ret = NULL;

  if(str == NULL || *str == '\0')
    return(cpystr(""));

  if((preferences.folder_path == NULL) 
     || (*preferences.folder_path == NUL_TERM))
    return(cpystr(str));

  buf = (char *) fs_get(strlen(str) + strlen(preferences.folder_path) + 2);
  ret = (char *) fs_get(strlen(str) + strlen(preferences.folder_path) + 2);

  strcpy(buf,preferences.folder_path);

  if(buf[strlen(buf) - 1] != '/')
    strcat(buf,"/");

  if(strncmp(str,buf,strlen(buf)) == STRMATCH) {
    strcpy(ret,"+");
    strcat(ret,str + strlen(buf));
    fs_give((void **) &buf);
  }
  else {
    fs_give((void **) &buf);
    strcpy(ret,str);
  }
  return(ret);
}


Server_List *get_active_server_list()
{
  Server_List *server_list = NULL;
  for(server_list = session.server_list;
      server_list; server_list = server_list->prev) 
    if(server_list->active == TRUE)
	return(server_list);
  return(NULL);
}


Server_List *find_server_list(server,type)
     char *server;
     int type;
{
  Server_List *server_list = NULL;
  for(server_list = session.server_list;
      server_list; server_list = server_list->prev) 
    if((strcasecmp(server_list->name,server) == STRMATCH) &&
      (server_list->type == type))
	return(server_list);
  return(NULL);
}




char *get_mailbox_name(w,mailbox,local_only,default_server,default_mailbox)
     Widget w;
     Mailbox *mailbox;
     Boolean local_only;
     char *default_server;
     char *default_mailbox;
{

  Arg args[ARGLISTSIZE];
  int n = 0;
  Position x, y;
  Widget form, menubar;
  XtTranslations translations;
  char *ret = NULL;
  char *server;
  char *mailboxname;
  char *fixedname;
  char *fixedserver;

  set_pirate_cursors();

  mailbox_win = (Mailbox_Win *) fs_get(sizeof(Mailbox_Win));

  mailbox_win->done = 0;
  mailbox_win->button_state = BTN_ON | BTN_MAILBOX_DOMAIL;
  if(local_only) 
    mailbox_win->button_state |= BTN_MAILBOX_NOEDIT;

  mailbox_win->type = MAILBOX_TYPE_MAIL;

  get_pointer_position(w, &x, &y);

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  mailbox_win->shell = XtCreatePopupShell("Mailboxes",
					  transientShellWidgetClass, w,
					  args, n); n = 0;

  XmAddTabGroup(mailbox_win->shell);

  form = XmCreateForm(mailbox_win->shell, "form", args, n ); n = 0;

  XtSetArg(args[n], XmNtopAttachment,   XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment,  XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form, "menubar", args, n); n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar, mailboxes_menu, 
		      XtNumber(mailboxes_menu),
		      mailbox_win->button_state, mailbox_win);



  fixedserver = get_qualified_host_name(default_server);
  mailbox_win->server =
    create_text_field(form, menubar, 
		      "Server   :", fixedserver,40, 
		      mblist_server, mailbox_win);

  if(local_only) {
    XtSetArg(args[n], XmNeditable, FALSE ); n ++;
    XtSetValues(mailbox_win->server, args, n); n = 0;
  }

  fixedname = unfix_mailboxpath(default_mailbox);

  mailbox_win->mailbox =
    create_text_field(form, mailbox_win->server, 
		      "Mailbox  :", fixedname,
		      40, mblist_accept, mailbox_win);

  fs_give((void **) &fixedname);

  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmVARIABLE); n ++;
  XtSetArg(args[n], XmNvisibleItemCount, 10 ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmSINGLE_SELECT); n ++;
  mailbox_win->list = XmCreateScrolledList(form,"list",args,n); n = 0;
  XtAddCallback(mailbox_win->list, XmNsingleSelectionCallback, 
		mblist_select, mailbox_win);
  XtAddCallback(mailbox_win->list, XmNdefaultActionCallback, 
		mblist_accept, mailbox_win);

  translations = 
    XtParseTranslationTable(GLOBAL_modal_list_translations);
  XtOverrideTranslations(mailbox_win->list,translations);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, mailbox_win->mailbox); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n ++;

  XtSetValues(XtParent(mailbox_win->list), args, n); n = 0;
  XtManageChild(mailbox_win->list);

  XtManageChild(form);
  XtManageChild(mailbox_win->shell);

  XtPopup(mailbox_win->shell,XtGrabExclusive);

  stuff_mailbox_list(mailbox_win, fixedserver, FALSE);
  fs_give((void **) &fixedserver);

  /* jump to mailbox field */

  XmProcessTraversal(mailbox_win->shell, XmTRAVERSE_NEXT_TAB_GROUP);
  XmProcessTraversal(mailbox_win->shell, XmTRAVERSE_NEXT_TAB_GROUP);

  modal_main_loop(&mailbox_win->done);

  XtPopdown(mailbox_win->shell);

  server = GetTextField(mailbox_win->server);
  fixedserver = get_qualified_host_name(server);

  mailboxname = GetTextField(mailbox_win->mailbox);
  fixedname = fix_mailboxpath(mailboxname);

  if(mailbox) {
    mailbox->host = cpystr(fixedserver);
    mailbox->mailboxname = cpystr(fixedname);
    mailbox->type = mailbox_win->type;
  }


  if(*fixedname != NUL_TERM) {
    if(local_only)
      ret = cpystr(fixedname);
    else
      ret = format_mailbox_name(fixedserver, fixedname, mailbox_win->type);
  }

  XtDestroyWidget(mailbox_win->shell);

  fs_give((void **) &server);
  fs_give((void **) &fixedserver);
  fs_give((void **) &mailboxname);
  fs_give((void **) &fixedname);
  fs_give((void **) &mailbox_win);

  reset_cursors();

  return(ret);

}



void free_mailbox_list(mailbox_list)
     Mailbox_List *mailbox_list;
{
  if(mailbox_list == NULL)
    return;

  free_mailbox_list(mailbox_list->prev);
  fs_give((void **) &mailbox_list->name);
  fs_give((void **) &mailbox_list);

  mailbox_list = NULL;
}




stuff_mailbox_list(mailbox_win,server, force)
     Mailbox_Win *mailbox_win;
     char *server;
     Boolean force;

{
  MAILSTREAM *mailstream;
  Server_List *server_list;
  Mailbox_List *mailbox_list;
  XmString xstr;
  char *fixedname;
  char *imap_server_str;
  Boolean new_server = FALSE;

  XmListDeselectAllItems(mailbox_win->list);
  XmListDeleteAllItems(mailbox_win->list);

  XDefineCursor(display, XtWindow(mailbox_win->shell), clock_cursor);

  if(add_server_to_session(server,mailbox_win->type))
    new_server = TRUE;

  server_list = find_server_list(server,mailbox_win->type);

  if(force) {
    free_mailbox_list(server_list->mailbox_list);
    server_list->mailbox_list = NULL;
  }

  if(new_server == TRUE || force) {
    server_list->active = TRUE;
    if(mailbox_win->type == MAILBOX_TYPE_MAIL) {
      imap_server_str = format_server_name(server, MAILBOX_TYPE_MAIL);
      mailstream = mail_open(NIL,imap_server_str,OP_HALFOPEN);
      mail_find(mailstream,"*");
      mail_close(mailstream);
      fs_give((void **) &imap_server_str);
    }
    else {
      imap_server_str = format_server_name(server, MAILBOX_TYPE_NEWS);
      mailstream = mail_open(NIL,imap_server_str,OP_HALFOPEN);
      mail_find_bboards(mailstream,"*");
      mail_close(mailstream);
      fs_give((void **) &imap_server_str);
    }

    server_list->active = FALSE;
  }

  if(! server_list) {
    XUndefineCursor(display, XtWindow(mailbox_win->shell));
    return;
  }

  if(server_list->mailbox_list) {
    sort_mailbox_list(server_list);
    for(mailbox_list = server_list->mailbox_list; mailbox_list;
	mailbox_list = mailbox_list->prev) {
      fixedname = unfix_mailboxpath(mailbox_list->name);
      xstr = XmStringCreateSimple(fixedname);
      fs_give((void **) &fixedname);
      XmListAddItemUnselected(mailbox_win->list,xstr,0);
      XmStringFree(xstr);
    }
  }
  XUndefineCursor(display, XtWindow(mailbox_win->shell));
}


int mailbox_compare(name1,name2)
     Mailbox_List **name1, **name2;
{
  char *fixedname1;
  char *fixedname2;
  int result;

  fixedname1 = unfix_mailboxpath((*(name1))->name);
  fixedname2 = unfix_mailboxpath((*(name2))->name);

  result = strcmp(fixedname1, fixedname2);

  /* ensure INBOX is first in the list */
  
  if(strcasecmp(fixedname1,INBOX) == STRMATCH)
    result = (-1);

  if(strcasecmp(fixedname2,INBOX) == STRMATCH)
    result = 1;

  fs_give((void **) &fixedname1);
  fs_give((void **) &fixedname2);

  return(result);

}



void sort_mailbox_list(server_list)
     Server_List *server_list;
{
  Mailbox_List **sort;
  Mailbox_List *mailbox_list;

  int n = 0;
  int total = 0;


  for(mailbox_list = server_list->mailbox_list; 
      mailbox_list; mailbox_list = mailbox_list->prev)
    total ++;

  if(total < 2)
    return;

  sort = (Mailbox_List **) fs_get(total * sizeof(Mailbox_List *));

  
  for(mailbox_list = server_list->mailbox_list; 
      mailbox_list; mailbox_list = mailbox_list->prev) {
    sort[n] = mailbox_list;
    n ++;
  }

  qsort((char *) sort, total, sizeof(Mailbox_List **), mailbox_compare);

  for(n = 0; n < total; n ++) {
    mailbox_list = sort[n];
    if(n == 0)
      server_list->mailbox_list = mailbox_list;
    if(n < (total - 1))
      mailbox_list->prev = sort[n + 1];
    else
      mailbox_list->prev = NULL;
  }
  
  fs_give((void **) &sort);

}






char *format_server_name(server,type)
     char *server;
     int type;
{

  char buff[FILEBUFFLEN];
  char *ret;

  if(type == MAILBOX_TYPE_MAIL) { 
    if((server) && (preferences.imap_port != DEFAULT_IMAPPORT))
      sprintf(buff,"{%s:%d}", server, preferences.imap_port);
    else {
      if(server)
	sprintf(buff,"{%s}", server);
    }
  }
  else {
    if(server) 
      sprintf(buff,"*{%s/nntp}",server);
  }
  ret = cpystr(buff);
  return(ret);

}


char *format_mailbox_name(server,mailbox,type)
     char *server;
     char *mailbox;
     int type;
{

  char buff[FILEBUFFLEN];
  char *ret;

  if(type == MAILBOX_TYPE_MAIL) {   /* regular mailbox */
    if((server) && (preferences.imap_port != DEFAULT_IMAPPORT)) {
      sprintf(buff,"{%s:%d}%s", server, preferences.imap_port, mailbox);
    }
    else {
      if(server)
	sprintf(buff,"{%s}%s",
		server, mailbox);
      else
	sprintf(buff,"%s",mailbox);
    }
  }
  else {
    if(server) 
      sprintf(buff,"*{%s/nntp}%s",server,mailbox);
    else
      sprintf(buff,"*%s",mailbox);
  }

  ret = cpystr(buff);
  return(ret);

}



char *get_mailbox_string_from_window(mailbox_win)
     Mailbox_Win *mailbox_win;
{
  char *server;
  char *mailbox;
  char *fixedmailbox;
  char *ret = NULL;

  mblist_set_server(mailbox_win);
  server = GetTextField(mailbox_win->server);
  if(*server == NUL_TERM) {
    fs_give((void **) &server);
    server = NULL;
  }

  mailbox = GetTextField(mailbox_win->mailbox);
  if(*mailbox != NUL_TERM) {
    fixedmailbox = fix_mailboxpath(mailbox);
    ret = format_mailbox_name(server,fixedmailbox,mailbox_win->type);
    fs_give((void **) &fixedmailbox);
  }

  if(server)
    fs_give((void **) &server);
  fs_give((void **) &mailbox);

  return(ret);

}


char *get_server_string_from_window(mailbox_win)
     Mailbox_Win *mailbox_win;
{
  char *rawname;
  char *fullname;

  mblist_set_server(mailbox_win);

  rawname = GetTextField(mailbox_win->server);
  if(*rawname == NUL_TERM)
    return(rawname);

  fullname = format_server_name(rawname,mailbox_win->type);
  fs_give((void **) &rawname);
  return(fullname);

}


void mblist_set_server(mailbox_win)
     Mailbox_Win *mailbox_win;
{
  char *rawname;
  char *fullname;

  rawname = GetTextField(mailbox_win->server);
  if(*rawname != NUL_TERM) {
    fullname = get_qualified_host_name(rawname);
    XmTextSetString(mailbox_win->server,fullname);
    fs_give((void **) &fullname);
  }

  fs_give((void **) &rawname);

}


