/* edfilter.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1994 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

void edfilter_destroy();
void edfilter_dismiss();
void edfilter_accept();
void edfilter_help();
void stuff_edfilter_list();
void edfilter_select();
void edfilter_add();
void edfilter_reset();
void edfilter_context_switch();
void edfilter_modify();
void edfilter_delete();
void edfilter_attach_one_current();
void edfilter_attach_current();
void edfilter_attach_one_mailbox();
void edfilter_attach_mailbox();
void edfilter_attach_dispatch();


Menu edfilter_file_menu[] = {
  { "Add Filter", "edfilter_add", 'A',
      edfilter_add, NULL, 0, NULL, NULL, BTN_SELECTION },
  { "Remove Filter", "edfilter_delete", 'R',
      edfilter_delete, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Modify Filter", "edfilter_modify", 'M',
      edfilter_modify, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Close Window", "edfilter_dismiss", 'C',
      edfilter_dismiss, NULL, 0, NULL, NULL, BTN_ON },

};

Menu edfilter_attach_once_menu[] = {
  { "Apply to Current View", "edfilter_attach_one_current", 'V',
      edfilter_attach_one_current, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Apply to Current Mailbox", "edfilter_attach_one_mailbox", 'M',
      edfilter_attach_one_mailbox, NULL, 0, NULL, NULL, BTN_NOSELECTION },

};

Menu edfilter_attach_menu[] = {
  { "Apply to Current View", "edfilter_attach_current", 'V',
      edfilter_attach_current, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Apply to Current Mailbox", "edfilter_attach_mailbox", 'M',
      edfilter_attach_mailbox, NULL, 0, NULL, NULL, BTN_NOSELECTION },

};


Menu edfilter_option_menu[] = {
  { "One Time Only", "edfilter_attach_once_menu", 'O',
     NULL, edfilter_attach_once_menu, XtNumber(edfilter_attach_once_menu),
      NULL, NULL, BTN_NOSELECTION },
  { "Continuous Update", "edfilter_attach_menu", 'O',
     NULL, edfilter_attach_menu, XtNumber(edfilter_attach_menu),
      NULL, NULL, BTN_NOSELECTION },
};


Menu edfilter_menu[] = {

  { "File", "edfilter_file_menu", 'F',
      NULL, edfilter_file_menu, XtNumber(edfilter_file_menu),
      NULL, NULL, BTN_ON },

  { "Apply", "edfilter_apply", 'A',
      NULL, edfilter_option_menu, XtNumber(edfilter_option_menu),
      NULL, NULL, BTN_NOSELECTION },

  { "Reset", "edfilter_reset", 'R',
      edfilter_reset, NULL, 0, NULL, NULL, BTN_ON },

  { "Help", "edfilter_HELP!", 'H',
      edfilter_help, NULL, 0, NULL, NULL, BTN_ON },

};





Boolean create_edfilter_window(w)
     Widget w;
{

  Arg args[ARGLISTSIZE];
  int n = 0;
  Edfilter *edfilter;
  Position x, y;
  Widget form, menubar,label1;
  XtTranslations translations;


  if(session.edfilter == NULL) {
    edfilter = (Edfilter *) fs_get(sizeof(Edfilter));
    edfilter->is_realized = FALSE;
    session.edfilter = edfilter;
  }
  else
    edfilter = session.edfilter;

  
  if(session.edfilter->is_realized == TRUE) {
    XRaiseWindow(display,XtWindow(edfilter->shell));
    return;
  }

  edfilter->selection = 0L;
  edfilter->button_state = BTN_ON;
  edfilter->current = NULL;

  get_pointer_position(w, &x, &y);

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  edfilter->shell = XtCreateWidget("Filter Edit",
				   topLevelShellWidgetClass, top,
				   args, n); n = 0;

  XmAddTabGroup(edfilter->shell);
  form = XmCreateForm(edfilter->shell, "form", args, n ); n = 0;

  XtSetArg(args[n], XmNtopAttachment,   XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment,  XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form, "menubar", args, n); n = 0;
  XtManageChild(menubar);

  create_menu_buttons(NULL, menubar, edfilter_menu, 
		      XtNumber(edfilter_menu),
		      edfilter->button_state, edfilter);

  edfilter->name =
    create_text_field(form, menubar, 
		      "Name   :", NULL, 40, NULL, NULL);
  XtAddCallback(edfilter->name, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_nospace_edit, NULL);

  edfilter->action =
    create_text_field(form, edfilter->name, 
		      "Action :", preferences.default_action, 40, NULL, NULL);


  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET);         n ++;
  XtSetArg(args[n], XmNtopWidget, edfilter->action);            n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);          n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);         n ++;

  label1 = XmCreateLabel(form,"Filter Definition:",args,n);     n = 0;

  XtManageChild(label1);

  XtSetArg(args[n], XmNeditable, TRUE);                         n ++;
  XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT);            n ++;
  XtSetArg(args[n], XmNwordWrap, TRUE);                         n ++;
  XtSetArg(args[n], XmNrows,     4);                            n ++;
  XtSetArg(args[n], XmNcolumns,  64);                           n ++;
  XtSetArg(args[n], XmNscrollVertical, TRUE);                   n ++; 
  XtSetArg(args[n], XmNscrollHorizontal, FALSE );               n ++;

  edfilter->text =
    XmCreateScrolledText(form, "filter_text", args, n);        n = 0;
  
  translations = XtParseTranslationTable(GLOBAL_text_translations);
  XtOverrideTranslations(edfilter->text, translations);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET);         n ++;
  XtSetArg(args[n], XmNtopWidget, label1);                      n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);          n ++;

  XtSetValues(XtParent(edfilter->text), args, n);        n = 0;
  XtManageChild(edfilter->text);


  XtAddCallback(edfilter->text, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_field_edit, NULL);


  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC);             n ++;
  XtSetArg(args[n], XmNlistSizePolicy,         XmCONSTANT);           n ++;
  XtSetArg(args[n], XmNvisibleItemCount,       10 );                  n ++; 
  XtSetArg(args[n], XmNselectionPolicy,        XmSINGLE_SELECT);      n ++;
  edfilter->list = XmCreateScrolledList(form,"list",args,n);      n = 0;

  XtAddCallback(edfilter->list, XmNsingleSelectionCallback, 
		edfilter_select, edfilter);
  
  XtAddCallback(edfilter->list, XmNdefaultActionCallback, 
		edfilter_accept, edfilter);

  XtSetArg(args[n], XmNtopAttachment,          XmATTACH_WIDGET);       n ++;
  XtSetArg(args[n], XmNtopWidget,              edfilter->text); n ++;
  XtSetArg(args[n], XmNleftAttachment,         XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNrightAttachment,        XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNbottomAttachment,       XmATTACH_FORM);         n ++;

  XtSetValues(XtParent(edfilter->list), args, n);                  n = 0;

  stuff_edfilter_list(edfilter,FALSE,NULL);


  XtAddCallback(edfilter->shell, XmNdestroyCallback, 
		(XtCallbackProc) edfilter_destroy, edfilter);

  XtManageChild(edfilter->list);
  XtManageChild(form);
  XtManageChild(edfilter->shell);
  XtRealizeWidget(edfilter->shell);
  edfilter->is_realized = TRUE;

}

void edfilter_reset(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;

{
  XmListDeselectAllItems(edfilter->list);
  edfilter->current = NULL;
  edfilter->selection = 0L;
  XmTextSetString(edfilter->name,EMPTYSTR);
  XmTextSetString(edfilter->action,EMPTYSTR);
  XmTextSetString(edfilter->text,EMPTYSTR);
  edfilter->button_state |= BTN_NOSELECTION;
  edfilter->button_state &= ~(BTN_SELECTION);
  check_buttons(edfilter_menu,XtNumber(edfilter_menu),edfilter->button_state);

}

void edfilter_destroy(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{
  edfilter->selection = 0L;
  edfilter->is_realized = FALSE;

}

void edfilter_help(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  help(edfilter->shell,"Filters","filteredit.help");

}



int edfilter_compare(flt1,flt2)
     Filter **flt1, **flt2;
{

  return(strcmp((*(flt1))->name,(*(flt2))->name));


}



void edfilter_sort(edfilter)
     Edfilter *edfilter;
{
  Filter **sort;
  Filter *info;
  int n = 0;
  int total = 0;

  for(info = session.filters; info; info = info->next)
    total ++;

  if(total < 2)
    return;

  sort = (Filter **) fs_get(total * sizeof(Filter *));

  for(info = session.filters; info; info = info->next) {
    sort[n] = info; 
    n ++;
  }
  qsort((char *) sort, total, sizeof(Filter **), edfilter_compare);

  for(n = 0; n < total; n ++) {
    info = sort[n];
    if(n == 0) {
      session.filters = info;
      info->prev = NULL;
      info->next = sort[n + 1];
    }
    else {
      info->prev = sort[n - 1];
      if(n == (total - 1))
	info->next = NULL;
      else
	info->next = sort[n + 1];
    }
  }

  fs_give((void **) &sort);

}



void stuff_edfilter_list(edfilter,update,active)
     Edfilter *edfilter;
     Boolean update;
     Filter *active;

{
  Filter *filter;
  XmString xstr;
  int n = 0;
  Boolean keep_active = FALSE;


  edfilter_sort(edfilter);

  if(update) {
    XmListDeselectAllItems(edfilter->list);
    XmListDeleteAllItems(edfilter->list);
  }

  for(filter = session.filters; filter; filter = filter->next) {
    filter->number = (++n);
    xstr = XmStringCreateSimple(filter->name);
    XmListAddItemUnselected(edfilter->list,xstr,0);
    XmStringFree(xstr);

    if((active != NULL) && (filter == active)) {
      XmListSelectPos(edfilter->list,filter->number,FALSE);
      edfilter->selection = filter->number;
      keep_active = TRUE;
      edfilter->button_state &= ~(BTN_NOSELECTION);
      edfilter->button_state |= BTN_NOSELECTION;
    }
  }
  if(keep_active == FALSE) {
    edfilter->button_state |= BTN_NOSELECTION;
    edfilter->button_state &= ~(BTN_SELECTION);
  }
  else
    edfilter_context_switch(edfilter,edfilter->current,active);

  check_buttons(edfilter_menu,XtNumber(edfilter_menu),
		edfilter->button_state);

}


void edfilter_select(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XmListCallbackStruct *xp;
{
  Filter *filter;
  
  for(filter = session.filters; filter; filter = filter->next) {
    if(filter->number == xp->item_position) {
      if(filter->number == edfilter->selection) {
	edfilter_reset(w,edfilter,xp);
	break;
      }
      else {
	edfilter->selection = filter->number;
	edfilter_context_switch(edfilter,edfilter->current,filter);
	break;
      }
    }
  }


}

Boolean duplicate_filter_name(s)
     char *s;
{
  Filter *filter;

  for(filter = session.filters; filter; filter = filter->next) {
    if((strcmp(s,filter->name)) == STRMATCH)
      return(TRUE);
  }

  if((strcmp(s,"ALL")) == STRMATCH)
    return(TRUE);

  return(FALSE);

}

void edfilter_add(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  Filter *filter = new_filter();
  Filter *prev = NULL;

  filter->filter_type = FILTER_TYPE_USER;

  filter->name = GetTextField(edfilter->name);
  if((filter->name == NULL)  
     || (*filter->name == NUL_TERM)) {
    free_filter(filter);
    return;
  }

  if((duplicate_filter_name(filter->name)) == TRUE) {
    mm_log("Duplicate filter names not allowed.",WARN);
    free_filter(filter);
    return;
  }

  filter->action = GetTextField(edfilter->action);
  filter->text = GetTextField(edfilter->text);

  if(session.filters)
    for(prev = session.filters; prev->next; prev = prev->next)
      ;

  if(prev == NULL) {
    session.filters = filter;
    filter->prev = NULL;
  }
  else {
    filter->prev = prev;
    prev->next = filter;
  }

  save_defaults();
  stuff_edfilter_list(edfilter,TRUE,filter);

}


void edfilter_modify(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  if(edfilter->current == NULL)
    return;

  if(edfilter->current->name) 
    fs_give((void **) &edfilter->current->name);
  if(edfilter->current->action)
    fs_give((void **) &edfilter->current->action);
  if(edfilter->current->text)
    fs_give((void **) &edfilter->current->text);

  edfilter->current->name   = GetTextField(edfilter->name);
  edfilter->current->action = GetTextField(edfilter->action);
  edfilter->current->text   = GetTextField(edfilter->text);

  save_defaults();
  stuff_edfilter_list(edfilter,TRUE,edfilter->current);

}


void edfilter_delete(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  Filter *filter;

  if(edfilter->current == NULL)
    return;

  if((edfilter->current->filter_type != FILTER_TYPE_USER)
     && (edfilter->current->filter_type != FILTER_TYPE_ONESHOT))
    return;
  for(filter = session.filters; filter; filter = filter->next) {
    if(filter == edfilter->current) {
      if(filter == session.filters) {
	session.filters = filter->next;
	if(filter->next)
	  filter->next->prev = NULL;
      }
      else {
	if(filter->next)
	  filter->next->prev = filter->prev;
	if(filter->prev)
	  filter->prev->next = filter->next;
      }
      free_filter(filter);
      break;
    }
  }

  edfilter_reset(w,edfilter,xp);
  save_defaults();
  stuff_edfilter_list(edfilter,TRUE,NULL);

}




void edfilter_accept(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  if(! edfilter->current)
    return;
  if(edfilter->current->action) 
    fs_give((void **) &edfilter->action);
  edfilter->current->action = GetTextField(edfilter->action);

  if(edfilter->current->filter_type == FILTER_TYPE_USER) {
    if(edfilter->current->name)
      fs_give((void **) &edfilter->current->name);
    edfilter->current->name = GetTextField(edfilter->name);
    if(edfilter->current->text)
      fs_give((void **) &edfilter->current->text);
    edfilter->current->text = GetTextField(edfilter->text);
    striplf(edfilter->current->text);

  }

  save_defaults();

  XtDestroyWidget(edfilter->shell);
  edfilter->selection = 0L;
  edfilter->is_realized = FALSE;

}






void edfilter_dismiss(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  XtDestroyWidget(edfilter->shell);
  edfilter->selection = 0L;
  edfilter->is_realized = FALSE;

}




void edfilter_context_switch(edfilter,old,new)
     Edfilter *edfilter;
     Filter *old;
     Filter *new;
{
  Arg args[ARGLISTSIZE];
  int n = 0;

  edfilter->current = new;

  XmTextSetString(edfilter->name,new->name);
  XmTextSetString(edfilter->action,new->action);
  XmTextSetString(edfilter->text,new->text);

  if(new->filter_type == FILTER_TYPE_USER) 
    XtSetArg(args[n], XmNeditable, TRUE); 
  else
    XtSetArg(args[n], XmNeditable, FALSE);
  n ++;
  XtSetValues(edfilter->text, args, n);
  XtSetValues(edfilter->name, args, n);
  n = 0;

  if(edfilter->selection) {
    edfilter->button_state |= BTN_SELECTION;
    edfilter->button_state &= ~(BTN_NOSELECTION);
  }
  check_buttons(edfilter_menu,XtNumber(edfilter_menu),edfilter->button_state);

}




void edfilter_attach_one_current(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  edfilter_attach_dispatch(w,edfilter,(Attach_Operation) ATTACH_ONCE_CURRENT);
  
}

void edfilter_attach_one_mailbox(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  edfilter_attach_dispatch(w,edfilter,(Attach_Operation) ATTACH_ONCE_MAILBOX);
  
}

void edfilter_attach_current(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  edfilter_attach_dispatch(w,edfilter,(Attach_Operation) ATTACH_CURRENT);
  
}

void edfilter_attach_mailbox(w,edfilter,xp)
     Widget w;
     Edfilter *edfilter;
     XtPointer xp;
{

  edfilter_attach_dispatch(w,edfilter,(Attach_Operation) ATTACH_MAILBOX);
  
}

Boolean filter_has_changed(edfilter)
     Edfilter *edfilter;
{
  char *name;
  char *action;
  char *text;

  Boolean different = FALSE;

  name   = GetTextField(edfilter->name);
  action = GetTextField(edfilter->action);
  text   = GetTextField(edfilter->text);

  if(strcmp(name,edfilter->current->name) != STRMATCH)
    different = TRUE;
  if(strcmp(action,edfilter->current->action) != STRMATCH)
    different = TRUE;
  if(strcmp(text,edfilter->current->text) != STRMATCH)
    different = TRUE;

  fs_give((void **) &name);
  fs_give((void **) &action);
  fs_give((void **) &text);

  return different;

}

void edfilter_attach_dispatch(w,edfilter,operation)
     Widget w;
     Edfilter *edfilter;
     Attach_Operation operation;
{

  Lview *lview;
  Lview *parent;
  Filter *filter;
  parse_errors errors;
  char errmsg[FILEBUFFLEN];
  char *temp_txt;

  if((! edfilter->selection) || (edfilter->current == NULL))
    return;

  if((filter_has_changed(edfilter)) == TRUE)
    mm_log("Filter modified but not saved.", WARN);

  filter = new_filter();

  filter->name = GetTextField(edfilter->name);
  temp_txt = GetTextField(edfilter->text);
  filter->action = GetTextField(edfilter->action);

  errors = lv_parse_filter(filter->name, temp_txt,
			   (LEAF **) &filter->search_tree,
			   &filter->text,
			   &filter->search_types,
			   errmsg);

  if(errors != parse_success) {
    free_filter(filter);
    fs_give((void **) &temp_txt);
    mm_log(errmsg,WARN);
    return;
  }

  lview = new_lview();

  switch(operation) {
    
  case ATTACH_ONCE_CURRENT:
      filter->filter_type = FILTER_TYPE_ONESHOT;
      lview->parent = session.view->current;
      lview->level = lview->parent->level + 1;
      break;

  case ATTACH_ONCE_MAILBOX:
      filter->filter_type = FILTER_TYPE_ONESHOT;
      lview->parent = 
	get_all_lview(session.view->current->mailbox->mailstream);
      lview->level = lview->parent->level + 1;
      break;

  case ATTACH_CURRENT:
      filter->filter_type = FILTER_TYPE_USER;
      lview->parent = session.view->current;
      lview->level = lview->parent->level + 1;
      break;

  case ATTACH_MAILBOX:
      filter->filter_type = FILTER_TYPE_USER;
      lview->parent = 
	get_all_lview(session.view->current->mailbox->mailstream);
      lview->level = lview->parent->level + 1;
      break;

  default:
    break;
  }


  lview->filter = filter;
  create_lview(lview,FALSE);

  stuff_lview_list(session.view,TRUE);

}
