/* compose.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1994 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

#include "imap/c-client/smtp.h"
#include "imap/c-client/nntp.h"

void compose_help();
void compose_rewrap();
void compose_add_prefix();
void compose_strip_prefix();
void compose_load_draft();
void compose_save_draft();
void compose_bye();
void toggle_news();
void send_message();
void create_compose_window();
void load_compose_window();
void compose_next();
void compose_prev();
void compose_destroy_current();
void free_compose_info();
void compose_watch_delivery();
void compose_context_switch();
Compose_Info *new_compose_info();
void compose_insert_file();
void compose_send_autograph();
void attach_file();
void add_attach_strings();
void destroy_part_list();
void delete_part();
void attach_multi();
void fix_empty_attachments();
void compose_keep_open();
void compose_window_list();
void compose_list_select();
void stuff_window_list();
void compose_list_cancel();
void compose_list_help();
char *address_popup();
char *lookup_address();
void complete_address();
void compose_pgp_sign();
void compose_pgp_cryp();
void compose_pgp_both();
void compose_print();
void compose_unwrap();
void compose_edit_dismiss();
void compose_edit_help();
void create_edit_menu();
void set_edit_buttons();
void compose_createattach();
void pgp_encode();
void get_multi_forward_info();
void compose_reply_insert();
void link_attachment();


char compose_address_translations[] = 
"      <Key>Tab:        complete_address()    \n";


Menu compose_file_menu[] = {

  { "Load Draft", "compose_load_draft", 'L', 
      compose_load_draft, NULL, 0, NULL, NULL, BTN_ON },

  { "Save Draft", "compose_save_draft", 'S', 
      compose_save_draft, NULL, 0, NULL, NULL, BTN_ON },

  { "Print", "compose_print", 'P', 
      compose_print, NULL, 0, NULL, NULL, BTN_ON },
  
  { "Insert File", "compose_insert_file", 'I',
      compose_insert_file, NULL, 0, NULL, NULL, BTN_ON },

  { "Address_Book", "compose_address_book", 'A',
      main_address_book, NULL, 0, NULL, NULL, BTN_ON },


  { "Close Window", "compose_bye", 'W', 
      compose_bye,   NULL, 0, NULL, NULL, BTN_ON },

};


Menu compose_attach_menu[] = {

  { "Attach File", "attach_file", 'A',
      attach_file, NULL, 0, NULL, NULL, BTN_ON },

  { "Attach Multipart", "attach_multi", 'M',
      attach_multi, NULL, 0, NULL, NULL, BTN_ON },

  { "Create Attachment", "compose_createattach", 'C',
      compose_createattach, NULL, 0, NULL, NULL, BTN_ON },

  { "Delete Attachment", "delete_part", 'D',
      delete_part, NULL, 0, NULL, NULL, BTN_NOATTACH|BTN_NOATTACHSELECT },

};


Menu compose_edit_menu[] = {

  { "Reply Insert", "compose_reply_insert", 'I',
      compose_reply_insert, NULL, 0, NULL, NULL, BTN_ON },

  { "Edit Tool", "create_edit_menu", 'T',
      create_edit_menu, NULL, 0, NULL, NULL, BTN_ON },

  { "Attachments", "compose_attach_menu", 'A',
      NULL, compose_attach_menu, XtNumber(compose_attach_menu),
      NULL, NULL, BTN_ON },
  
  { "Kill Message", "compose_destroy_current", 'K',
      compose_destroy_current, NULL, 0, NULL, NULL, BTN_ON },


};


Menu compose_key_submenu[] = {

  { "Sign", "compose_pgp_sign", 'S',
      compose_pgp_sign, NULL, 0, NULL, NULL, BTN_ON },

  { "Encrypt", "compose_pgp_encrypt", 'E',
      compose_pgp_cryp, NULL, 0, NULL, NULL, BTN_ON },

  { "Sign And Encrypt", "compose_pgp_both", 'A',
      compose_pgp_both, NULL, 0, NULL, NULL, BTN_ON },

};

Menu compose_options_menu[] = {

  { "Mail<->News", "toggle_news", 'M', 
      toggle_news, NULL, 0, NULL, NULL, BTN_ON },

  { "Watch Delivery", "compose_watch_delivery", 'W',
      compose_watch_delivery, NULL, 0, NULL, NULL, BTN_ON },


  { "Keep Open", "compose_keep_open", 'O',
      compose_keep_open, NULL, 0, NULL, NULL, BTN_ON },

#ifdef PGP

  { "Public Key", "compose_key_menu", 'K',
      NULL, compose_key_submenu, XtNumber(compose_key_submenu), 
      NULL, NULL, BTN_ON|BTN_NOPGP },

#endif /* PGP */

};


Menu compose_send_menu[] = {
  { "Send Message", "compose_send_message", 'S', 
      send_message, NULL, 0, NULL, NULL, BTN_ON },
  { "Send With Autograph", "compose_send_autograph", 'A',
      compose_send_autograph, NULL, 0, NULL, NULL, BTN_NOSIGFILE },

};

Menu compose_menu[] = {
  { "File", "compose_file_menu", 'F', 
      NULL, compose_file_menu, XtNumber(compose_file_menu), 
      NULL, NULL, BTN_ON },
  { "Edit", "compose_edit_menu", 'E', 
      NULL, compose_edit_menu, XtNumber(compose_edit_menu), 
      NULL, NULL, BTN_ON },
  { "Options", "compose_options_menu", 'O',
      NULL, compose_options_menu, XtNumber(compose_options_menu), 
      NULL, NULL, BTN_ON },
  { "Send", "compose_send_menu", 'S',
      NULL, compose_send_menu, XtNumber(compose_send_menu), 
      NULL, NULL, BTN_ON },

  { "<", "compose_next", '<',
      compose_next, NULL, 0, NULL, NULL, BTN_NONEXT },
  { "?", "compose_window_list", '?',
      compose_window_list, NULL, 0, NULL, NULL, BTN_ON },
  { ">", "compose_prev", '>',
      compose_prev, NULL, 0, NULL, NULL, BTN_NOPREV },


  { "Help", "compose_HELP!", 'H', 
      compose_help, NULL, 0,
      NULL, NULL, BTN_ON },
};



Menu compose_format_menu[] = {
  { "Dismiss", "compose_edit_dismiss", 'D',
      compose_edit_dismiss, NULL, 0, NULL, NULL, BTN_ON },
  { "Help",  "compose_edit_HELP!", 'H',
      compose_edit_help, NULL, 0, NULL, NULL, BTN_ON },

};



/*
 * Here's the logical layout...
 * We have a Compose structure which contains window information,
 * and it contains a linked list of Compose_Info structures,
 * which point to the data used by any particular compose session.
 * The window itself can be closed during the mail session, but
 * the messages-in-progress will survive when it is invoked
 * again. The "Window" menu bar contains the functions for
 * navigating between compose sessions. A successful send or
 * message kill removes that session and brings back the previous
 * session in the list, or the next if there's no previous, or
 * shuts the window if there's nothing left. The main Compose
 * structure is only created once during the program, although
 * the widgets it holds may need to be recreated. We keep a reference
 * to this structure in the top level Session structure.
 */


#ifdef __STDC__
Compose_Info *compose(char *title, int behaviour, 
		      Compose_Type type, Read_Info *read_info, 
		      Lview *lview, Message *message)
#else
Compose_Info *compose(title,behaviour,type,read_info,lview, message)
     char *title;
     int behaviour;
     Compose_Type type;
     Read_Info *read_info;
     Lview *lview;
     Message *message;

#endif
{
  Compose *compose = session.compose;
  Compose_Info *compose_info, *old;
  Boolean preserve = FALSE;

  /* 
   * If the first time through -- grab a Compose structure and 
   * initialize it.
   *
   */

  if(compose == NULL) {
    compose = (Compose *) fs_get(sizeof(Compose));
    session.compose = compose;
    compose->compose_info = NULL;
    compose->is_realized = FALSE;
    compose->current = NULL;
    compose->compose_info = NULL;
    compose->edit = new_edit();
  }

  /* 
   * Create a Compose_Info structure for this compose session.
   */
  
  compose_info = new_compose_info();

  compose_info->title = cpystr(title);

  /* 
   * Link in the Compose_Info structure to our other sessions. This will
   * become the "last" session, and anything else is pushed back, i.e. they
   * become "previous" sessions. Set the buttons of our next-door neighbor 
   * so that its button state reflects the fact that we are now in the chain.
   *
   */

  compose_info->prev = compose->compose_info;

  if(compose_info->prev == NULL)
    compose_info->button_state |= BTN_NOPREV;
  else
    compose_info->prev->button_state &= (~BTN_NONEXT);

  compose_info->button_state |= BTN_NONEXT;

  if(preferences.pgp_pathname && *preferences.pgp_pathname) {
    if((access(preferences.pgp_pathname,X_OK)) == SYSCALL_FAILURE)
      compose_info->button_state |= BTN_NOPGP;
  }
  else
    compose_info->button_state |= BTN_NOPGP;
    
    

  if(compose->compose_info != NULL) 
    compose->compose_info->next = compose_info;

  compose->compose_info = compose_info;

  /* 
   * Either the compose window has never been created,
   * or it was closed at some point, and we need to re-create
   * our widgets. 
   */  

  if(compose->is_realized == FALSE) {
    create_compose_window(compose,behaviour);
    preserve = TRUE; /* create_compose_window() will set is_realized. */
                     /* We still will need to know this state later.  */
  }
  else
    XRaiseWindow(display,XtWindow(compose->shell));

  XFlush(display);

  if((preferences.signature_file == NULL)
     || (*preferences.signature_file == NUL_TERM) 
     || ((access(preferences.signature_file,R_OK)) != SYSCALL_SUCCESS))
    compose_info->button_state |= BTN_NOSIGFILE;


  /* 
   * Stuff the present structure with anything applicable, such as
   * forwarding or remailed text.
   *
   */

  if(read_info != NULL || type == COMPOSE_NEW) 
    load_compose_info(compose_info,type,read_info);
  else
    load_compose_info_message(compose_info,type,lview, message);
  /* 
   * Now make this new session the active one. When it comes
   * back, we will be compose->current, the window will hold
   * our new editing sesion, and the buttons will have been checked.
   * The preserve check is because if the window has been regenerated,
   * it will now be empty. We don't want to store the current contents
   * into our old structures, because it will overwrite them.
   */

  compose_context_switch(compose,
			 (preserve == TRUE) ? NULL : compose_info->prev,
			 compose_info);

  return(compose_info);
}





load_compose_info(compose_info,type,read_info)
     Compose_Info *compose_info;
     Compose_Type type;
     Read_Info *read_info;
{

  char *to;                  /* don't free */
  char *reply_subject;       /* don't free */
  char *cc;                  /* don't free */
  char *msgid;
  char *ptr1;
  char *ptr2;

  char *fulltext;
  char *fulleverything;
  PART *part;
  BODY *body;
  char *tmp;
  char buffer[FILEBUFFLEN];


  if((type == COMPOSE_NEW) || (read_info == NULL))
    return;

  compose_info->compose_type = type;
  compose_info->mailstream   = read_info->mailstream;
  compose_info->msgno        = read_info->msgno;

  set_watch_cursors();

  switch(type) {

  case COMPOSE_REPLYALL:
    ptr1 = get_header_field_contents("to",read_info->header);
    if(ptr1 == NULL)
      ptr1 = cpystr(EMPTYSTR);
    ptr2 = get_header_field_contents("cc",read_info->header);
    if(ptr2 == NULL)
      ptr2 = cpystr(EMPTYSTR);
    cc = (char *) fs_get(strlen(ptr1) + strlen(ptr2) + 8 );
    *cc = NUL_TERM;
    
    if(*ptr1 != NUL_TERM) {
      strcpy(cc,ptr1);
      if(*ptr2 != NUL_TERM)
	strcat(cc,", ");
    }
    if(*ptr2 != NUL_TERM)
      strcat(cc,ptr2);
    fs_give((void **) &ptr1);
    fs_give((void **) &ptr2);
    fs_give((void **) &compose_info->cc);
    compose_info->cc = cc;
    
    /* fall through to set other info */
    
  case COMPOSE_REPLY:
    to = get_header_field_contents("reply-to",read_info->header);
    if(to == NULL)
      to = get_header_field_contents("from", read_info->header);
    if(to) {
      fs_give((void **) &compose_info->to);
      compose_info->to = to; 
    }
    reply_subject = get_reply_subject(read_info->header);
    fs_give((void **) &compose_info->subject);
    compose_info->subject = reply_subject;
    msgid = get_header_field_contents("message-id", read_info->header);
    compose_info->in_reply_to = msgid;
    if(read_info->alt_text)
      compose_info->reply_text = cpystr(read_info->alt_text);
    else
      if(read_info->current_text)
	compose_info->reply_text = cpystr(read_info->current_text);
    break;
  case COMPOSE_FORWARD:
    if(read_info->alt_text) {
      fs_give((void **) &compose_info->text);
      compose_info->text = get_forward_text(read_info->alt_text);
    }
    else {
      if(read_info->current_text) {
	fs_give((void **) &compose_info->text);
	compose_info->text = get_forward_text(read_info->current_text);
      }
    }
    fs_give((void **) &compose_info->subject);
    compose_info->subject = get_forward_subject(read_info->header);
    break;
  case COMPOSE_FORWARDATTACH:
    if((read_info->mailstream == NULL) 
       || (read_info->msgno == 0L)) {
      mm_log("Message not valid (expunged?). Cannot attach contents.",WARN);
      break;
    }
    fulltext = mail_fetchtext(read_info->mailstream,read_info->msgno);
    if(! fulltext)
      break;
    stripcr(fulltext);
    fulleverything = (char *) fs_get(strlen(read_info->header) +
				     strlen(fulltext) + 256);
    strcpy(fulleverything,read_info->header);
    strcat(fulleverything,LFSTR);
    strcat(fulleverything,fulltext);
    part = mail_newbody_part();
    body = &part->body;
    
    body->type = TYPEMESSAGE;
    body->subtype = cpystr("RFC822");
    body->description = get_forward_subject(read_info->header);
    body->encoding = ENC7BIT;
    body->contents.msg.text = fulleverything;
    compose_info->attachments = part;
    break;
    
  case COMPOSE_REMAIL:
    if(read_info->mailstream && read_info->msgno) {
      compose_info->remail_header 
	= cpystr(mail_fetchheader(read_info->mailstream,read_info->msgno));
      fs_give((void **) &compose_info->text);
      compose_info->text 
	= cpystr(mail_fetchtext(read_info->mailstream, read_info->msgno));
      stripcr(compose_info->text);
    }
    break;
    
  default:
    break;
  }

  if((preferences.default_cc != NULL)
     && (*preferences.default_cc != NUL_TERM)) {
    tmp = (char *) fs_get(strlen(compose_info->cc) 
			  + strlen(preferences.default_cc) + 16);
    if(strlen(compose_info->cc)) {
      strcpy(tmp,compose_info->cc);
      strcat(tmp,", ");
      strcat(tmp,preferences.default_cc);
    }
    else
      strcpy(tmp,preferences.default_cc);
    fs_give((void **) &compose_info->cc);
    compose_info->cc = tmp;

  }

  if((preferences.default_bcc != NULL) 
     && (*preferences.default_bcc != NUL_TERM)) {
    fs_give((void **) &compose_info->bcc);
    compose_info->bcc = cpystr(preferences.default_bcc);

  }

  reset_cursors();

}

load_compose_info_message(compose_info,type, lview, message)
     Compose_Info *compose_info;
     Compose_Type type;
     Lview *lview;
     Message *message;
{

  char *tmp;
  char *to;                  /* don't free */
  char *reply_subject;       /* don't free */
  char *cc;                  /* don't free */
  char *msgid;
  char *ptr1;
  char *ptr2;

  char *fulleverything;
  PART *oldpart;
  PART *part;
  BODY *body;

  char *fetched_header = NULL;
  char *fetched_text = NULL;

  Message_List *message_list;

  char buffer[FILEBUFFLEN];
  unsigned long num_selected = 0L;

  compose_info->compose_type = type;

  set_watch_cursors();

  if(message) {
    compose_info->mailstream   = message->mailstream;
    compose_info->msgno        = message->msgno;

    fetched_header = cpystr(mail_fetchheader(message->mailstream,
					     message->msgno));
    
    stripcr(fetched_header);
    
    if(fetched_header[strlen(fetched_header) - 1] == LFCHAR)
      fetched_header[strlen(fetched_header) - 1] = NUL_TERM;
    
    fetched_text = cpystr(mail_fetchtext(message->mailstream,
					 message->msgno));
    
    stripcr(fetched_text);
  }
  
  switch(type) {
    
  case COMPOSE_REPLYALL:
    ptr1 = get_header_field_contents("to",fetched_header);
    if(ptr1 == NULL)
      ptr1 = cpystr(EMPTYSTR);
    ptr2 = get_header_field_contents("cc",fetched_header);
    if(ptr2 == NULL)
      ptr2 = cpystr(EMPTYSTR);
    cc = (char *) fs_get(strlen(ptr1) + strlen(ptr2) + 8 );
    *cc = NUL_TERM;
    
    if(*ptr1 != NUL_TERM) {
      strcpy(cc,ptr1);
      if(*ptr2 != NUL_TERM)
	strcat(cc,", ");
    }
    if(*ptr2 != NUL_TERM)
      strcat(cc,ptr2);
    fs_give((void **) &ptr1);
    fs_give((void **) &ptr2);
    fs_give((void **) &compose_info->cc);
    compose_info->cc = cc;
    
    /* fall through to set other info */
    
  case COMPOSE_REPLY:
    to = get_header_field_contents("reply-to",fetched_header);
    if(to == NULL)
      to = get_header_field_contents("from", fetched_header);
    if(to) {
      fs_give((void **) &compose_info->to);
      compose_info->to = to; 
    }
    reply_subject = get_reply_subject(fetched_header);
    fs_give((void **) &compose_info->subject);
    compose_info->subject = reply_subject;
    msgid = get_header_field_contents("message-id", fetched_header);
    compose_info->in_reply_to = msgid;
    if(fetched_text)
      compose_info->reply_text = cpystr(fetched_text);
    break;
  case COMPOSE_FORWARD:
    compose_info->mailstream   = NULL;
    compose_info->msgno        = 0L;

    get_multi_forward_info(lview,compose_info);
    break;
    
  case COMPOSE_FORWARDATTACH:
    compose_info->mailstream   = NULL;
    compose_info->msgno        = 0L;

    for(message_list = lview->message_list;
	message_list; message_list = message_list->next) {
      if(message_list->selected == TRUE) {
	
	fetched_header 
	  = cpystr(mail_fetchheader(message_list->message->mailstream,
				    message_list->message->msgno));
	
	stripcr(fetched_header);
	
	if(fetched_header[strlen(fetched_header) - 1] == LFCHAR)
	  fetched_header[strlen(fetched_header) - 1] = NUL_TERM;
	
	fetched_text 
	  = cpystr(mail_fetchtext(message_list->message->mailstream,
				  message_list->message->msgno));
	
	stripcr(fetched_text);
	
	
	fulleverything = (char *) fs_get(strlen(fetched_header) +
					 strlen(fetched_text) + 256);
	strcpy(fulleverything,fetched_header);
	strcat(fulleverything,LFSTR);
	strcat(fulleverything,fetched_text);
	part = mail_newbody_part();
	body = &part->body;
	
	body->type = TYPEMESSAGE;
	body->subtype = cpystr("RFC822");
	body->description = get_forward_subject(fetched_header);
	body->encoding = ENC7BIT;
	body->contents.msg.text = fulleverything;
	if(compose_info->attachments == NULL)
	  compose_info->attachments = part;
	else {
	  for(oldpart = compose_info->attachments; oldpart->next;
	      oldpart = oldpart->next)
	    ;
	  oldpart->next = part;
	}
	fs_give((void **) &fetched_header);
	fs_give((void **) &fetched_text);
      }
    }
    break;
    
  case COMPOSE_REMAIL:
    compose_info->remail_header = cpystr(fetched_header);
    fs_give((void **) &compose_info->text);
    compose_info->text = cpystr(fetched_text);
    break;
    
  default:
    break;
  }

  if(fetched_header)  
    fs_give((void **) &fetched_header);
  if(fetched_text)
    fs_give((void **) &fetched_text);



  if((preferences.default_cc != NULL)
     && (*preferences.default_cc != NUL_TERM)) {
    tmp = (char *) fs_get(strlen(compose_info->cc) 
			  + strlen(preferences.default_cc) + 16);
    if(strlen(compose_info->cc)) {
      strcpy(tmp,compose_info->cc);
      strcat(tmp,", ");
      strcat(tmp,preferences.default_cc);
    }
    else
      strcpy(tmp,preferences.default_cc);
    fs_give((void **) &compose_info->cc);
    compose_info->cc = tmp;

  }

  if((preferences.default_bcc != NULL) 
     && (*preferences.default_bcc != NUL_TERM)) {
    fs_give((void **) &compose_info->bcc);
    compose_info->bcc = cpystr(preferences.default_bcc);

  }






  reset_cursors();

}


void get_multi_forward_info(lview,compose_info)
  Lview *lview;
  Compose_Info *compose_info;
{
  Message_List *message_list;
  unsigned long count = 0L;
  Message *message = NULL;
  char *old_header = NULL;
  char *old_text = NULL;
  char *new_header = NULL;
  char *new_text = NULL;
  char *tmp_header = NULL;
  char *tmp_text = NULL;
  char *fetched_header = NULL;
  char *fetched_text = NULL;
  char *ret = NULL;

  for(message_list = lview->message_list; message_list; 
      message_list = message_list->next) {
    if(message_list->selected) {
      message = message_list->message;
      count ++;
    }
  }

  if(count == 0)
    return;

  if((count == 1) && (message)) {
    fetched_header = cpystr(mail_fetchheader(message->mailstream,
					     message->msgno));
    stripcr(fetched_header);
    if(fetched_header[strlen(fetched_header) - 1] == LFCHAR)
      fetched_header[strlen(fetched_header) - 1] = NUL_TERM;
    
    fetched_text = cpystr(mail_fetchtext(message->mailstream,
					 message->msgno));
    stripcr(fetched_text);
    fs_give((void **) &compose_info->text);
    compose_info->text = get_forward_text(fetched_text);
    
    fs_give((void **) &compose_info->subject);
    compose_info->subject = get_forward_subject(fetched_header);
    return;
  }

  count = 0;

  fs_give((void **) &compose_info->subject);
  compose_info->subject = cpystr("[Multiple Messages Forwarded]");
  old_header = cpystr("Forwarded Messages:\n\n");
  old_text = cpystr(EMPTYSTR);

  for(message_list = lview->message_list; message_list; 
      message_list = message_list->next) {
    if(message_list->selected) {
      message = message_list->message;
      count ++;

      fetched_header = cpystr(mail_fetchheader(message->mailstream,
					       message->msgno));
      stripcr(fetched_header);
      if(fetched_header[strlen(fetched_header) - 1] == LFCHAR)
	fetched_header[strlen(fetched_header) - 1] = NUL_TERM;
      
      tmp_header = get_short_forward_line(fetched_header,count);
      new_header = (char *)fs_get(strlen(old_header) + strlen(tmp_header) + 1);
      strcpy(new_header,old_header);
      strcat(new_header,tmp_header);
      fs_give((void **) &old_header);
      fs_give((void **) &tmp_header);
      fs_give((void **) &fetched_header);
      old_header = new_header;

      fetched_text = cpystr(mail_fetchtext(message->mailstream,
					   message->msgno));
			    
      stripcr(fetched_text);
      tmp_text = get_forward_text(fetched_text);
      new_text = (char *) fs_get(strlen(old_text) + strlen(tmp_text) + 2);
      strcpy(new_text,old_text);
      strcat(new_text,LFSTR);
      strcat(new_text,tmp_text);
      fs_give((void **) &old_text);
      fs_give((void **) &tmp_text);
      fs_give((void **) &fetched_text);
      old_text = new_text;


    }
  }

  ret = (char *) fs_get(strlen(old_header) + strlen(old_text) + 1);
  strcpy(ret,old_header);
  strcat(ret,old_text);

  fs_give((void **) &old_header);
  fs_give((void **) &old_text);

  fs_give((void **) &compose_info->text);
  compose_info->text = ret;
  return;
}

/*
 * Create a Compose_Info structure, and fill it with default values. The 
 * defaults are good for an original (new) mail message. 
 */

#ifdef __STDC__
Compose_Info *new_compose_info(void)
#else
Compose_Info * new_compose_info()
#endif
{

  Compose_Info *compose_info = (Compose_Info *) fs_get(sizeof(Compose_Info));

  compose_info->mailstream      = NULL;
  compose_info->msgno           = 0;

  compose_info->title           = cpystr(EMPTYSTR);
  compose_info->from            = cpystr(EMPTYSTR);
  compose_info->to              = cpystr(EMPTYSTR);
  compose_info->newsgroups      = cpystr(EMPTYSTR);
  compose_info->subject         = cpystr(EMPTYSTR);
  compose_info->cc              = cpystr(EMPTYSTR);
  compose_info->bcc             = cpystr(EMPTYSTR);
  compose_info->text            = cpystr(EMPTYSTR);
  compose_info->remail_header   = NULL;
  compose_info->in_reply_to     = NULL;
  compose_info->reply_text      = NULL;
  compose_info->button_state    = BTN_ON | BTN_NOATTACH | BTN_NOATTACHSELECT;
  compose_info->behaviour       = MAIL_BEHAVIOUR;
  compose_info->watch_send      = preferences.smtp_debug;
  compose_info->keep_open       = FALSE;
  compose_info->compose_type    = COMPOSE_NEW;
  compose_info->attachments     = NULL;
  compose_info->part_list       = NULL;
  compose_info->selection       = 0;
  compose_info->next            = NULL;
  compose_info->prev            = NULL;
  
  return(compose_info);

}


/* 
 * Called on message kill or successful delivery to remove the current
 * session, and free up its data. We're done with it. 
 *
 */


#ifdef __STDC__
void compose_destroy_current(Widget w, Compose *compose, XtPointer xp)
#else
void compose_destroy_current(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{

  Compose_Info *doomed;   /* This guy is history. */

  Boolean has_prev = FALSE;
  Boolean has_next = FALSE;

  if((compose == NULL) || (compose->current == NULL))
    return;

  doomed = compose->current;

  /* 
   * Extract ourselves form the double linked list.
   * At the same time set our neighbors' button states
   * to reflect the change in neighbors.
   */

  if(doomed->prev) { 
    doomed->prev->next = doomed->next;
    if(doomed->prev->next) 
      doomed->prev->button_state &= (~BTN_NONEXT);
    else
      doomed->prev->button_state |= BTN_NONEXT;
    has_prev = TRUE;
  }

  if(doomed->next) {
    doomed->next->prev = doomed->prev;
    if(doomed->next->prev)
      doomed->next->button_state &= (~BTN_NOPREV);
    else
      doomed->next->button_state |= BTN_NOPREV;
    has_next = TRUE;
  }
  else
    compose->compose_info = doomed->prev;

  /* Bring up whichever neighbor we might have, older first.  */

  if(has_prev == TRUE)
    compose_context_switch(compose,doomed,doomed->prev);
  else {
    if(has_next == TRUE)
      compose_context_switch(compose,doomed,doomed->next);
    else {                      /* no neighbors... */
      compose->current = NULL;
      compose->compose_info = NULL;
    }
  }

  free_compose_info(doomed);

  /* No more messages in the queue? Close the window. Bye, bye... */

  if(compose->current == NULL) {
    compose_bye(w,compose,xp);
  }    

}


/*
 * This functions frees the memory of a once used Compose_Info structure.
 */

#ifdef __STDC__
void free_compose_info(Compose_Info *compose_info)
#else
void free_compose_info(compose_info)
     Compose_Info *compose_info;
#endif
{

  if(! compose_info)
    return;

  if(compose_info->title)
    fs_give((void **) &compose_info->title);
  if(compose_info->from)
    fs_give((void **) &compose_info->from);
  if(compose_info->to)
    fs_give((void **) &compose_info->to);
  if(compose_info->newsgroups)
    fs_give((void **) &compose_info->newsgroups);
  if(compose_info->subject)
    fs_give((void **) &compose_info->subject);
  if(compose_info->cc)
    fs_give((void **) &compose_info->cc);
  if(compose_info->bcc)
    fs_give((void **) &compose_info->bcc);
  if(compose_info->text)
    fs_give((void **) &compose_info->text);

  if(compose_info->remail_header)
    fs_give((void **) &compose_info->remail_header);
  if(compose_info->in_reply_to)
    fs_give((void **) &compose_info->in_reply_to);

  if(compose_info->reply_text)
    fs_give((void **) & compose_info->reply_text);

  if(compose_info->attachments)
    mail_free_body_part(&compose_info->attachments);
  
  if(compose_info->part_list) {
    destroy_part_list(compose_info->part_list);
    compose_info->part_list = NULL;
  }

  fs_give((void **) &compose_info);

  return;

}


/* 
 * This function toggles the window contents between two
 * editing sessions, saving the contents of the old session, (currently
 * visible), setting the contents appropriate to the new session,
 * (making it visible) and sets compose->current to the new one. It 
 * also sets the button sensitivity applicable to this sesion. 
 *
 */


#ifdef __STDC__
void compose_context_switch(Compose *compose, 
			    Compose_Info *old, Compose_Info *new)
#else
void compose_context_switch(compose, old, new ) 
     Compose *compose;
     Compose_Info *old;
     Compose_Info *new;
#endif
{
  Arg args[ARGLISTSIZE];
  int n = 0;

  if(compose == NULL)
    return;

  if(old) {

    fs_give((void **) &old->to);
    old->to = GetTextField(compose->to_text);
    fs_give((void **) &old->newsgroups);
    old->newsgroups = GetTextField(compose->ng_text);
    fs_give((void **) &old->subject);
    old->subject = GetTextField(compose->subject_text);
    fs_give((void **) &old->cc);
    old->cc = GetTextField(compose->cc_text);
    fs_give((void **) &old->bcc);
    old->bcc = GetTextField(compose->bcc_text);
    fs_give((void **) &old->text);
    old->text = XmTextGetString(compose->compose_text);

  }

  if(new) {

    XtSetArg(args[n],XmNtitle,new->title); n ++;
    XtSetValues(compose->shell, args, n); n = 0;

    XmTextSetString(compose->to_text,new->to);
    XmTextSetString(compose->ng_text,new->newsgroups);
    XmTextSetString(compose->subject_text,new->subject);
    XmTextSetString(compose->cc_text,new->cc);
    XmTextSetString(compose->bcc_text,new->bcc);
    XmTextSetString(compose->compose_text,new->text);

    compose->current = new;
    /* make_attachment_list() will also set our buttons. */
    make_attachment_list(compose);

  }

  return;

}

#ifdef __STDC__
void compose_help(Widget w, Compose *compose, XtPointer xp)
#else 
void compose_help(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{
  help(compose->shell, "Compose Help", "compose.help");
}


#ifdef __STDC__
void compose_keep_open(Widget w, Compose *compose, XtPointer xp)
#else 
void compose_keep_open(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{
  if((compose) && (compose->current)) {
    compose->current->keep_open = TRUE;
  }

}


/* Go to the next sesion on the list. */


#ifdef __STDC__
void compose_next(Widget w, Compose *compose, XtPointer xp)
#else 
void compose_next(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{

  if(compose) {
    if(compose->current->next)
      compose_context_switch(compose,compose->current,compose->current->next);
    
    check_buttons(compose_menu,XtNumber(compose_menu),
		  compose->current->button_state); 
  }
  return;
}


/* Go to the previous session on the list. */

#ifdef __STDC__
void compose_prev(Widget w, Compose *compose, XtPointer xp)
#else
void compose_prev(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{

  if(compose) {
    if(compose->current->prev)
      compose_context_switch(compose,compose->current,compose->current->prev);
    
    check_buttons(compose_menu,XtNumber(compose_menu),
		  compose->current->button_state);
  } 
  return;
}


/* 
 * Called when there are either no more compose sessions, or
 * the user closes the window. All we do is remove the widgets,
 * so if there are any compose sessions, they'll still be there.
 */


#ifdef __STDC__
void compose_bye(Widget w, Compose *compose, XtPointer xp)
#else
void compose_bye(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{

  if(compose) {

    /* save the current window contents */
    compose_context_switch(compose,compose->current, NULL);

    XtDestroyWidget(compose->shell);
    compose->is_realized = FALSE;
  }
  return;
}

/*
 * Called when something external to the window kills it. We should
 * save the current edit, and make sure the is_realized flag gets
 * set to false or we'll never be able to open it again.
 */

#ifdef __STDC__
void compose_destroy(Widget w, Compose *compose, XtPointer xp)
#else
void compose_destroy(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{

  if(compose) {

    /* save the current window contents */
    compose_context_switch(compose,compose->current, NULL);
    compose->is_realized = FALSE;
  }
  return;
}


/* 
 * Turn on the mail debugging flag. We can also do it globally
 * from preferences.smtp_debug. This sets it for the current
 * message only. There is no way to turn it off again.
 */

#ifdef __STDC__
void compose_watch_delivery(Widget w, Compose *compose, XtPointer xp)
#else
void compose_watch_delivery(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{

  if(compose && compose->current)
    compose->current->watch_send = TRUE;

  return;
}


/* 
 * Change the window to reflect a NEWS message, with a "Newsgroups:"
 * line instead of "To:". "Cc" and "Bcc" are on all messages,
 * but only go through SMTP. Newsgroups go through NNTP. You can 
 * mix and match either. Note that the window which is hidden as 
 * a result of this is still active. If it has contents,
 * they will be used at delivery time.
 *
 */


#ifdef __STDC__
void toggle_news(Widget w, Compose *compose, XtPointer xp)
#else
void toggle_news(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
#endif
{
  if((compose != NULL) && (compose->current != NULL)) {

    if(compose->current->behaviour == NEWS_BEHAVIOUR) {
      compose->current->behaviour = MAIL_BEHAVIOUR;
      XtUnmanageChild(compose->ng_label);
      XtUnmanageChild(compose->ng_text);
      XtManageChild(compose->to_label);
      XtManageChild(compose->to_text);
      
    }
    else { /* (compose->current->behaviour == MAIL_BEHAVIOUR) */
      compose->current->behaviour = NEWS_BEHAVIOUR;
      XtUnmanageChild(compose->to_label);
      XtUnmanageChild(compose->to_text);
      XtManageChild(compose->ng_label);
      XtManageChild(compose->ng_text);
      
    }
  }
  return;
}




/*
 * This is a fat function, and does almost all the dirty work. When it 
 * comes back, the message should be on its way.
 *
 * It takes a compose structure which
 * points to an open compose window, extracts the stuff from
 * the windows (which are presumably more current than the Compose_Info
 * structures), connects any attachments, and sends it off. It also 
 * determines whether it is appropriate to send news, mail or both.
 * On success, the current session is killed; the compose window is 
 * restored to any previous (or next) composings, or otherwise closes 
 * the window. On failure the message is left intact so the user can 
 * take some form of action to correct the condition, or perhaps save 
 * the work, or whatever.
 */

void send_message(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  SMTPSTREAM *smtp_stream = NIL;
  SMTPSTREAM *nntp_stream = NIL;

  ENVELOPE *envelope = NULL;
  ENVELOPE *news_envelope = NULL;
  BODY *body = NULL;
  PART *part = NULL;
  PART *curr = NULL;
  
  char **mailhosts = NULL;
  char **newshosts = NULL;

  char *default_host = cpystr(preferences.imap_server);

  Boolean sendnews    = FALSE;
  Boolean sendmail    = TRUE;
  Boolean retry       = compose->current->keep_open;
  Boolean attach_text = FALSE;
  char *from          = NULL;
  char *to            = NULL;
  char *newsgroups    = NULL;
  char *subject       = NULL;
  char *cc            = NULL;
  char *bcc           = NULL;
  char *temp_text     = NULL;
  char *text          = NULL;

  char msgno_string[32];

  int errors          = 0;


  set_watch_cursors();

  if(! default_host)
    default_host = cpystr(EMPTYSTR);

  mailhosts = (char **) fs_get( 2 * sizeof(char *));
  mailhosts[0] = cpystr(preferences.smtp_server);
  mailhosts[1] = NULL;

  newshosts = (char **) fs_get( 2 * sizeof(char *));
  newshosts[0] = cpystr(preferences.nntp_server);
  newshosts[1] = NULL;

  from       = cpystr(preferences.reply_address);
  newsgroups = GetTextField(compose->ng_text);
  to         = GetTextField(compose->to_text);
  subject    = GetTextField(compose->subject_text);
  cc         = GetTextField(compose->cc_text);
  bcc        = GetTextField(compose->bcc_text);
  text       = XmTextGetString(compose->compose_text);


  /* 
   * Do two conversions. First translate paragraphs to LF terminated
   * text. Then turn LF to CRLF for sending. The reason for doing this
   * in two steps is so we can allocate approximately the right amount 
   * of space in each function, and not have to overkill the memory 
   * requirements. It also compartmentalizes the functionality.
   *
   */

  temp_text = wrap_text(text, COMPOSEWIDTH - 1);
  fs_give((void **) &text);
  text = lftocrlf(temp_text);
  fs_give((void **) &temp_text);

  if((newsgroups != NULL) && (*(first_nonwhite(newsgroups)) != NUL_TERM))
    sendnews = TRUE;

  if(   (*(first_nonwhite(to))  == NUL_TERM) 
     && (*(first_nonwhite(cc))  == NUL_TERM) 
     && (*(first_nonwhite(bcc)) == NUL_TERM))
    sendmail = FALSE;

  /* No recipients. Make sure it fails. */
  if((sendnews == FALSE) && (sendmail == FALSE))
    retry = TRUE;

  if(sendnews == TRUE) {
    nntp_stream = nntp_open(newshosts, 0L );
    if((nntp_stream != NIL) && (compose->current->watch_send == TRUE))
      smtp_debug(nntp_stream);
    if(nntp_stream == NIL) {
      mm_log(NNTP_OPEN_ERR, WARN);
      retry = TRUE;
    }
  }

  if(sendmail == TRUE) {
    smtp_stream = smtp_open(mailhosts, 0L );
    if((smtp_stream != NIL) && (compose->current->watch_send == TRUE))
      smtp_debug(smtp_stream);
    if(smtp_stream == NIL) {
      mm_log(SMTP_OPEN_ERR, WARN);
      retry = TRUE;
    }
  }

  body = mail_newbody();
  envelope = mail_newenvelope();

  fix_empty_attachments(compose);


  if(compose->current->attachments || body->contents.part) {
      body->contents.part = compose->current->attachments;
      body->type = TYPEMULTIPART;
      body->encoding = ENC7BIT;

    /* 
     * Our text portion (if there is one)
     * will become the first part of the multipart 
     */
    
    if(*text != NUL_TERM) {
      attach_text = TRUE;
      part = mail_newbody_part();
      part->body.type = TYPETEXT;
      part->body.encoding = ENC7BIT;
      part->body.contents.text = (unsigned char *) text;
      part->next = body->contents.part;
      body->contents.part = part;
    }
  }
  else {                   /* Not a multipart. Just the text message. */
    
    body->contents.text = (unsigned char *) text;
    body->type = TYPETEXT;
    body->encoding = ENC7BIT;
    
  }

  /*
   * Warning: The c-client appears to trash the calling text strings 
   * in rfc822_parse_addr(), which text_to_address() calls... So, don't 
   * parse the same  string twice to get a duplicate. Use copy_address() 
   * instead.
   *
   */

  envelope->from        = text_to_address(from,default_host);
  envelope->to          = text_to_address(to,  default_host);
  envelope->cc          = text_to_address(cc,  default_host);
  envelope->bcc         = text_to_address(bcc, default_host);

  envelope->reply_to    = copy_address(envelope->from); 

  envelope->return_path = mail_newaddr();
  envelope->return_path->mailbox = cpystr(envelope->from->mailbox);
  envelope->return_path->host    = cpystr(envelope->from->host);

  /* 
   * The sender field is our real bonafide Unix name.
   * This is done to thwart forgeries. This address is also used
   * for delivery bounces. Making certain that it can be used for
   * this purpose is a local system administration issue. That is,
   * if incoming SMTP is not allowed on the system, it should be 
   * MX forwarded to a mail hub which can process the errors.
   *
   */

  if(compose->current->compose_type == COMPOSE_REMAIL) {
    envelope->remail = cpystr(compose->current->remail_header);
    body->type = TYPETEXT;
    if(body->subtype)
      fs_give((void **) &body->subtype);
    body->subtype = NULL;
    body->encoding = ENC7BIT;
  }
  else {
    envelope->sender = mail_newaddr();
    envelope->sender->mailbox = cpystr(local_auth.username);
    envelope->sender->host    = cpystr(local_auth.hostname);
  }

  envelope->subject     = cpystr(subject);

  if(compose->current->in_reply_to)
    envelope->in_reply_to = cpystr(compose->current->in_reply_to);

  /* 
   * Generate a message and date. Note that we also supply the
   * real Unix credentials here as the second stage of
   * forgery suppression.
   */

  envelope->message_id = (char *) fs_get(MESSAGE_ID_LENGTH);
  sprintf(envelope->message_id,"<%s-%s.%d.%d.%s@%s>",
	  PROGRAM, VERSION, time(NULL), rand() % 10000, 
	  local_auth.username, local_auth.hostname);

  envelope->date = (char *) fs_get(MESSAGE_ID_LENGTH);
  rfc822_date(envelope->date);


  if((sendnews == TRUE) && (nntp_stream != NIL)) {

    /* 
     * We create a separate envelope for news posts, containing
     * newsgroups line, and stripping out to,cc,bcc.
     */

    news_envelope = mail_newenvelope();
    news_envelope->from        = copy_address(envelope->from);
    news_envelope->reply_to    = copy_address(envelope->reply_to);
    news_envelope->return_path = copy_address(envelope->return_path);
    news_envelope->sender      = copy_address(envelope->sender);
    news_envelope->subject     = cpystr(envelope->subject);
    news_envelope->message_id  = cpystr(envelope->message_id);
    news_envelope->date        = cpystr(envelope->date);
    news_envelope->newsgroups  = cpystr(newsgroups);
    errors = nntp_mail(nntp_stream, news_envelope, body);
    if(errors != T ) {
      mm_log(NNTP_SEND_ERR,WARN);
      retry = TRUE;
    }
    else
      mm_log(NEWSSENT,NIL);
    smtp_close(nntp_stream);
    mail_free_envelope(&news_envelope);
  }

  if((sendmail == TRUE) && (smtp_stream != NIL)) {
    errors = smtp_mail(smtp_stream,"MAIL", envelope, body);
    if(errors != T ) {
      mm_log(SMTP_SEND_ERR, WARN);
      retry = TRUE;
    }
    else
      mm_log(MAILSENT,NIL);
    smtp_close(smtp_stream);
    mail_free_envelope(&envelope);
  }

  /* 
   * If the send failed, protect all our attachments. mail_free_body()
   * is recursive. If the first part is our text, we'll let it go,
   * since it will be rebuilt next time through. This also means we
   * don't explicity free it. It will be nuked as a result of
   * mail_free_body().
   */

  if(retry == TRUE) {
    if(attach_text == TRUE) 
      body->contents.part->next = NIL;
    else 
      body->contents.part = NIL;
  }
  else {
    body->contents.part = compose->current->attachments;

    /* Set answered flag on original message if appropriate */

    if(((compose->current->compose_type == COMPOSE_REPLY) ||
       (compose->current->compose_type == COMPOSE_REPLYALL))
       && compose->current->mailstream && compose->current->msgno) {
      sprintf(msgno_string,"%d",compose->current->msgno);
      set_watch_cursors();
      mail_setflag(compose->current->mailstream,
		   msgno_string, ANSWERED_FLAG );
      reset_cursors();

    }

  }

  /*
   * Clean up everything and go home.
   */

  mail_free_body(&body);    /* and therefore our allocated text */

  fs_give((void **) &from);
  fs_give((void **) &to);
  fs_give((void **) &newsgroups);
  fs_give((void **) &subject);
  fs_give((void **) &cc);
  fs_give((void **) &bcc);

  fs_give((void **) &mailhosts[0]);
  fs_give((void **) &mailhosts);
  fs_give((void **) &newshosts[0]);
  fs_give((void **) &newshosts);

  fs_give((void **) &default_host);

  if(retry == FALSE) {
    /*
     * Our attachments are already gone. Make sure we don't
     * free them a second time.
     */
    compose->current->attachments = NIL;
    compose_destroy_current(w,compose,xp);
  }

  reset_cursors();

  return;
}


void fix_empty_attachments(compose)
     Compose *compose;
{
  Part_List *part_list;
  BODY *body;
  PART *part;
  BODY *newbody;
  Boolean changed = FALSE;
  
  for(part_list = compose->current->part_list;
      part_list; part_list = part_list->next) {
    body = &(part_list->part->body);

    if((body->type == TYPEMULTIPART) && (body->contents.part == NULL)) {
      /* 
       * Somebody tried to send an empty part. It would be rather
       * painful to try and prune it, so we'll gratuitously supply
       * a part for them. Otherwise, the c-client will core dump.
       *
       */
      part = mail_newbody_part();
      body->contents.part = part;
      newbody = &part->body;
      newbody->type = TYPETEXT;
      newbody->encoding = ENC7BIT;
      newbody->description = cpystr(FAKEPART);
      newbody->subtype = cpystr("plain");
      newbody->contents.text = (unsigned char *) cpystr(EMPTYSTR);
      newbody->size.bytes = 0;
      changed = TRUE;
    }
  }
  if(changed == TRUE) {
    mm_log(EMPTYMULTIWARN,WARN);
    make_attachment_list(compose);
  }

}



void delete_part(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  Part_List *part_list;
  PART *currpart = NULL;
  PART *nextpart = NULL;
  PART *prevpart = NULL;
  BODY *body = NULL;

  if((compose->current->selection == 0) 
     || (compose->current->part_list == NULL))
    return;
  
  /* point to the Part_List of the selected part.*/

  for(part_list = compose->current->part_list; 
      part_list->partnumber != compose->current->selection; 
      part_list = part_list->next);
    

  /* First see if it's a multipart which still has contents */

  body = &(part_list->part->body);
  if(body->type == TYPEMULTIPART 
     && (body->contents.part != NULL)) {
    mm_log(MULTI_NOT_EMPTY, WARN);
    return;
  }


  /* first part in our attachment list */

  if(part_list->part == compose->current->attachments) {
    compose->current->attachments = part_list->part->next;
    part_list->part->next = NULL;
    mail_free_body_part(&part_list->part);
    make_attachment_list(compose);
    return;
  }

  /* Some other part. See if our previous neighbor points to us. */

  if((part_list->prev) && (part_list->prev->part->next == part_list->part)) {
    part_list->prev->part->next = part_list->part->next;
    part_list->part->next = NULL;
  }
  else {
    /* Hmmm. Heirarchical. Find the parent. */
    body = (part_list->prev) ? &(part_list->prev->part->body) : NULL;
    if((body) && (body->type == TYPEMULTIPART))
      body->contents.part = part_list->part->next;
    part_list->part->next = NULL;
  }

  mail_free_body_part(&part_list->part);
  make_attachment_list(compose);
  return;


}


/* 
 * Callback when an item in the attachment list is selected. 
 */



void select_attach(w,compose,xp)
     Widget w;
     Compose *compose;
     XmListCallbackStruct *xp;
{
    
   if(xp->item_position == compose->current->selection) {
    compose->current->selection = 0;
    compose->current->button_state |= BTN_NOATTACHSELECT;
  }
  else {
    compose->current->selection = xp->item_position;
    compose->current->button_state &= ~(BTN_NOATTACHSELECT);
  }

  check_buttons(compose_menu,XtNumber(compose_menu),
		compose->current->button_state);

}


void double_click_attach(w,compose,xp)
     Widget w;
     Compose *compose;
     XmListCallbackStruct *xp;
{
  Part_List *part_list;
  BODY *body;

  if(! xp->item_position)
    return;

  for(part_list = compose->current->part_list; 
      part_list && part_list->partnumber != xp->item_position; 
      part_list = part_list->next);

  if(! part_list)
    return;

  /* Make sure the window shows it as selected */

  XmListDeselectAllItems(compose->compose_attach);
  compose->current->selection = xp->item_position;
  compose->current->button_state &= ~(BTN_NOATTACHSELECT);
  XmListSelectPos(compose->compose_attach,xp->item_position,FALSE);


  check_buttons(compose_menu,XtNumber(compose_menu),
		compose->current->button_state);

  body = &(part_list->part->body);

  /* invoke message handler */

  show_mime(compose->shell, body);

}


void make_attachment_list(compose)
     Compose *compose;
{

  int nitems = 0;
  int i;

  /* Erase current list contents and start over */

  XmListDeselectAllItems(compose->compose_attach);
  XmListDeleteAllItems(compose->compose_attach); 

  compose->current->selection = 0;
  compose->current->button_state |= BTN_NOATTACHSELECT;

  /* No contents. Check buttons and get outa' here */

  if(compose->current->attachments == NULL) { 
    compose->current->button_state |= BTN_NOATTACH;
    check_buttons(compose_menu,XtNumber(compose_menu),
		  compose->current->button_state);
    return;
  }

  
  if(compose->current->part_list) {
    destroy_part_list(compose->current->part_list);
  }

  compose->current->part_list = NULL;

  /* 
   * Recursively add strings to the attachment list widget 
   * and add to our part_list structure. We use this to look up
   * specific parts from the list management functions without having
   * to go through all this recursive hullabaloo all over again.
   */

  add_attach_strings(compose->compose_attach, compose,
		     compose->current->attachments,0);

  compose->current->button_state &= ~(BTN_NOATTACH);
  check_buttons(compose_menu,XtNumber(compose_menu),
		compose->current->button_state);
  return;

}

void destroy_part_list(part_list)
  Part_List *part_list;
{
  if(part_list == NULL)
    return;
  destroy_part_list(part_list->next);
  if(part_list->partstr)
    fs_give((void **) &part_list->partstr);
  fs_give((void **) &part_list);
  part_list = NULL;
}



void add_attach_strings(w,compose,part,level)
     Widget w;
     Compose *compose;
     PART *part;
     int level;
{
  int len;
  PART *curr;
  char buffer[FILEBUFFLEN];
  XmString xstr;
  Part_List *part_list;
  Part_List *curr_part_list;

  if(part == NULL)
    return;

  for(curr = part; curr; curr = curr->next) {
    strcpy(buffer,EMPTYSTR);
    for(len = 0; len < level; len ++)
      strcat(buffer,SPACESTR);
    strcat(buffer,type_to_name(curr->body.type));
    strcat(buffer,TYPE_SEPARATOR_STR);
    strcat(buffer,(curr->body.subtype) ? curr->body.subtype : NOSUBTYPE );
    for(len = strlen(buffer); len < (COMPOSEWIDTH/2); len ++)
      strcat(buffer,SPACESTR);
    strncat(buffer,
	   (curr->body.description) ? curr->body.description : EMPTYSTR,
	    COMPOSEWIDTH);
    buffer[COMPOSEWIDTH - 2] = NUL_TERM;
    xstr = XmStringCreateSimple(buffer);
    XmListAddItemUnselected(w,xstr,0);
    XmStringFree(xstr);
    
    part_list = (Part_List *) fs_get(sizeof(Part_List));
    part_list->next = NULL;
    part_list->part = curr;
    part_list->partstr = NULL;

    if(compose->current->part_list == NULL) {
      compose->current->part_list = part_list;
      part_list->partnumber = 1;
      part_list->prev = NULL;
    }
    else {
      for(curr_part_list = compose->current->part_list; curr_part_list->next;
	curr_part_list = curr_part_list->next);

      curr_part_list->next = part_list;
      part_list->prev = curr_part_list;
      part_list->partnumber = curr_part_list->partnumber + 1;
    }

    if(curr->body.type == TYPEMULTIPART)
      add_attach_strings(w,compose,curr->body.contents.part,level+1);

  }

}

/*
 * Generate and display the compose window.
 */


#ifdef __STDC__
void create_compose_window(Compose *compose,int behaviour)
#else
void create_compose_window(compose,behaviour)
     Compose *compose;
     int behaviour;
#endif
{

  Arg args[ARGLISTSIZE];
  int n = 0;

  /* These widgets we don't need to keep. */

  Widget form, menubar, label1, label2, label3, label4;

  XtTranslations translations;


  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  compose->shell = XtCreateWidget("compose_shell",
					 topLevelShellWidgetClass,
					 top, args, n ); n = 0;

  form = XmCreateForm(compose->shell, "form", args, n); n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form,"menubar", args, n); n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar,compose_menu, XtNumber(compose_menu), 
		      0, (XtPointer) compose);


  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, menubar); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNborderWidth, 0); n ++;
  compose->to_label = XmCreateLabel(form,"To        :", args, n); n = 0;


  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, menubar); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftWidget,compose->to_label); n ++;
  compose->to_text = XmCreateTextField(form,"to", args, n); n = 0;

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, menubar); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNborderWidth, 0); n ++;
  compose->ng_label = XmCreateLabel(form,"Newsgroups:", args, n); n = 0;

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, menubar); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftWidget,compose->ng_label); n ++;
  compose->ng_text = XmCreateTextField(form,"ng", args, n); n = 0;

  translations = XtParseTranslationTable(GLOBAL_text_field_translations);
  XtOverrideTranslations(compose->to_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->to_text,translations);
  translations =
    XtParseTranslationTable(compose_address_translations);
  XtOverrideTranslations(compose->to_text,translations);


  XtAddCallback(compose->to_text, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_field_edit, NULL);

  translations = XtParseTranslationTable(GLOBAL_text_field_translations);
  XtOverrideTranslations(compose->ng_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->ng_text,translations);
  translations =
    XtParseTranslationTable(compose_address_translations);
  XtOverrideTranslations(compose->ng_text,translations);


  XtAddCallback(compose->ng_text, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_field_edit, NULL);


  switch(behaviour) {
  case NEWS_BEHAVIOUR:
    XtManageChild(compose->ng_label);
    XtManageChild(compose->ng_text);
    break;
  case MAIL_BEHAVIOUR:
  default:
    XtManageChild(compose->to_label);
    XtManageChild(compose->to_text);
    break;
  }

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, compose->to_text); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNborderWidth, 0); n ++;

  label2 = XmCreateLabel(form,"Subject   :", args, n); n = 0;
  XtManageChild(label2);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, compose->to_text); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftWidget,label2); n ++;
  compose->subject_text = XmCreateTextField(form,"subject", args, n); n = 0;
  XtManageChild(compose->subject_text);

  translations = XtParseTranslationTable(GLOBAL_text_field_translations);
  XtOverrideTranslations(compose->subject_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->subject_text,translations);

  XtAddCallback(compose->subject_text, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_field_edit, NULL);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, compose->subject_text); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;

  XtSetArg(args[n], XmNborderWidth, 0); n ++;

  label3 = XmCreateLabel(form,"Cc        :", args, n); n = 0;
  XtManageChild(label3);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, compose->subject_text); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftWidget,label3); n ++;
  compose->cc_text = XmCreateTextField(form,"cc", args, n); n = 0;
  XtManageChild(compose->cc_text);

  translations = XtParseTranslationTable(GLOBAL_text_field_translations);
  XtOverrideTranslations(compose->cc_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->cc_text,translations);
  translations =
    XtParseTranslationTable(compose_address_translations);
  XtOverrideTranslations(compose->cc_text,translations);

  XtAddCallback(compose->cc_text, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_field_edit, NULL);


  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, compose->cc_text); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNborderWidth, 0); n ++;

  label4 = XmCreateLabel(form,"Bcc       :", args, n); n = 0;
  XtManageChild(label4);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, compose->cc_text); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftWidget,label4); n ++;
  compose->bcc_text = XmCreateTextField(form,"bcc", args, n); n = 0;
  XtManageChild(compose->bcc_text);

  translations = XtParseTranslationTable(GLOBAL_text_field_translations);
  XtOverrideTranslations(compose->bcc_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->bcc_text,translations);
  translations =
    XtParseTranslationTable(compose_address_translations);
  XtOverrideTranslations(compose->bcc_text,translations);

  XtAddCallback(compose->bcc_text, XmNmodifyVerifyCallback, 
		(XtCallbackProc) text_field_edit, NULL);

  XtSetArg(args[n], XmNnavigationType,XmNONE);                  n ++;
  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC);       n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmVARIABLE);              n ++;
  XtSetArg(args[n], XmNvisibleItemCount, VISIBLE_ATTACHMENTS ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmSINGLE_SELECT);        n ++;
  compose->compose_attach = 
    XmCreateScrolledList(form,"partlist",args,n);               n = 0;
  XtAddCallback(compose->compose_attach,
		XmNsingleSelectionCallback, select_attach, compose);
  XtAddCallback(compose->compose_attach,
		XmNdefaultActionCallback, double_click_attach, compose);


  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET);         n ++;
  XtSetArg(args[n], XmNtopWidget, compose->bcc_text);           n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);          n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);         n ++;

  XtSetValues(XtParent(compose->compose_attach), args, n); n = 0;
  XtManageChild(compose->compose_attach); 

  XtSetArg(args[n], XmNeditable, TRUE);                         n ++;
  XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT);            n ++;
  XtSetArg(args[n], XmNwordWrap, TRUE);                         n ++;
  XtSetArg(args[n], XmNrows,     preferences.compose_height);   n ++;
  XtSetArg(args[n], XmNcolumns,  COMPOSEWIDTH - 1);             n ++;
  XtSetArg(args[n], XmNscrollVertical, TRUE);                   n ++; 
  XtSetArg(args[n], XmNscrollHorizontal, FALSE );               n ++;

  compose->compose_text =
    XmCreateScrolledText(form, "compose_text", args, n);        n = 0;
  
  translations = XtParseTranslationTable(GLOBAL_text_translations);
  XtOverrideTranslations(compose->compose_text, translations);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET);         n ++;
  XtSetArg(args[n], XmNtopWidget, compose->compose_attach);     n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);        n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);          n ++;

  XtSetValues(XtParent(compose->compose_text), args, n);        n = 0;
  XtManageChild(compose->compose_text);

  XtAddCallback(compose->shell, XmNdestroyCallback, 
		(XtCallbackProc) compose_destroy, compose);

  XtManageChild(form);
  XtManageChild(compose->shell);
  XtRealizeWidget(compose->shell);
  compose->is_realized = TRUE;
  return;

}


void compose_insert_file(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  FILE *fp;
  char *filename;
  Binary_Buffer *binary_buffer;
  char buffer[FILEBUFFLEN];

  filename = file_select(compose->shell, NULL, NULL, NULL, FALSE, NULL);
  if(filename == NULL)
     return;
  
  set_watch_cursors();

  binary_buffer = load_binary_file(filename);
  if(binary_buffer) {
    text_blast(compose->compose_text,binary_buffer->data);
    free_binary_buffer(binary_buffer);
  }

  fs_give((void **) &filename);
  reset_cursors();
  return;
}


/* 
 * Append signature file into current text window and call 
 * send_message(). We throw in an extra linefeed between the two
 * just in case the last line of the message doesn't have one.
 */

void compose_send_autograph(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  FILE *fp;
  char *filename;
  char buffer[FILEBUFFLEN];

  filename = cpystr(preferences.signature_file);
  
  if((*filename != NUL_TERM) && (fp = fopen(filename,"r")) != NULL) {
    AppendText(compose->compose_text,LFSTR);
    while((fgets(buffer,sizeof(buffer),fp)) != NULL)
      AppendText(compose->compose_text,buffer);
    fclose(fp);
  }
  fs_give((void **) &filename);
  send_message(w,compose,xp);
  return;
}




void attach_file(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  char *filename = NULL;
  Binary_Buffer *binary_buffer;
  PART *part;
  BODY *body;
  Boolean auto_attach;
  Boolean keep_going;
  Boolean attached = FALSE;


  filename = file_select(compose->shell, NULL, NULL, NULL, FALSE, NULL);
  if(filename == NULL)
     return;

  binary_buffer = load_binary_file(filename);

  if(binary_buffer) {

    part = mail_newbody_part();
    body = &part->body;

    set_default_attach_type(body,filename);

    if((auto_attach = get_type_from_suffix(body,filename)) == FALSE)
      keep_going = create_mime_attach_window(compose->shell,body,FALSE);
  
    if(auto_attach || keep_going)
      link_attachment(compose,part,body,binary_buffer); 
    else 
      mail_free_body_part(&part);

    free_binary_buffer(binary_buffer);

  }

  fs_give((void **) &filename);
  
}


void attach_multi(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  PART *part;
  BODY *body;


  part = mail_newbody_part();
  body = &part->body;
  
  body->type = TYPEMULTIPART;
  body->subtype = cpystr("mixed");

  if(create_mime_attach_window(compose->shell,body,TRUE))
    link_attachment(compose,part,body,NULL);
  else
    mail_free_body_part(&part);
  
}


void compose_createattach(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  PART *part;
  BODY *body;
  Binary_Buffer *binary_buffer; 

  part = mail_newbody_part();
  body = &part->body;
  
  binary_buffer = create_mime_compose_window(compose->shell,body);

  if(binary_buffer) {
    link_attachment(compose,part,body,binary_buffer);
    free_binary_buffer(binary_buffer);
  }
  else
    mail_free_body_part(&part);

}


Menu compose_list_menu[] = {

  { "Cancel", "compose_list_cancel", 'C',
      compose_list_cancel, NULL, 0, NULL, NULL, BTN_ON },
  { "Help", "compose_list_HELP!", 'H',
      compose_list_help, NULL, 0, NULL, NULL, BTN_ON },
};



void compose_window_list(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Position x, y;
  XtTranslations translations;
  Widget form, menubar, dummy;

  compose->list_done = 0;
  set_pirate_cursors();

  get_pointer_position(w, &x, &y);

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  compose->list_shell = XtCreatePopupShell("Compose Sessions",
					 transientShellWidgetClass,
					 compose->shell, args, n ); n = 0;

  form = XmCreateForm(compose->list_shell, "form", args, n); n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form,"menubar", args, n); n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar,compose_list_menu, 
		      XtNumber(compose_list_menu), 0, (XtPointer) compose);

  /* An invisible text field. This is used only to set the width. */

  XtSetArg(args[n], XmNcolumns, COMPOSEWIDTH - 20); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  dummy = XmCreateTextField(form,"dummy", args, n); n = 0;
  XtManageChild(dummy);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, menubar); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n ++;
  
  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n ++;
  XtSetArg(args[n], XmNlistSizePolicy,XmCONSTANT); n ++;
  XtSetArg(args[n], XmNvisibleItemCount, 10 ); n ++; 
  XtSetArg(args[n], XmNselectionPolicy,XmSINGLE_SELECT); n ++;
  compose->list_list = 
    XmCreateScrolledList(form,"list",args,n); n = 0;
  XtAddCallback(compose->list_list,
		XmNsingleSelectionCallback, compose_list_select, compose);
  XtAddCallback(compose->list_list,
		XmNdefaultActionCallback, compose_list_select, compose);

  XtAddCallback(compose->list_shell,
		XmNdestroyCallback, compose_list_cancel, compose);
 
  translations = 
    XtParseTranslationTable(GLOBAL_modal_list_translations);
  XtOverrideTranslations(compose->list_list,translations);
  stuff_window_list(compose);

  XtManageChild(compose->list_list); 
  XtManageChild(form);
  XtPopup(compose->list_shell, XtGrabExclusive);
  
  modal_main_loop(&compose->list_done);


  XtPopdown(compose->list_shell);
  XtDestroyWidget(compose->list_shell);
  reset_cursors();

}



void compose_list_select(w,compose,xp)
     Widget w;
     Compose *compose;
     XmListCallbackStruct *xp;
{

  Compose_Info *compose_info;

  for(compose_info = compose->compose_info;
      compose_info->number != xp->item_position;
      compose_info = compose_info->prev);

  compose_context_switch(compose,compose->current,compose_info);
  compose->list_done = 1;

}


void compose_list_cancel(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  compose->list_done = 1;

}


void compose_list_help(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  help(compose->list_shell,"Compose Window Selections","composelist.help");

}





void stuff_window_list(compose)
     Compose *compose;
{
  Compose_Info *compose_info;
  int n = 1;
  char buffer[FILEBUFFLEN];
  XmString xstr;

  /* Update the structure contents in case they've changed. */

  compose_context_switch(compose,compose->current,NULL);


  for(compose_info = compose->compose_info;
      compose_info; compose_info = compose_info->prev) {

    compose_info->number = n;

    sprintf(buffer,"[%s] %-32.32s [%s] %-32.32s [%s] %-32.32s",
	    SUBJECT_TEXT,
	    (*compose_info->subject) ? compose_info->subject : "<unset>",
	    TO_TEXT,
	    (*compose_info->to) ? compose_info->to : "<unset>",
	    "Title:",
	    compose_info->title);
    
    xstr = XmStringCreateSimple(buffer);
    XmListAddItemUnselected(compose->list_list,xstr,0);
    XmStringFree(xstr);
    n ++;

  }
}

/* 
 * We've been called by an address text field widget. At least that's the
 * assumption, so don't try and change it. All we have is the widget.
 * It contains something presumably that we will have to complete from
 * the address book. But it might contain other addresses, and the one to be
 * completed could be anywhere. We presume that it is under the cursor.
 * We then split up the string into three parts. The part before the current
 * word (comma delimited), and anything afterward (delimited by comma or a 
 * space). Any of these parts might be empty. After we extract our pattern,
 * we look it up, and replace it with the address book address. More on 
 * that later. We then combine it with any or all of the other parts, and 
 * make sure we have comma delimiters (plus a gratuitous space) between 
 * all the fields. On failure we leave it alone, and advance to the next
 * tab group, since the most likely key binding for this will be a tab key.
 * Therefore, the tab completes, if possible, and in all other cases moves to 
 * the next tab group.
 */

void complete_address(w,xp)
     Widget w;
     XtPointer xp;
{
  char *text = GetTextField(w);
  XmTextPosition pos = XmTextGetInsertionPosition(w);
  char *begin = NULL;
  char *end = NULL;
  char *pattern = NULL;
  char *lookup = NULL;
  XmTextPosition newpos = 0;

  if(*text == NUL_TERM) {
    fs_give((void **) &text);
    XmProcessTraversal(w,XmTRAVERSE_NEXT_TAB_GROUP);
    return;
  }


  /* point to current position. */

  pattern = text + ((pos > 0) ? (pos - 1) : 0 );


  /* find the beginning of this address */

  while((pattern > text) && (*pattern != ','))
	pattern --;


  /* We're at the beginning */

  if(pattern == text) {
    begin = NULL;
  }

  /* 
   * Not the beginning, so set the begin pointer to it,
   * tie it off, and point pattern just beyond it.
   */

  else {
    *pattern = NUL_TERM;
    pattern = first_nonwhite(pattern + 1);
    begin = text;
  }

  /* 
   * find any  subsequent addresses. If we find one, tie off
   * our pattern string, and point end at the next address.
   */

  end = strpbrk(pattern,", ");
  if(end) {
    *end = NUL_TERM;
    end = first_nonwhite(end + 1);
  }

  /* 
   * Try and match it. If we do, rebuild the visible text. Set the
   * cursor at the end of the address we just completed. 
   */

  if((lookup = lookup_address(pattern)) != NULL) {
    XmTextSetString(w,(begin) ? begin : EMPTYSTR);
    AppendText(w,(begin) ? ", " : EMPTYSTR );
    AppendText(w,lookup);
    newpos = XmTextGetInsertionPosition(w);
    if(end) {
      AppendText(w,", ");
      AppendText(w,end);
    }
    XmTextSetInsertionPosition(w,newpos);
  }

  if(lookup)
    fs_give((void **) &lookup);
  else
    XmProcessTraversal(w,XmTRAVERSE_NEXT_TAB_GROUP);

  fs_give((void **) &text);
  return;
}



char *lookup_address(str) 
  char *str;
{
  int cnt = 0;
  
  Address_Book_Info *address_book_info;
  Address_Book_Info *saved = NULL;

  fflush(stdout);

  if(session.address_book == NULL) 
    return;

  for(address_book_info = session.address_book->address_book_info;
      address_book_info ; address_book_info = address_book_info->prev) {
    if(strncasecmp(address_book_info->name,str,strlen(str)) == STRMATCH) {
      cnt ++;
      saved = address_book_info;
    }
  }

  if(cnt == 1)
    return(cpystr(saved->address));

  if(cnt > 1)
    return(address_popup(str));

  return(NULL);
}



char *address_popup(str)
     char *str;
{
  XBell(XtDisplay(top), 500);
  return(NULL);

}




void compose_pgp_sign(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{


  pgp_encode(w,compose,PGP_SIGN);


}



void compose_pgp_cryp(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{


  pgp_encode(w,compose,PGP_CRYP);


}


void compose_pgp_both(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{


  pgp_encode(w,compose,PGP_BOTH);


}



void pgp_encode(w,compose,method)
     Widget w;
     Compose *compose;
     int method;
{

  FILE *fp;
  char *pw;
  Remote_Auth *remote_auth;
  char *text;
  char *wrapped_text;
  char *filename;
  char *users;
  char filenamebuff[MAXPATHLEN];
  char buffer[MAXPATHLEN];
  PART *part;
  BODY *body;
  Binary_Buffer *binary_buffer;

  remote_auth = login(compose->shell, "PGP", "PGP",
		      preferences.pgp_username, NULL);

  if(remote_auth == NULL)
    return;


  if(remote_auth->username == NULL || remote_auth->password == NULL) {
    free_Remote_Auth(remote_auth);
    return;
  }

  filename = tmpnam(filenamebuff);

  if(method != PGP_SIGN) {
    users = make_kv_window(compose->shell);
    if((! users) || (*users == NUL_TERM)) {
      if(users)
	fs_give((void **) &users);
      free_Remote_Auth(remote_auth);
      return;
    }
  }

  switch(method) {
  case PGP_SIGN:
    sprintf(buffer,PGP_SIGN_CMD,remote_auth->username,filename);
    break;
  case PGP_CRYP:
    sprintf(buffer,PGP_CRYP_CMD,remote_auth->username,users,filename);
    break;
  case PGP_BOTH:
  default:
    sprintf(buffer,PGP_BOTH_CMD,remote_auth->username,users,filename);
    break;

  }
      
  text = XmTextGetString(compose->compose_text);

  wrapped_text = wrap_text(text, COMPOSEWIDTH - 1);
  fs_give((void **) &text);

  if((fp = fopen(filename,"w")) != NULL) {
    (void) chmod(filename,S_IRWXU);
    (void) fclose(fp);

    set_watch_cursors();

    mm_log("Executing PGP command:", NIL);
    mm_log(buffer,NIL);

    pw = scramble(remote_auth->password);
    if((write_to_pipe(buffer,pw,wrapped_text,strlen(wrapped_text))) 
       == SYSCALL_SUCCESS) {
      wipeout(pw);
      binary_buffer = load_binary_file(filename);

      part = mail_newbody_part();
      body = &part->body;

      body->type = TYPEAPPLICATION;
      body->subtype = cpystr("pgp");
      body->description = GetTextField(compose->subject_text);

      link_attachment(compose,part,body,binary_buffer);
      free_binary_buffer(binary_buffer);
      XmTextSetString(compose->compose_text,EMPTYSTR);
    }
    else
      mm_log("PGP command failed.",ERROR);

    wipeout(pw);
    fs_give((void **) &pw);
    unlink(filename);

  }

  if(users)
    fs_give((void **) &users);
  fs_give((void **) &wrapped_text);
  free_Remote_Auth(remote_auth);
  reset_cursors();

}



void link_attachment(compose,part,body,binary_buffer)
     Compose *compose;
     PART *part;
     BODY *body;
     Binary_Buffer *binary_buffer;
{
  Part_List *part_list;
  PART *current;
  BODY *subbody;
  Boolean attached = FALSE;
  
  add_attachment_contents_to_body(body,binary_buffer);

  /* 
   * See if there is a selected attachment. Then check to see if
   * it is a multipart. If so, then we add to that multipart.
   */

  if(compose->current->selection && compose->current->part_list) {
    for(part_list = compose->current->part_list;
	part_list->partnumber != compose->current->selection;
	part_list = part_list->next);
    subbody = &(part_list->part->body);
    if(subbody && subbody->type == TYPEMULTIPART) {
      current = subbody->contents.part;
      if(! current)
	subbody->contents.part = part;
      else {
	while(current->next)
	  current = current->next;
	current->next = part;
      }
      attached = TRUE;
    }
  }

  /* 
   * Wasn't a subpart, or we couldn't attach it as one. Attach to the
   * root part.
   */

  if(! attached) {
    if(compose->current->attachments == NULL) {
      compose->current->attachments = part;
    }
    else {
      current = compose->current->attachments;
      while(current->next)
	current = current->next;
      current->next = part;
    }
  }
  
  make_attachment_list(compose);

}

/*
 * This is just to calculate how much space to allocate for storing
 * a text field. (Used in the print function). We over-allocate by
 * a little bit to make up for psuedo-tabs and line breaks.
 */

unsigned long compose_field_approx_size(prefix, str, len)
     char *prefix;
     char *str;
     unsigned long len;
{
  unsigned long retval = 0L;

  if(!str)
    return(0L);

  if(len == 0L)
    return((unsigned long) strlen(str));

  if(prefix) 
    retval = strlen(prefix);
  retval += (unsigned long) strlen(str) + 
    ((unsigned long) strlen(str) / len ) + (unsigned long) 16L;

  return(retval);

}



void compose_print(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  char *s;
  char *wrapped_text;
  unsigned long size = 0L;

  if(!preferences.print_command || *preferences.print_command == NUL_TERM)
    return;

  set_watch_cursors();
  compose_context_switch(compose,compose->current,compose->current);

  /* Allow enough room for our header text and a tab stop. */

#define HEADER_MARGIN (COMPOSEWIDTH - 20) 

  size +=  compose_field_approx_size(TO_TEXT,
				     compose->current->to,
				     HEADER_MARGIN);
  size +=  compose_field_approx_size(NG_TEXT,
				     compose->current->newsgroups,
				     HEADER_MARGIN);

  size +=  compose_field_approx_size(SUBJECT_TEXT,
				     compose->current->subject,
				     HEADER_MARGIN);

  size +=  compose_field_approx_size(CC_TEXT,
				     compose->current->cc,
				     HEADER_MARGIN);

  size +=  compose_field_approx_size(BCC_TEXT,
				     compose->current->bcc,
				     HEADER_MARGIN);

  size +=  compose_field_approx_size(NULL, 
				     compose->current->text,
				     COMPOSEWIDTH - 1);

  s = (char *) fs_get(size);

  *s = NUL_TERM;

  if(*compose->current->to) {
    strcat(s,TO_TEXT);
    strcat(s,TABSTR);
    wrapped_text = wrap_text(compose->current->to, HEADER_MARGIN);
    strcat(s,wrapped_text);
    strcat(s,LFSTR);
    fs_give((void **) &wrapped_text);
  }
  if(*compose->current->newsgroups) {
    strcat(s,NG_TEXT);
    strcat(s,TABSTR);
    wrapped_text = wrap_text(compose->current->newsgroups, HEADER_MARGIN);
    strcat(s,wrapped_text);
    strcat(s,LFSTR);
    fs_give((void **) &wrapped_text);
  }
  if(*compose->current->subject) {
    strcat(s,SUBJECT_TEXT);
    strcat(s,TABSTR);
    wrapped_text = wrap_text(compose->current->subject, HEADER_MARGIN);
    strcat(s,wrapped_text);
    strcat(s,LFSTR);
    fs_give((void **) &wrapped_text);

  }
  if(*compose->current->cc) {
    strcat(s,CC_TEXT);
    strcat(s,TABSTR);
    wrapped_text = wrap_text(compose->current->cc, HEADER_MARGIN);
    strcat(s,wrapped_text);
    strcat(s,LFSTR);
    fs_give((void **) &wrapped_text);
  }
  if(*compose->current->bcc) {
    strcat(s,BCC_TEXT);
    strcat(s,TABSTR);
    wrapped_text = wrap_text(compose->current->bcc, HEADER_MARGIN);
    strcat(s,wrapped_text);
    strcat(s,LFSTR);
    fs_give((void **) &wrapped_text);
  }

  strcat(s,LFSTR);

  if(*compose->current->text) {
    wrapped_text = wrap_text(compose->current->text, COMPOSEWIDTH - 1);
    strcat(s,wrapped_text);
    strcat(s,LFSTR);
    fs_give((void **) &wrapped_text);
  }

  if((write_to_pipe(preferences.print_command, NULL,s,strlen(s))) 
     != SYSCALL_SUCCESS)
    mm_log("Print failed.", WARN);
  else
    mm_log("Print successful.", NIL);

  fs_give((void **) &s);
  reset_cursors();
}



void compose_save_draft(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  FILE *fp;
  char *filename;
  Boolean append_it;

  filename = file_select(compose->shell, NULL, "*.draft", NULL, 
			 TRUE, &append_it);
  if(filename == NULL)
     return;

  set_watch_cursors();
  compose_context_switch(compose,compose->current,compose->current);

  if((fp = fopen(filename,(append_it == TRUE) ? "a" : "w")) != NULL) {
    (void) chmod(filename,S_IRWXU);
    fprintf(fp,"%s\n",ML_DRAFT_HEADER);
    if(*compose->current->to) 
      fprintf(fp,"%s\t%s\n",TO_TEXT,compose->current->to);
    if(*compose->current->newsgroups) 
      fprintf(fp,"%s\t%s\n",NG_TEXT,compose->current->newsgroups);
    if(*compose->current->subject) 
      fprintf(fp,"%s\t%s\n",SUBJECT_TEXT,compose->current->subject);
    if(*compose->current->cc) 
      fprintf(fp,"%s\t%s\n",CC_TEXT,compose->current->cc);
    if(*compose->current->bcc) 
      fprintf(fp,"%s\t%s\n",BCC_TEXT,compose->current->bcc);
    fprintf(fp,"\n");
    fprintf(fp,"%s",compose->current->text);
    fclose(fp);
  }

  fs_give((void **) &filename);
  reset_cursors();
}






void compose_load_draft(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  FILE *fp;
  char *filename;
  char buffer[2 * FILEBUFFLEN];
  struct stat st;
  int c;
  char *str;
  char *dst;

  filename = file_select(compose->shell, NULL, "*.draft", NULL, FALSE, NULL);
  if(filename == NULL)
     return;

  if(stat(filename,&st) != SYSCALL_SUCCESS) {
    fs_give((void **) &filename);
    return;
  }
  if((fp = fopen(filename,"r")) != NULL) {
    fgets(buffer,sizeof(buffer),fp);
    buffer[strlen(buffer) - 1] = NUL_TERM;
    if(strcmp(buffer,ML_DRAFT_HEADER) != STRMATCH) {
      fs_give((void **) &filename);
      mm_log(NOT_A_DRAFT,ERROR);
      return;
    }
    set_watch_cursors();
    while(fgets(buffer,sizeof(buffer),fp) != NULL) {
      buffer[strlen(buffer) - 1] = NUL_TERM;
      if(! strlen(buffer))
	break;
      if(strncmp(buffer,TO_TEXT,strlen(TO_TEXT)) == STRMATCH) 
	XmTextSetString(compose->to_text,buffer + strlen(TO_TEXT) + 1);
      if(strncmp(buffer,NG_TEXT,strlen(NG_TEXT)) == STRMATCH) 
	XmTextSetString(compose->ng_text,buffer + strlen(NG_TEXT) + 1);
      if(strncmp(buffer,SUBJECT_TEXT,strlen(SUBJECT_TEXT)) == STRMATCH) 
	XmTextSetString(compose->subject_text,
			buffer + strlen(SUBJECT_TEXT) + 1);
      if(strncmp(buffer,CC_TEXT,strlen(CC_TEXT)) == STRMATCH) 
	XmTextSetString(compose->cc_text,buffer + strlen(CC_TEXT) + 1);
      if(strncmp(buffer,BCC_TEXT,strlen(BCC_TEXT)) == STRMATCH) 
	XmTextSetString(compose->bcc_text,buffer + strlen(BCC_TEXT) + 1);
    }

    str = (char *) fs_get(st.st_size + 1);
    dst = str;

    while((c = fgetc(fp)) != EOF) { 
      *dst = c;
      dst ++;
    }
    *dst = NUL_TERM;
    fclose(fp);
    XmTextSetString(compose->compose_text,str);
    fs_give((void **) &str);
  }

  fs_give((void **) &filename);
  reset_cursors();

}


void compose_unwrap(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  
  text_unwrap(compose->compose_text,NULL);
  set_edit_buttons(w,compose,xp);

}



void compose_strip_prefix(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  char *prefix;

  prefix = GetTextField(compose->edit->strip_text);
  if(prefix == NULL)
    return;
  if(*prefix == NUL_TERM) {
    fs_give((void **) &prefix);
    return;
  }
  text_strip_prefix(compose->compose_text,prefix);
  fs_give((void **) &prefix);
  set_edit_buttons(w,compose,xp);
  
}


void compose_rewrap(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  char *value;
  int len;

  value = GetTextField(compose->edit->wrap_text);
  if(value == NULL)
    return;
  len = atoi(value);
  if(len <= 0 || len > COMPOSEWIDTH) {
    mm_log("Illegal Wrap Length.",WARN);
    fs_give((void **) &value);
    return;
  }

  text_rewrap(compose->compose_text,len);
  fs_give((void **) &value);
  set_edit_buttons(w,compose,xp);
  
}



void compose_add_prefix(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  char *prefix;

  prefix = GetTextField(compose->edit->insert_text);
  if(prefix == NULL)
    return;
  if(*prefix == NUL_TERM) {
    fs_give((void **) &prefix);
    return;
  }
  text_prefix(compose->compose_text,prefix);
  fs_give((void **) &prefix);
  set_edit_buttons(w,compose,xp);
  
}

void compose_reply_insert(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  XmTextPosition left, right;
  char *the_text;
  Time timestamp;
  Boolean found = FALSE;

  set_watch_cursors();
  if(session.read && session.read->is_realized) {
    XmTextGetSelectionPosition(session.read->read_text,&left,&right);
    if(left != right) {
      the_text = XmTextGetSelection(session.read->read_text);
      found = TRUE;
    }
  }
  if(! found) {
    if(compose->current->reply_text) {
      the_text = cpystr(compose->current->reply_text);
      found ++;
    }
  }
  if((found)  && (*the_text != NUL_TERM)) {

    left = XmTextGetInsertionPosition(compose->compose_text);
    right = left + strlen(the_text);
    text_blast(compose->compose_text,the_text);

    timestamp = XtLastTimestampProcessed(XtDisplay(compose->compose_text));
    XmTextSetSelection(compose->compose_text,left,right,timestamp);
  }
  if(the_text)
    fs_give((void **) &the_text);

  create_edit_menu(w,compose,xp);

  reset_cursors();
  return;
}




void compose_edit_copy(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  XmTextPosition left, right;
  char *the_text;
  Time timestamp;

  XmTextGetSelectionPosition(compose->compose_text,&left,&right);

  if(left == right)
    return;

  the_text = XmTextGetSelection(compose->compose_text);

  if(compose->edit->buffer)
    fs_give((void **) &compose->edit->buffer);

  compose->edit->buffer = the_text;
  compose->edit->begin = left;
  compose->edit->end   = right;

  set_edit_buttons(w,compose,xp);

}


void compose_edit_paste(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  if(compose->edit->buffer) 
    text_blast(compose->compose_text,compose->edit->buffer);

}




void compose_edit_undo(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  Time timestamp;

  if(compose->edit->buffer) {
    XmTextReplace(compose->compose_text,
		  compose->edit->begin,
		  compose->edit->end,compose->edit->buffer);
    timestamp = XtLastTimestampProcessed(XtDisplay(compose->compose_text));
    XmTextSetSelection(compose->compose_text,
		       compose->edit->begin,compose->edit->end,timestamp);

    fs_give((void **) &compose->edit->buffer);
    compose->edit->buffer = NULL;
  }

  set_edit_buttons(w,compose,xp);
}



void compose_edit_destroy(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  compose->edit->is_realized = FALSE;

}


void compose_edit_dismiss(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  XtDestroyWidget(compose->edit->shell);
  compose->edit->is_realized = FALSE;

}



void compose_edit_help(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{

  help(compose->edit->shell,"Compose Format","editcompose.help");

}


void set_edit_buttons(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  Arg args[1];
  int n = 0;

  if(compose->edit->buffer) 
    XtSetArg(args[0], XmNsensitive,TRUE);
  else
    XtSetArg(args[0], XmNsensitive, FALSE);

  XtSetValues(compose->edit->undo,args,1);
  XtSetValues(compose->edit->paste,args,1);

}


void create_edit_menu(w,compose,xp)
     Widget w;
     Compose *compose;
     XtPointer xp;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Position x, y;
  XtTranslations translations;
  Widget form, menubar, button1, button2, button3, button4, button5, button6;

  if(compose->edit->is_realized == TRUE) {
    XRaiseWindow(display,XtWindow(compose->edit->shell));
    return;
  }

  get_pointer_position(w, &x, &y);

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  compose->edit->shell = XtCreateWidget("Edit",
					 topLevelShellWidgetClass,
					 compose->shell, args, n ); n = 0;

  form = XmCreateForm(compose->edit->shell, "form", args, n); n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form,"menubar", args, n); n = 0;
  XtManageChild(menubar);
  create_menu_buttons(NULL, menubar,compose_format_menu, 
		      XtNumber(compose_format_menu), BTN_ON, 
		      (XtPointer) compose);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n ++;
  XtSetArg(args[n], XmNtopWidget, menubar);             n ++;

  compose->edit->undo = XmCreatePushButton(form,"Undo!", args, n); n = 0;
  XtAddCallback(compose->edit->undo,XmNactivateCallback,
		compose_edit_undo,compose);
  XtManageChild(compose->edit->undo);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET);  n ++;
  XtSetArg(args[n], XmNtopWidget, menubar);             n ++;
  XtSetArg(args[n], XmNleftWidget, compose->edit->undo); n ++;
  button1 = XmCreatePushButton(form,"Copy", args, n); n = 0;
  XtAddCallback(button1,XmNactivateCallback,compose_edit_copy,compose);
  XtManageChild(button1);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET);  n ++;
  XtSetArg(args[n], XmNtopWidget, menubar);             n ++;
  XtSetArg(args[n], XmNleftWidget, button1);            n ++;
  compose->edit->paste = XmCreatePushButton(form,"Paste", args, n); n = 0;
  XtAddCallback(compose->edit->paste,XmNactivateCallback,
		compose_edit_paste,compose);
  XtManageChild(compose->edit->paste);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n ++;
  XtSetArg(args[n], XmNtopWidget, compose->edit->paste); n ++;
  button2 = XmCreatePushButton(form,"Remove Line Breaks ", args, n); n = 0;
  XtAddCallback(button2,XmNactivateCallback,compose_unwrap,compose);
  XtManageChild(button2);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n ++;
  XtSetArg(args[n], XmNtopWidget, button2);             n ++;
  button3 = XmCreatePushButton(form,"Wrap - Column", args, n); n = 0;
  XtAddCallback(button3,XmNactivateCallback,compose_rewrap,compose);
  XtManageChild(button3);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, button2);             n ++;
  XtSetArg(args[n], XmNleftWidget,button3);             n ++; 
  XtSetArg(args[n], XmNcolumns, 3);                     n ++; 
  compose->edit->wrap_text = 
    XmCreateTextField(form,"wrap_text", args, n); n = 0;
  AppendText(compose->edit->wrap_text, DEFAULTEDITWRAPCOL_STR);
  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->edit->wrap_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_terminal_text_field_translations);
  XtOverrideTranslations(compose->edit->wrap_text,translations);
  XtAddCallback(compose->edit->wrap_text,XmNactivateCallback,
		compose_rewrap,compose);

  XtManageChild(compose->edit->wrap_text);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n ++;
  XtSetArg(args[n], XmNtopWidget, button3);             n ++;
  button4 = XmCreatePushButton(form,"Strip Prefix ", args, n); n = 0;
  XtAddCallback(button4,XmNactivateCallback,compose_strip_prefix,compose);
  XtManageChild(button4);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, button3);             n ++;
  XtSetArg(args[n], XmNleftWidget,button4);             n ++; 
  XtSetArg(args[n], XmNcolumns, 3);                     n ++; 
  compose->edit->strip_text = 
    XmCreateTextField(form,"strip_text", args, n); n = 0;

  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->edit->strip_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_terminal_text_field_translations);
  XtOverrideTranslations(compose->edit->strip_text,translations);
  XtAddCallback(compose->edit->strip_text,XmNactivateCallback,
		compose_strip_prefix,compose);

  AppendText(compose->edit->strip_text, DEFAULTSTRIP_STR);

  XtManageChild(compose->edit->strip_text);


  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n ++;
  XtSetArg(args[n], XmNtopWidget, button4);             n ++;
  button5 = XmCreatePushButton(form,"Add Prefix   ", args, n); n = 0;
  XtAddCallback(button5,XmNactivateCallback,compose_add_prefix,compose);
  XtManageChild(button5);

  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n ++;
  XtSetArg(args[n], XmNtopWidget, button4);             n ++;
  XtSetArg(args[n], XmNleftWidget,button5);             n ++; 
  XtSetArg(args[n], XmNcolumns, 3);                     n ++; 
  compose->edit->insert_text = 
    XmCreateTextField(form,"insert_text", args, n); n = 0;

  translations = 
    XtParseTranslationTable(GLOBAL_nonterminal_text_field_translations);
  XtOverrideTranslations(compose->edit->insert_text,translations);
  translations = 
    XtParseTranslationTable(GLOBAL_terminal_text_field_translations);
  XtOverrideTranslations(compose->edit->insert_text,translations);
  XtAddCallback(compose->edit->insert_text,XmNactivateCallback,
		compose_add_prefix,compose);
  AppendText(compose->edit->insert_text, DEFAULTINSERT_STR);

  XtManageChild(compose->edit->insert_text);

  XtAddCallback(compose->edit->shell, XmNdestroyCallback, 
		compose_edit_destroy, compose);


  XtManageChild(form);
  XtManageChild(compose->edit->shell);
  XtRealizeWidget(compose->edit->shell);
  compose->edit->is_realized = TRUE;
  set_edit_buttons(NULL, compose, NULL);
  return;
}


/*
 * Called from mm_expunge() on an expunge event. We will be passed the 
 * mailstream initiating the event, and the message number in question.
 * That message will no longer exist. Since "Reply" messages reference
 * the mailstream to set the Answered flag upon send, we have to update
 * any of these references. All message numbers higher than the one in
 * question will also need to be decreased by one (we only handle a single
 * message at a time, even though the function might get called repeatedly.)
 */


#ifdef __STDC__
void compose_expunge(MAILSTREAM *mailstream, unsigned long msgno)
#else
void compose_expunge(mailstream,msgno)
     MAILSTREAM *mailstream;
     unsigned long msgno;
#endif
{
  Compose_Info *compose_info;

  /*
   * Act I. See if we even need to do anything. Lack of a 
   * compose structure or any compose_info structures means
   * there's no compose sessions to worry about.
   */

  if(session.compose == NULL)
    return;
  if(session.compose->compose_info == NULL)
    return;


  /* 
   * We weren't so lucky. run through the list of compose_info's
   * Searching for our mailstream. Then check the msgno. If it's ours,
   * zero it, so it won't be referenced again. (The message can still
   * be composed -- we just won't set any flags at the end). If it's
   * a higher message number than ours, decrement it. 
   */


  for(compose_info = session.compose->compose_info;
      compose_info; compose_info = compose_info->prev) {
    if(compose_info->mailstream != mailstream)
      continue;
    if(compose_info->msgno == msgno) {
      compose_info->msgno = 0L;
      compose_info->mailstream = NULL;
    }
    if(compose_info->msgno > msgno)
      compose_info->msgno-- ;

  }
}


#ifdef __STDC__
void compose_mailbox_close(MAILSTREAM *mailstream)
#else
void compose_mailbox_close(mailstream)
     MAILSTREAM *mailstream;
#endif
{
  Compose_Info *compose_info;

  /*
   * Act I. See if we even need to do anything. Lack of a 
   * compose structure or any compose_info structures means
   * there's no compose sessions to worry about.
   */

  if(session.compose == NULL)
    return;
  if(session.compose->compose_info == NULL)
    return;

  /* 
   * We weren't so lucky. run through the list of compose_info's
   * Searching for our mailstream. If it's ours, zero it, so it 
   * won't be referenced again. (The message can still
   * be composed -- we just won't set any flags at the end).
   */

  for(compose_info = session.compose->compose_info;
      compose_info; compose_info = compose_info->prev) {
    if(compose_info->mailstream != mailstream)
      continue;
    compose_info->msgno = 0L;
    compose_info->mailstream = NULL;
  }
}


