/*
 *  Copyright (c) 1990 Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  ufn.c
 */

#ifndef lint 
static char copyright[] = "@(#) Copyright (c) 1993 Regents of the University of Michigan.\nAll rights reserved.\n";
#endif

#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifdef MACOS
#include <stdlib.h>
#include "macos.h"
#else /* MACOS */
#ifdef DOS
#include "msdos.h"
#else /* DOS */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#endif /* DOS */
#endif /* MACOS */

#ifdef NEEDPROTOS
typedef int (*cancelptype)( long cancellong );
#else /* NEEDPROTOS */
typedef int (*cancelptype)();
#endif /* NEEDPROTOS */

#ifdef NEEDPROTOS
static LDAPMessage *ldap_msg_merge( LDAP *ld, LDAPMessage *a, LDAPMessage *b );
static LDAPMessage *ldap_ufn_expand( LDAP *ld, cancelptype cancelproc,
	long cancellong, LDAPMessage *candidates, char *filter, int scope,
	char **attrs, int aonly, int *err );
#else
static LDAPMessage *ldap_msg_merge();
static LDAPMessage *ldap_ufn_expand();
#endif /* NEEDPROTOS */

/*
 * ldap_ufn_search_c - do user friendly searching; provide cancel feature
 *
 *	ld		LDAP descriptor
 *	ufn		the user friendly name to look for
 *	attrs		list of attribute types to return for matches
 *	attrsonly	1 => attributes only 0 => attributes and values
 *	res		will contain the result of the search
 *	cancelproc	routine that returns non-zero if operation should be
 *			cancelled.  This can be NULL.  If it is non-NULL, the
 *			routine will be called periodically.
 *	cancellong	long that is passed to cancelproc
 *
 * Example:
 *	char		*attrs[] = { "mail", "title", 0 };
 *	LDAPMessage	*res;
 *	error = ldap_ufn_search_c( ld, "howes, umich, us", attrs, attrsonly,
 *			&res, acancelproc, along );
 */

ldap_ufn_search_c( ld, ufn, attrs, attrsonly, res, cancelproc, cancellong )
    LDAP	*ld;
    char	*ufn;
    char	**attrs;
    int		attrsonly;
    LDAPMessage	**res;
    cancelptype	cancelproc;
    long	cancellong;
{
	char		*comma, *comp;
	char		filter[256];
	int		err, scope;
	LDAPMessage	*tmpcand;
	LDAPMessage	*candidates;
	LDAPMessage	*ldap_msg_merge(), *ldap_ufn_expand();
	static char	*objattrs[] = { "objectClass", NULL };

	/* make a copy we can mess with */
	ufn = strdup( ufn );
	*res = NULL;
	candidates = NULL;

	/*
	 * look up the most significant component of the ufn first, to
	 * generate an initial list of candidate DNs.
	 */

	if ( (comma = strrchr( ufn, ',' )) != NULL ) {
		int	countrycode;

		comp = comma + 1;
		while ( isascii( *comp ) && isspace( *comp ) )
			comp++;

		countrycode = (strlen( comp ) == 2);
		if ( countrycode ) {
			sprintf( filter, "(c=%s)", comp );
		} else {
			sprintf( filter, "(|(o=%s)(l=%s)(co=%s))",
			    comp, comp, comp );
		}

		if ( (candidates = ldap_ufn_expand( ld, cancelproc, cancellong,
		    NULL, filter, LDAP_SCOPE_ONELEVEL, objattrs, 1, &err ))
		    == NULL ) {
			if ( err == -1 || err == LDAP_USER_CANCELLED ) {
				free( ufn );
				return( err );
			}

			sprintf( filter, "(|(o~=%s)(l~=%s)(co~=%s))",
			    comp, comp, comp );

			if ( (candidates = ldap_ufn_expand( ld, cancelproc,
			    cancellong, NULL, filter, LDAP_SCOPE_ONELEVEL,
			    objattrs, 1, &err )) == NULL ) {
				free( ufn );
				return( err );
			}
		}

		*comma = '\0';
	}

	/* 
	 * look up subsequent components from most to least significant.
	 * we are building a list of candidate DNs, below which we will
	 * search for the final component of the ufn.  this loop is for
	 * each component of the ufn except the last.  for each component
	 * we try a single level exact followed by single level approx
	 * followed by subtree exact followed by subtree approx, giving
	 * up only if all of these fail.  if any of them produce any
	 * results, we go on to the next component.
	 */

	while ( (comma = strrchr( ufn, ',' )) != NULL ) {
		comp = comma + 1;
		while ( isascii( *comp ) && isspace( *comp ) )
			comp++;
		scope = LDAP_SCOPE_ONELEVEL;

	tryagain:
		sprintf( filter, "(|(ou=%s)(o=%s)(l=%s))",
		    comp, comp, comp );

		if ( (tmpcand = ldap_ufn_expand( ld, cancelproc, cancellong,
		    candidates, filter, scope, objattrs, 1, &err )) == NULL ) {
			if ( err == -1 || err == LDAP_USER_CANCELLED ) {
				ldap_msgfree( candidates );
				free( ufn );
				return( err );
			}

			sprintf( filter, "(|(ou~=%s)(o~=%s)(l~=%s))",
			    comp, comp, comp );

			if ( (tmpcand = ldap_ufn_expand( ld, cancelproc,
			    cancellong, candidates, filter, scope, objattrs,
			    1, &err )) == NULL ) {
				if ( err == -1 || err == LDAP_USER_CANCELLED ) {
					ldap_msgfree( candidates );
					free( ufn );
					return( err );
				}

				if ( scope == LDAP_SCOPE_SUBTREE ) {
					ldap_msgfree( candidates );
					free( ufn );
					return( err );
				}

				scope = LDAP_SCOPE_SUBTREE;
				goto tryagain;
			}

		}

		ldap_msgfree( candidates );
		candidates = tmpcand;

		/* go on to the next component */
		*comma = '\0';
	}

	/* look up the last component */
	sprintf( filter, "(|(cn=%s)(sn=%s)(uid=%s))", ufn, ufn, ufn );

	if ( (*res = ldap_ufn_expand( ld, cancelproc, cancellong, candidates,
	    filter, LDAP_SCOPE_SUBTREE, attrs, attrsonly, &err )) == NULL ) {
		if ( err == -1 || err == LDAP_USER_CANCELLED ) {
			ldap_msgfree( candidates );
			free( ufn );
			return( err );
		}

		sprintf( filter, "(|(cn~=%s)(sn~=%s)(uid~=%s))", ufn, ufn,
		    ufn );

		*res = ldap_ufn_expand( ld, cancelproc, cancellong, candidates,
		    filter, LDAP_SCOPE_SUBTREE, attrs, attrsonly, &err );
	}

	free( ufn );
	ldap_msgfree( candidates );

	return( err );
}

/*
 * same as ldap_ufn_search without the cancel function
 */
ldap_ufn_search_s( ld, ufn, attrs, attrsonly, res )
    LDAP	*ld;
    char	*ufn;
    char	**attrs;
    int		attrsonly;
    LDAPMessage	**res;
{
	return( ldap_ufn_search_c( ld, ufn, attrs, attrsonly, res, NULL, 0L ));

}


/*
 * ldap_msg_merge - merge two ldap search result chains.  the more
 * serious of the two error result codes is kept.
 */

static LDAPMessage *ldap_msg_merge( ld, a, b )
    LDAP	*ld;
    LDAPMessage	*a;
    LDAPMessage	*b;
{
	LDAPMessage	*end, *aprev, *aend, *bprev, *bend;

	if ( a == NULL )
		return( b );

	if ( b == NULL )
		return( a );

	/* find the ends of the a and b chains */
	aprev = NULL;
	for ( aend = a; aend->lm_chain != NULL; aend = aend->lm_chain )
		aprev = aend;
	bprev = NULL;
	for ( bend = b; bend->lm_chain != NULL; bend = bend->lm_chain )
		bprev = bend;

	/* keep result a */
	if ( ldap_result2error( ld, aend, 0 ) != LDAP_SUCCESS ) {
		/* remove result b */
		ldap_msgfree( bend );
		if ( bprev != NULL )
			bprev->lm_chain = NULL;
		else
			b = NULL;
		end = aend;
		if ( aprev != NULL )
			aprev->lm_chain = NULL;
		else
			a = NULL;
	/* keep result b */
	} else {
		/* remove result a */
		ldap_msgfree( aend );
		if ( aprev != NULL )
			aprev->lm_chain = NULL;
		else
			a = NULL;
		end = bend;
		if ( bprev != NULL )
			bprev->lm_chain = NULL;
		else
			b = NULL;
	}

	if ( (a == NULL && b == NULL) || (a == NULL && bprev == NULL) ||
	    (b == NULL && aprev == NULL) )
		return( end );

	if ( a == NULL ) {
		bprev->lm_chain = end;
		return( b );
	} else if ( b == NULL ) {
		aprev->lm_chain = end;
		return( a );
	} else {
		bprev->lm_chain = end;
		aprev->lm_chain = b;
		return( a );
	}
}

static LDAPMessage *ldap_ufn_expand( ld, cancelproc, cancellong, candidates,
	filter, scope, attrs, aonly, err )
    LDAP	*ld;
    cancelptype	cancelproc;
    long	cancellong;
    LDAPMessage	*candidates;
    char	*filter;
    int		scope;
    char	**attrs;
    int		aonly;
    int		*err;
{
	LDAPMessage	*tmpcand, *tmpres;
	char		*dn;
	int		msgid;
	struct timeval	tv;

	/* search for this component below the current candidates */
	tmpcand = NULL;
	do {
		if ( candidates == NULL )
			dn = "";
		else {
			if ( candidates->lm_msgtype == LDAP_RES_SEARCH_RESULT )
				break;
			dn = ldap_get_dn( ld, candidates );
		}

		if (( msgid = ldap_search( ld, dn, scope, filter, attrs,
		    aonly )) == -1 ) {
			ldap_msgfree( tmpcand );
			free( dn );
			*err = ld->ld_errno;
			return( NULL );
		}

		tv.tv_sec = 0;
		tv.tv_usec = 100000;	/* 1/10 of a second */

		do {
			*err = ldap_result( ld, msgid, 1, &tv, &tmpres );
			if ( *err == 0 && cancelproc != NULL &&
			    (*cancelproc)( cancellong ) != 0 ) {
				ldap_abandon( ld, msgid );
				*err = LDAP_USER_CANCELLED;
			}
		} while ( *err == 0 );

		if ( *err == LDAP_USER_CANCELLED || *err < 0 ||
		    ( *err = ldap_result2error( ld, tmpres, 0 )) == -1 ) {
			ldap_msgfree( tmpcand );
			free( dn );
			return( NULL );
		}
		
		tmpcand = ldap_msg_merge( ld, tmpcand, tmpres );

		if ( candidates != NULL ) {
			free( dn );
			candidates = candidates->lm_chain;
		}
	} while ( candidates != NULL && candidates->lm_msgtype !=
	    LDAP_RES_SEARCH_ENTRY );

	if ( ldap_count_entries( ld, tmpcand ) > 0 ) {
		return( tmpcand );
	} else {
		ldap_msgfree( tmpcand );
		return( NULL );
	}
}
