#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#ifdef MACOS
#include <stdlib.h>
#include <console.h>
#include <unix.h>
#include <fcntl.h>
#include "macos.h"
#else /* MACOS */
#ifdef DOS
#include "msdos.h"
#ifdef WINSOCK
#include "console.h"
#endif /* WINSOCK */
#else /* DOS */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/file.h>
#endif /* DOS */
#endif /* MACOS */

#ifndef L_SET
#define L_SET	SEEK_SET
#define L_XTND	SEEK_END
#endif /* L_SET */

#if !defined( PCNFS ) && !defined( WINSOCK )
#define MOD_USE_BVALS
#endif /* !PCNFS && !WINSOCK */

#ifdef NEEDPROTOS
static int handle_result( LDAP *ld, LDAPMessage *lm );
static int print_ldap_result( LDAP *ld, LDAPMessage *lm, char *s );
static int print_search_entry( LDAP *ld, LDAPMessage *res );
#else
static int handle_result();
static int print_ldap_result();
static int print_search_entry();
#endif /* NEEDPROTOS */


#ifndef WINSOCK
static char *getline( line, len, s, prompt )
char	*line;
int	len;
FILE	*s;
char	*prompt;
{
	printf(prompt);

	if ( fgets( line, len, s ) == NULL )
		return( NULL );

	line[ strlen( line ) - 1 ] = '\0';

	return( line );
}
#endif /* WINSOCK */

static char **get_list( prompt )
char	*prompt;
{
	static char	buf[256];
	int		num;
	char		**result;

	num = 0;
	result = (char **) 0;
	while ( 1 ) {
		getline( buf, sizeof(buf), stdin, prompt );

		if ( *buf == '\0' )
			break;

		if ( result == (char **) 0 )
			result = (char **) malloc( sizeof(char *) );
		else
			result = (char **) realloc( result,
			    sizeof(char *) * (num + 1) );

		result[num++] = (char *) strdup( buf );
	}
	if ( result == (char **) 0 )
		return( NULL );
	result = (char **) realloc( result, sizeof(char *) * (num + 1) );
	result[num] = NULL;

	return( result );
}


static
free_list( list )
char	**list;
{
	int	i;

	if ( list != NULL ) {
		for ( i = 0; list[ i ] != NULL; ++i ) {
			free( list[ i ] );
		}
		free( (char *)list );
	}
}


#ifdef MOD_USE_BVALS
static int file_read( path, bv )
	char		*path;
	struct berval	*bv;
{
	int		fd;

	if (( fd = open( path, O_RDONLY )) < 0 ) {
	    	perror( path );
		return( -1 );
	}

	lseek( fd, 0, L_XTND );
	bv->bv_len = tell( fd );
	if (( bv->bv_val = (char *)malloc( bv->bv_len )) == NULL ) {
		perror( "malloc" );
		close( fd );
		return( -1 );
	}

	lseek( fd, 0, L_SET );
	if ( read( fd, bv->bv_val, bv->bv_len ) != bv->bv_len ) {
		perror( path );
		free( bv->bv_val );
		return( -1 );
	}

	return( bv->bv_len );
}
#endif /* MOD_USE_BVALS */


static LDAPMod **get_modlist( prompt1, prompt2, prompt3 )
char	*prompt1;
char	*prompt2;
char	*prompt3;
{
	static char	buf[256];
	int		num;
	LDAPMod		tmp;
	LDAPMod		**result;
	struct berval	**bvals;

	num = 0;
	result = NULL;
	while ( 1 ) {
		if ( prompt1 ) {
			getline( buf, sizeof(buf), stdin, prompt1 );
			tmp.mod_op = atoi( buf );

			if ( tmp.mod_op == -1 || buf[0] == '\0' )
				break;
		}

		getline( buf, sizeof(buf), stdin, prompt2 );
		if ( buf[0] == '\0' )
			break;
		tmp.mod_type = strdup( buf );

		tmp.mod_values = get_list( prompt3 );
#ifdef MOD_USE_BVALS
		if ( tmp.mod_values != NULL ) {
			int	i;

			for ( i = 0; tmp.mod_values[i] != NULL; ++i )
				;
			bvals = (struct berval **)calloc( i + 1,
			    sizeof( struct berval *));
			for ( i = 0; tmp.mod_values[i] != NULL; ++i ) {
				bvals[i] = (struct berval *)malloc(
				    sizeof( struct berval ));
				if ( strncmp( tmp.mod_values[i], "{FILE}",
				    6 ) == 0 ) {
					if ( file_read( tmp.mod_values[i] + 6,
					    bvals[i] ) < 0 ) {
						return( NULL );
					}
				} else {
					bvals[i]->bv_val = tmp.mod_values[i];
					bvals[i]->bv_len =
					    strlen( tmp.mod_values[i] );
				}
			}
			tmp.mod_bvalues = bvals;
			tmp.mod_op |= LDAP_MOD_BVALUES;
		}
#endif /* MOD_USE_BVALS */

		if ( result == NULL )
			result = (LDAPMod **) malloc( sizeof(LDAPMod *) );
		else
			result = (LDAPMod **) realloc( result,
			    sizeof(LDAPMod *) * (num + 1) );

		result[num] = (LDAPMod *) malloc( sizeof(LDAPMod) );
		*(result[num]) = tmp;	/* struct copy */
		num++;
	}
	if ( result == NULL )
		return( NULL );
	result = (LDAPMod **) realloc( result, sizeof(LDAPMod *) * (num + 1) );
	result[num] = NULL;

	return( result );
}

#ifdef WINSOCK
ldapmain( argc, argv )
#else /* WINSOCK */
main( argc, argv )
#endif /* WINSOCK */
int	argc;
char	**argv;
{
	LDAP		*ld;
	int		method, id, msgtype;
	char		line[256], command1, command2, command3;
	char		passwd[64], dn[256], rdn[64], attr[64], value[256];
	char		filter[256], dnsuffix[256];
	char		**types;
	int		bound, all, scope, attrsonly;
	LDAPMessage	*res;
	LDAPMod		**mods, **attrs;
	struct timeval	timeout;

#ifdef MACOS
	ccommand( &argv );
	for ( argc = 0; argv[ argc ] != NULL; ++argc ) {
		;
	}
	cshow( stdout );
#endif /* MACOS */

	if ( (ld = ldap_open( argc > 1 ? argv[1] : NULL, argc > 3 ?
	    atoi( argv[3] ) : 0 )) == NULL ) {
		perror( "ldap_open" );
		exit(1);
	}

	bound = 0;
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	if ( argc > 2 ) {
		strcpy( dnsuffix, argv[ 2 ] );
	} else {
	    *dnsuffix = '\0';
	}
	(void) memset( line, '\0', sizeof(line) );
	while ( getline( line, sizeof(line), stdin, "\ncommand? " ) != NULL ) {
		command1 = line[0];
		command2 = line[1];
		command3 = line[2];

		switch ( command1 ) {
		case 'a':	/* add or abandon */
			switch ( command2 ) {
			case 'd':	/* add */
				getline( dn, sizeof(dn), stdin, "dn? " );
				strcat( dn, dnsuffix );
				if ( (attrs = get_modlist( NULL, "attr? ",
				    "value? " )) == NULL )
					break;
				if ( (id = ldap_add( ld, dn, attrs )) == -1 )
					ldap_perror( ld, "ldap_add" );
				else
					printf( "Add initiated with id %d\n",
					    id );
				break;

			case 'b':	/* abandon */
				getline( line, sizeof(line), stdin, "msgid? " );
				id = atoi( line );
				if ( ldap_abandon( ld, id ) != 0 )
					ldap_perror( ld, "ldap_abandon" );
				else
					printf( "Abandon successful\n" );
				break;
			default:
				printf( "Possibilities: [ad]d, [ab]ort\n" );
			}
			break;

		case 'b':	/* asynch bind */
#ifdef KERBEROS
			getline( line, sizeof(line), stdin,
			    "method (0->simple, 1->krbv41, 2->krbv42)? " );
			method = atoi( line ) | 0xa0;
#else /* KERBEROS */
			method = LDAP_AUTH_SIMPLE;
#endif /* KERBEROS */
			getline( dn, sizeof(dn), stdin, "dn? " );
			strcat( dn, dnsuffix );

			if ( dn[0] != '\0' )
				getline( passwd, sizeof(passwd), stdin,
				    "password? " );
			else
				passwd[0] = '\0';

			if ( ldap_bind( ld, dn, passwd, method ) == -1 ) {
				fprintf( stderr, "ldap_bind failed\n" );
				ldap_perror( ld, "ldap_bind" );
			} else {
				printf( "Bind initiated\n" );
				bound = 1;
			}
			break;

		case 'B':	/* synch bind */
#ifdef KERBEROS
			getline( line, sizeof(line), stdin,
			    "method 0->simple 1->krbv41 2->krbv42 3->krb? " );
			method = atoi( line );
			if ( method == 3 )
				method = LDAP_AUTH_KRBV4;
			else
				method = method | 0xa0;
#else /* KERBEROS */
			method = LDAP_AUTH_SIMPLE;
#endif /* KERBEROS */
			getline( dn, sizeof(dn), stdin, "dn? " );
			strcat( dn, dnsuffix );

			if ( dn[0] != '\0' )
				getline( passwd, sizeof(passwd), stdin,
				    "password? " );
			else
				passwd[0] = '\0';

			if ( ldap_bind_s( ld, dn, passwd, method ) !=
			    LDAP_SUCCESS ) {
				fprintf( stderr, "ldap_bind_s failed\n" );
				ldap_perror( ld, "ldap_bind_s" );
			} else {
				printf( "Bind successful\n" );
				bound = 1;
			}
			break;

		case 'c':	/* compare */
			getline( dn, sizeof(dn), stdin, "dn? " );
			strcat( dn, dnsuffix );
			getline( attr, sizeof(attr), stdin, "attr? " );
			getline( value, sizeof(value), stdin, "value? " );

			if ( (id = ldap_compare( ld, dn, attr, value )) == -1 )
				ldap_perror( ld, "ldap_compare" );
			else
				printf( "Compare initiated with id %d\n", id );
			break;

		case 'd':	/* turn on debugging */
#ifdef LDAP_DEBUG
			getline( line, sizeof(line), stdin, "debug level? " );
			ldap_debug = atoi( line );
			if ( ldap_debug & LDAP_DEBUG_PACKETS ) {
				lber_debug = ldap_debug;
			}
#else
			printf( "Compile with -DLDAP_DEBUG for debugging\n" );
#endif
			break;

		case 'g':	/* set next msgid */
			getline( line, sizeof(line), stdin, "msgid? " );
			ld->ld_msgid = atoi( line );
			break;

		case 'v':	/* set version number */
			getline( line, sizeof(line), stdin, "version? " );
			ld->ld_version = atoi( line );
			break;

		case 'm':	/* modify or modifyrdn */
			if ( strncmp( line, "modify", 4 ) == 0 ) {
				getline( dn, sizeof(dn), stdin, "dn? " );
				strcat( dn, dnsuffix );
				if ( (mods = get_modlist(
				    "mod (0=>add, 1=>delete, 2=>replace -1=>done)? ",
				    "attribute type? ", "attribute value? " ))
				    == NULL )
					break;
				if ( (id = ldap_modify( ld, dn, mods )) == -1 )
					ldap_perror( ld, "ldap_modify" );
				else
					printf( "Modify initiated with id %d\n",
					    id );
			} else if ( strncmp( line, "modrdn", 4 ) == 0 ) {
				getline( dn, sizeof(dn), stdin, "dn? " );
				strcat( dn, dnsuffix );
				getline( rdn, sizeof(rdn), stdin, "newrdn? " );
				if ( (id = ldap_modrdn( ld, dn, rdn )) == -1 )
					ldap_perror( ld, "ldap_modrdn" );
				else
					printf( "Modrdn initiated with id %d\n",
					    id );
			} else {
				printf( "Possibilities: [modi]fy, [modr]dn\n" );
			}
			break;

		case 'q':	/* quit */
			if ( bound )
				ldap_unbind( ld );
			exit( 0 );
			break;

		case 'r':	/* result or remove */
			switch ( command3 ) {
			case 's':	/* result */
				getline( line, sizeof(line), stdin,
				    "msgid (-1=>any)? " );
				id = atoi( line );
				getline( line, sizeof(line), stdin,
				    "all (0=>any, 1=>all)? " );
				all = atoi( line );
				if ( (msgtype = ldap_result( ld, id, all,
				    &timeout, &res )) < 1 ) {
					ldap_perror( ld, "ldap_result" );
					break;
				}
				printf( "\nresult: msgtype %d msgid %d\n",
				    msgtype, res->lm_msgid );
				handle_result( ld, res );
				res = NULLMSG;
				break;

			case 'm':	/* remove */
				getline( dn, sizeof(dn), stdin, "dn? " );
				strcat( dn, dnsuffix );
				if ( (id = ldap_delete( ld, dn )) == -1 )
					ldap_perror( ld, "ldap_delete" );
				else
					printf( "Remove initiated with id %d\n",
					    id );
				break;

			default:
				printf( "Possibilities: [rem]ove, [res]ult\n" );
				break;
			}
			break;

		case 's':	/* search */
			getline( dn, sizeof(dn), stdin, "searchbase? " );
			strcat( dn, dnsuffix );
			getline( line, sizeof(line), stdin,
			    "scope (0=Base, 1=One Level, 2=Subtree)? " );
			scope = atoi( line );
			getline( filter, sizeof(filter), stdin,
			    "search filter (e.g. sn=jones)? " );
			types = get_list( "attrs to return? " );
			getline( line, sizeof(line), stdin,
			    "attrsonly (0=attrs&values, 1=attrs only)? " );
			attrsonly = atoi( line );

			if ( (id = ldap_search( ld, dn, scope, filter, types,
			    attrsonly  )) == -1 )
				ldap_perror( ld, "ldap_search" );
			else
				printf( "Search initiated with id %d\n", id );
			free_list( types );
			break;

		case 't':	/* set timeout value */
			getline( line, sizeof(line), stdin, "timeout? " );
			timeout.tv_sec = atoi( line );
			break;

		case 'u':	/* user friendly search */
			getline( dn, sizeof(dn), stdin, "ufn? " );
			strcat( dn, dnsuffix );
			types = get_list( "attrs to return? " );
			getline( line, sizeof(line), stdin,
			    "attrsonly (0=attrs&values, 1=attrs only)? " );
			attrsonly = atoi( line );

			if ( (id = ldap_ufn_search_s( ld, dn, types, attrsonly,
			    &res  )) == -1 )
				ldap_perror( ld, "ldap_ufn_search" );
			else {
				if ( res == NULL )
					ldap_perror( ld, "ldap_ufn_search" );
				else {
					printf( "\nresult: err %d msgid %d\n",
					    id, res->lm_msgid );
					handle_result( ld, res );
					res = NULLMSG;
				}
			}
			free_list( types );
			break;

		case 'n':	/* set dn suffix, for convenience */
			getline( line, sizeof(line), stdin, "DN suffix? " );
			strcpy( dnsuffix, line );
			break;

		case 'e':	/* enable cache */
			getline( line, sizeof(line), stdin, "Cache timeout (secs)? " );
			timeout.tv_sec = atoi( line );
			getline( line, sizeof(line), stdin, "Maximum memory to use (bytes)? " );
			if ( ldap_enable_cache( ld, timeout.tv_sec,
			    atoi( line )) == 0 ) {
				printf( "Cache enable OK\n" ); 
			} else {
				printf( "Cache enable failed\n" ); 
			}
			break;

		case 'x':	/* uncache entry */
			getline( line, sizeof(line), stdin, "DN? " );
			ldap_uncache_entry( ld, line );
			break;

		case '?':	/* help */
		printf( "Commands: [a]dd, [b]ind, [c]ompare, [d]ebug\n" );
		printf( "          [modi]fy, [modr]dn, [rem]ove, [res]ult\n" );
		printf( "          [s]earch, [u]fn search, [t]imeout, d[n]suffix\n" );
		printf( "          [e]nable cache, [x]uncache entry, [?]help\n" );
		printf( "          [v]ersion\n" );
			break;

		default:
			printf( "Invalid command.  Type ? for help.\n" );
			break;
		}

		(void) memset( line, '\0', sizeof(line) );
	}

	return( 0 );
}

static handle_result( ld, lm )
LDAP		*ld;
LDAPMessage	*lm;
{
	switch ( lm->lm_msgtype ) {
	case LDAP_RES_COMPARE:
		printf( "Compare result\n" );
		print_ldap_result( ld, lm, "compare" );
		break;

	case LDAP_RES_SEARCH_RESULT:
		printf( "Search result\n" );
		print_ldap_result( ld, lm, "search" );
		break;

	case LDAP_RES_SEARCH_ENTRY:
		printf( "Search entry\n" );
		print_search_entry( ld, lm );
		break;

	case LDAP_RES_ADD:
		printf( "Add result\n" );
		print_ldap_result( ld, lm, "add" );
		break;

	case LDAP_RES_DELETE:
		printf( "Delete result\n" );
		print_ldap_result( ld, lm, "delete" );
		break;

	case LDAP_RES_MODRDN:
		printf( "ModRDN result\n" );
		print_ldap_result( ld, lm, "modrdn" );
		break;

	case LDAP_RES_BIND:
		printf( "Bind result\n" );
		print_ldap_result( ld, lm, "bind" );
		break;

	default:
		printf( "Unknown result type 0x%x\n", lm->lm_msgtype );
		print_ldap_result( ld, lm, "unknown" );
	}
}

static print_ldap_result( ld, lm, s )
LDAP		*ld;
LDAPMessage	*lm;
char		*s;
{
	ldap_result2error( ld, lm, 1 );
	ldap_perror( ld, s );
	if ( ld->ld_error != NULL && *ld->ld_error != '\0' )
		fprintf( stderr, "Additional info: %s\n", ld->ld_error );
	if ( NAME_ERROR( ld->ld_errno ) && ld->ld_matched != NULL )
		fprintf( stderr, "Matched DN: %s\n", ld->ld_matched );
}

static print_search_entry( ld, res )
LDAP		*ld;
LDAPMessage	*res;
{
	BerElement	*ber;
	char		*a, *dn, *ufn;
	struct berval	**vals;
	int		i;
	LDAPMessage	*e;

	for ( e = ldap_first_entry( ld, res ); e != NULLMSG;
	    e = ldap_next_entry( ld, e ) ) {
		if ( e->lm_msgtype == LDAP_RES_SEARCH_RESULT )
			break;

		dn = ldap_get_dn( ld, e );
		printf( "\tDN: %s\n", dn );

		ufn = ldap_dn2ufn( dn );
		printf( "\tUFN: %s\n", ufn );
#ifdef WINSOCK
		ldap_memfree( dn );
		ldap_memfree( ufn );
#else /* WINSOCK */
		free( dn );
		free( ufn );
#endif /* WINSOCK */

		for ( a = ldap_first_attribute( ld, e, &ber ); a != NULL;
		    a = ldap_next_attribute( ld, e, ber ) ) {
			printf( "\t\tATTR: %s\n", a );
			if ( (vals = ldap_get_values_len( ld, e, a ))
			    == NULL ) {
				printf( "\t\t\t(no values)\n" );
			} else {
				for ( i = 0; vals[i] != NULL; i++ ) {
					int	j, nonascii;

					nonascii = 0;
					for ( j = 0; j < vals[i]->bv_len; j++ )
						if ( !isascii( vals[i]->bv_val[j] ) ) {
							nonascii = 1;
							break;
						}

					if ( nonascii ) {
						printf( "\t\t\tlength (%ld) (not ascii)\n", vals[i]->bv_len );
#ifdef BPRINT_NONASCII
						lber_bprint( vals[i]->bv_val,
						    vals[i]->bv_len );
#endif /* BPRINT_NONASCII */
						continue;
					}
					printf( "\t\t\tlength (%ld) %s\n",
					    vals[i]->bv_len, vals[i]->bv_val );
				}
				ber_bvecfree( vals );
			}
		}
	}

	if ( res->lm_msgtype == LDAP_RES_SEARCH_RESULT
	    || res->lm_chain != NULLMSG )
		print_ldap_result( ld, res, "search" );
}


#ifdef WINSOCK
void
ldap_perror( LDAP *ld, char *s )
{
	char	*errs;

	if ( ld == NULL ) {
		perror( s );
		return;
	}

	errs = ldap_err2string( ld->ld_errno );
	printf( "%s: %s\n", s, errs == NULL ? "unknown error" : errs );
	if ( ld->ld_error != NULL && *ld->ld_error != '\0' ) {
		printf( "%s: additional info: %s\n", s, ld->ld_error );
	}
}
#endif /* WINSOCK */
