/*
 *  Copyright (c) 1993 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  cache.c - local caching support for LDAP
 */

#ifndef lint 
static char copyright[] = "@(#) Copyright (c) 1993 The Regents of the University of Michigan.\nAll rights reserved.\n";
#endif

#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <string.h>
#ifdef MACOS
#include <stdlib.h>
#include <time.h>
#include "macos.h"
#else /* MACOS */
#ifdef DOS
#include <malloc.h>
#include "msdos.h"
#ifdef NCSA
#include "externs.h"
#endif /* NCSA */
#ifdef WINSOCK
#include <time.h>
#endif /* WINSOCK */
#else /* DOS */
#include <sys/types.h>
#endif /* DOS */
#endif /* MACOS */

#ifdef NEEDPROTOS
static int		cache_hash( BerElement *ber );
static LDAPMessage	*msg_dup( LDAPMessage *msg );
static int		request_cmp( BerElement	*req1, BerElement *req2 );
static int		chain_contains_dn( LDAPMessage *msg, char *dn );
static long		msg_size( LDAPMessage *msg );
static void		check_cache_memused( LDAPCache *lc );
#else /* NEEDPROTOS */
static int		cache_hash();
static LDAPMessage	*msg_dup();
static int		request_cmp();
static int		chain_contains_dn();
static long		msg_size();
static void		check_cache_memused();
#endif /* NEEDPROTOS */


int
ldap_enable_cache( ld, timeout, maxmem )
LDAP		*ld;
long		timeout;
long		maxmem;
{
	if ( ld->ld_cache == NULLLDCACHE ) {
		if (( ld->ld_cache = (LDAPCache *)malloc( sizeof( LDAPCache )))
		    == NULLLDCACHE ) {
			return( -1 );
		}
		(void) memset( ld->ld_cache, 0, sizeof( LDAPCache ));
		ld->ld_cache->lc_memused = sizeof( LDAPCache );
	}

	ld->ld_cache->lc_timeout = timeout;
	ld->ld_cache->lc_maxmem = maxmem;
	check_cache_memused( ld->ld_cache );
	ld->ld_cache->lc_enabled = 1;
	return( 0 );
}


void
ldap_disable_cache( ld )
LDAP		*ld;
{
	if ( ld->ld_cache != NULLLDCACHE ) {
		ld->ld_cache->lc_enabled = 0;
	}
}


void
ldap_destroy_cache( ld )
LDAP		*ld;
{
	if ( ld->ld_cache != NULLLDCACHE ) {
		ldap_flush_cache( ld );
		free( (char *)ld->ld_cache );
		ld->ld_cache = NULLLDCACHE;
	}
}


void
ldap_flush_cache( ld )
LDAP		*ld;
{
	int		i;
	LDAPMessage	*m, *next;

	Debug( LDAP_DEBUG_TRACE, "ldap_flush_cache\n", 0, 0, 0 );

	if ( ld->ld_cache != NULLLDCACHE ) {
		/* delete all requests in the queue */
		for ( m = ld->ld_cache->lc_requests; m != NULLMSG; m = next ) {
			next = m->lm_next;
			ldap_msgfree( m );
		}
		ld->ld_cache->lc_requests = NULLMSG;

		/* delete all messages in the cache */
		for ( i = 0; i < LDAP_CACHE_BUCKETS; ++i ) {
			for ( m = ld->ld_cache->lc_buckets[ i ].lm_next;
			    m != NULLMSG; m = next ) {
				next = m->lm_next;
				ldap_msgfree( m );
			}
			ld->ld_cache->lc_buckets[ i ].lm_next = NULLMSG;
		}
		ld->ld_cache->lc_memused = sizeof( LDAPCache );
	}
}


void
ldap_uncache_entry( ld, dn )
LDAP		*ld;
char		*dn;
{
	int		i;
	LDAPMessage	*m, *prev, *next;

	Debug( LDAP_DEBUG_TRACE, "ldap_uncache_entry %s ld_cache %x\n",
	    dn, ld->ld_cache, 0 );

	if ( ld->ld_cache == NULLLDCACHE ) {
	    return;
	}

	/* first check the request queue */
	prev = NULLMSG;
	for ( m = ld->ld_cache->lc_requests; m != NULLMSG; m = next ) {
		next = m->lm_next;
		if ( chain_contains_dn( m, dn )) {
			if ( prev == NULLMSG ) {
				ld->ld_cache->lc_requests = next;
			} else {
				prev->lm_next = next;
			}
			ld->ld_cache->lc_memused -= msg_size( m );
			ldap_msgfree( m );
		} else {
			prev = m;
		}
	}

	/* now check the rest of the cache */
	for ( i = 0; i < LDAP_CACHE_BUCKETS; ++i ) {
		prev = NULLMSG;
		for ( m = ld->ld_cache->lc_buckets[ i ].lm_next; m != NULLMSG;
		    m = next ) {
			next = m->lm_next;
			if ( chain_contains_dn( m, dn )) {
				if ( prev == NULLMSG ) {
					ld->ld_cache->lc_buckets[ i ].lm_next
					    = next;
				} else {
					prev->lm_next = next;
				}
				ld->ld_cache->lc_memused -= msg_size( m );
				ldap_msgfree( m );
			} else {
				prev = m;
			}
		}
	}
}


void
add_request_to_cache( ld, msgtype, request )
LDAP		*ld;
int		msgtype;
BerElement	*request;
{
	LDAPMessage	*new;
	long		len;

	Debug( LDAP_DEBUG_TRACE, "add_request_to_cache\n", 0, 0, 0 );

	if ( ld->ld_cache == NULLLDCACHE ||
	    ( ld->ld_cache->lc_enabled == 0 )) {
		return;
	}


	if (( new = (LDAPMessage *) calloc( 1, sizeof(LDAPMessage) ))
	    != NULL ) {
		if (( new->lm_ber = ber_alloc()) == NULLBER ) {
			free( (char *)new );
			return;
		}
		len = request->ber_ptr - request->ber_buf;
		if (( new->lm_ber->ber_buf = (char *) malloc( (size_t)len ))
		    == NULL ) {
			ber_free( new->lm_ber, 0 );
			free( (char *)new );
			return;
		}
		(void) memcpy( new->lm_ber->ber_buf, request->ber_buf,
		    (size_t)len );
		new->lm_ber->ber_ptr = new->lm_ber->ber_buf;
		new->lm_ber->ber_end = new->lm_ber->ber_buf + len;
		new->lm_msgid = ld->ld_msgid;
		new->lm_msgtype = msgtype;;
	}

	new->lm_next = ld->ld_cache->lc_requests;
	ld->ld_cache->lc_requests = new;
}


void
add_result_to_cache( ld, result )
LDAP		*ld;
LDAPMessage	*result;
{
	LDAPMessage	*m, *req, *new, *prev;

	Debug( LDAP_DEBUG_TRACE, "add_result_to_cache\n", 0, 0, 0 );

	if ( ld->ld_cache == NULLLDCACHE ||
	    ( ld->ld_cache->lc_enabled == 0 )) {
		Debug( LDAP_DEBUG_TRACE, "add_result_to_cache DISABLED\n", 0, 0, 0 );
		return;
	}

	if ( result->lm_msgtype != LDAP_RES_SEARCH_ENTRY &&
	    result->lm_msgtype != LDAP_RES_SEARCH_RESULT &&
	    result->lm_msgtype != LDAP_RES_COMPARE ) {
		/*
		 * only cache search and compare operations
		 */
		Debug( LDAP_DEBUG_TRACE,
		    "add_result_to_cache NOT SEARCH OR COMPARE\n", 0, 0, 0 );
		return;
	}

	/*
	 * if corresponding request is in the lc_requests list, add this
	 * result to it.  if this result completes the results for the
	 * request, add the request/result chain to the cache proper.
	 */
	prev = NULLMSG;
	for ( m = ld->ld_cache->lc_requests; m != NULL; m = m->lm_next ) {
		if ( m->lm_msgid == result->lm_msgid ) {
			break;
		}
		prev = m;
	}

	if ( m != NULLMSG ) {	/* found request; add to end of chain */
		req = m;
		for ( ; m->lm_chain != NULLMSG; m = m->lm_chain )
			;
		if (( new = msg_dup( result )) != NULLMSG ) {
			new->lm_chain = NULLMSG;
			m->lm_chain = new;
			Debug( LDAP_DEBUG_TRACE,
			    "add_result_to_cache MSGID %d  TYPE %d\n",
			    result->lm_msgid, result->lm_msgtype, 0 );
		}
		if ( result->lm_msgtype == LDAP_RES_SEARCH_RESULT ||
		    result->lm_msgtype == LDAP_RES_COMPARE ) {
			m = &ld->ld_cache->lc_buckets[
			    cache_hash( req->lm_ber ) ];
			if ( prev == NULLMSG ) {
				ld->ld_cache->lc_requests = req->lm_next;
			} else {
				prev->lm_next = req->lm_next;
			}
			req->lm_next = m->lm_next;
			m->lm_next = req;
			time( &req->lm_time );
			ld->ld_cache->lc_memused += msg_size( req );
			check_cache_memused( ld->ld_cache );
		}
	} else {
		Debug( LDAP_DEBUG_TRACE,
		    "add_result_to_cache didn't find msgid %d\n",
		    result->lm_msgid, 0, 0 );
	}
}


/*
 * look in the cache for this request
 * return 0 if found, -1 if not
 * if found, the corresponding result messages are added to the incoming
 * queue with the correct (new) msgid so that subsequent ldap_result calls
 * will find them.
 */
int
check_cache( ld, msgtype, request )
LDAP		*ld;
int		msgtype;
BerElement	*request;
{
	LDAPMessage	*m, *new, *prev, *next;
	BerElement	reqber;
	int		first, hash;
	unsigned long	validtime;

	Debug( LDAP_DEBUG_TRACE, "check_cache\n", 0, 0, 0 );

	if ( ld->ld_cache == NULLLDCACHE ||
	    ( ld->ld_cache->lc_enabled == 0 )) {
		return( -1 );
	}

	reqber.ber_buf = reqber.ber_ptr = request->ber_buf;
	reqber.ber_end = request->ber_ptr;

	time( &validtime );
	validtime -= ld->ld_cache->lc_timeout;

	prev = NULLMSG;
	hash = cache_hash( &reqber );
	for ( m = ld->ld_cache->lc_buckets[ hash ].lm_next; m != NULLMSG;
	    m = next ) {
		Debug( LDAP_DEBUG_TRACE,"check_cache examining id %d,type %d\n",
		    m->lm_msgid, m->lm_msgtype, 0 );
		if ( m->lm_time < validtime ) {
			/* delete expired message */
			next = m->lm_next;
			if ( prev == NULL ) {
				ld->ld_cache->lc_buckets[ hash ].lm_next = next;
			} else {
				prev->lm_next = next;
			}
			Debug( LDAP_DEBUG_TRACE, "check_cache expired id %d\n",
			    m->lm_msgid, 0, 0 );
			ld->ld_cache->lc_memused -= msg_size( m );
			ldap_msgfree( m );
		} else {
		    if ( m->lm_msgtype == msgtype &&
			request_cmp( m->lm_ber, &reqber ) == 0 ) {
			    break;
		    }
		    next = m->lm_next;
		    prev = m;
		}
	}

	if ( m == NULLMSG ) {
		return( -1 );
	}

	/*
	 * add duplicates of responses to incoming queue
	 */
	first = 1;
	for ( m = m->lm_chain; m != NULLMSG; m = m->lm_chain ) {
		if (( new = msg_dup( m )) == NULLMSG ) {
			return( -1 );
		}

		new->lm_msgid = ld->ld_msgid;
		new->lm_chain = NULLMSG;
		if ( first ) {
			new->lm_next = ld->ld_responses;
			ld->ld_responses = new;
			first = 0;
		} else {
			prev->lm_chain = new;
		}
		prev = new;
		Debug( LDAP_DEBUG_TRACE, "check_cache added type %d\n",
		    new->lm_msgtype, 0, 0 );
	}

	Debug( LDAP_DEBUG_TRACE, "check_cache OK\n", 0, 0, 0 );
	return( 0 );
}


static int
cache_hash( ber )
BerElement	*ber;
{
	BerElement	bercpy;
	unsigned long	len;

	/*
         * just take the length of the packet and mod with # of buckets
	 */
	bercpy = *ber;
	if ( ber_skip_tag( &bercpy, &len ) == -1
		|| ber_scanf( &bercpy, "x" ) == -1 ) {
	    len = 0;	/* punt: just return zero */
	} else {
	    len = bercpy.ber_end - bercpy.ber_ptr;
	}

	Debug( LDAP_DEBUG_TRACE, "cache_hash: len is %d, returning %d\n",
	    len, len % LDAP_CACHE_BUCKETS, 0 );
	return( len % LDAP_CACHE_BUCKETS );
}


static LDAPMessage *
msg_dup( msg )
LDAPMessage	*msg;
{
	LDAPMessage	*new;
	long		len;

	if (( new = (LDAPMessage *)malloc( sizeof(LDAPMessage))) != NULL ) {
		*new = *msg;	/* struct copy */
		if (( new->lm_ber = ber_dup( msg->lm_ber )) == NULLBER ) {
			free( (char *)new );
			return( NULLMSG );
		}
		len = msg->lm_ber->ber_end - msg->lm_ber->ber_buf;
		if (( new->lm_ber->ber_buf = (char *) malloc(
		    (size_t)len )) == NULL ) {
			ber_free( new->lm_ber, 0 );
			free( (char *)new );
			return( NULLMSG );
		}
		(void) memcpy( new->lm_ber->ber_buf, msg->lm_ber->ber_buf,
		    (size_t)len );
		new->lm_ber->ber_ptr = new->lm_ber->ber_buf +
			( msg->lm_ber->ber_ptr - msg->lm_ber->ber_buf );
		new->lm_ber->ber_end = new->lm_ber->ber_buf + len;
	}

	return( new );
}


static int
request_cmp( req1, req2 )
BerElement	*req1, *req2;
{
	unsigned long	len;
	BerElement	r1, r2;

	r1 = *req1;	/* struct copies */
	r2 = *req2;

	/*
	 * skip the enclosing tags (sequence markers) and the msg ids
	 */
	if ( ber_skip_tag( &r1, &len ) == -1 || ber_scanf( &r1, "x" ) == -1 ) {
	    return( -1 );
	}
	if ( ber_skip_tag( &r2, &len ) == -1 || ber_scanf( &r2, "x" ) == -1 ) {
	    return( -1 );
	}

	/*
	 * check remaining length and bytes if necessary
	 */
	if (( len = r1.ber_end - r1.ber_ptr ) != r2.ber_end - r2.ber_ptr ) {
		return( -1 );	/* different lengths */
	}
	return( memcmp( r1.ber_ptr, r2.ber_ptr, (size_t)len ));
}	


static int
chain_contains_dn( msg, dn )
LDAPMessage	*msg;
char		*dn;
{
	LDAPMessage	*m;
	BerElement	ber;
	long		msgid;
	char		*s;
	int		rc;


	/*
	 * first check the base or dn of the request
	 */
	ber = *msg->lm_ber;	/* struct copy */
	if ( ber_scanf( &ber, "{i{{a", &msgid, &s ) != -1 ) {
	    rc = ( strcasecmp( dn, s ) == 0 ) ? 1 : 0;
	    free( s );
	    if ( rc != 0 ) {
		return( rc );
	    }
	}

	if ( msg->lm_msgtype == LDAP_REQ_COMPARE ) {
		return( 0 );
	}

	/*
	 * now check the dn of each search result
	 */
	rc = 0;
	for ( m = msg->lm_chain; m != NULLMSG && rc == 0 ; m = m->lm_chain ) {
		if ( m->lm_msgtype != LDAP_RES_SEARCH_ENTRY ) {
			continue;
		}
		ber = *m->lm_ber;	/* struct copy */
		if ( ber_scanf( &ber, "i{{a", &msgid, &s ) != -1 ) {
			rc = ( strcasecmp( dn, s ) == 0 ) ? 1 : 0;
			free( s );
		}
	}

	return( rc );
}


static long
msg_size( msg )
LDAPMessage	*msg;
{
	LDAPMessage	*m;
	long		size;

	size = 0;
	for ( m = msg; m != NULLMSG; m = m->lm_chain ) {
		size += ( sizeof( LDAPMessage ) + m->lm_ber->ber_end -
		    m->lm_ber->ber_buf );
	}

	return( size );
}


#define THRESHOLD_FACTOR	3 / 4
#define SIZE_FACTOR		2 / 3

static void
check_cache_memused( lc )
LDAPCache	*lc;
{
/*
 * this routine is called to check if the cache is too big (lc_maxmem >
 * minimum cache size and lc_memused > lc_maxmem).  If too big, it reduces
 * the cache size to < SIZE_FACTOR * lc_maxmem. The algorithm is as follows:
 *    remove_threshold = lc_timeout seconds;
 *    do {
 *        remove everything older than remove_threshold seconds;
 *        remove_threshold = remove_threshold * THRESHOLD_FACTOR;
 *    } while ( cache size is > SIZE_FACTOR * lc_maxmem )
 */
	int		i;
	unsigned long	remove_threshold, validtime;
	LDAPMessage	*m, *prev, *next;

	Debug( LDAP_DEBUG_TRACE, "check_cache_memused %d/%d\n", lc->lc_memused,
	    lc->lc_maxmem, 0 );

	if ( lc->lc_maxmem <= sizeof( LDAPCache )
	    || lc->lc_memused < lc->lc_maxmem ) {
		return;
	}

	remove_threshold = lc->lc_timeout;
	while ( lc->lc_memused > lc->lc_maxmem * SIZE_FACTOR ) {
		time( &validtime );
		validtime -= remove_threshold;
		for ( i = 0; i < LDAP_CACHE_BUCKETS; ++i ) {
			prev = NULLMSG;
			for ( m = lc->lc_buckets[ i ].lm_next; m != NULLMSG;
			    m = next ) {
				next = m->lm_next;
				if ( m->lm_time < validtime ) {
					if ( prev == NULLMSG ) {
						lc->lc_buckets[ i ].lm_next
						    = next;
					} else {
						prev->lm_next = next;
					}
					lc->lc_memused -= msg_size( m );
					Debug( LDAP_DEBUG_TRACE,
					    "check_cache_memused remove %d\n",
					    m->lm_msgid, 0, 0 );
					ldap_msgfree( m );
				} else {
					prev = m;
				}
			}
		}
		remove_threshold *= THRESHOLD_FACTOR;
	}

	Debug( LDAP_DEBUG_TRACE, "check_cache_memused reduced to %d/%d\n",
	    lc->lc_memused, lc->lc_maxmem, 0 );
}
