/* io.c - ber general i/o routines */
/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <ctype.h>
#ifdef MACOS
#include <stdlib.h>
#include "macos.h"
#else /* MACOS */
#ifdef NeXT
#include <stdlib.h>
#else /* NeXT */
#include <malloc.h>
#endif /* NeXT */
#include <sys/types.h>
#include <netinet/in.h>
#ifdef PCNFS
#include <tklib.h>
#endif /* PCNFS */
#endif /* MACOS */

#include <memory.h>
#include <string.h>
#include "lber.h"

#ifdef DOS
#include "msdos.h"
#endif /* DOS */

#ifdef NEEDPROTOS
static int ber_realloc( BerElement *ber, unsigned long len );
static int ber_filbuf( Sockbuf *sb );
static long BerRead( Sockbuf *sb, char *buf, long len );
#ifdef PCNFS
static int BerWrite( Sockbuf *sb, char *buf, long len );
#endif /* PCNFS */
#else
int ber_filbuf();
long BerRead();
static int ber_realloc();
#endif /* NEEDPROTOS */

#define bergetc( sb )    ( sb->sb_ber.ber_end > sb->sb_ber.ber_ptr ? \
			  (unsigned char)*sb->sb_ber.ber_ptr++ : \
			  ber_filbuf( sb ))

#ifdef MACOS
#define read( s, b, l ) tcpread( s, 0, (unsigned char *)b, l, NULL )
#define BerWrite( sb, b, l )   tcpwrite( sb->sb_sd, (unsigned char *)(b), l )
#else /* MACOS */
#ifdef DOS
#ifdef PCNFS
#define BerWrite( s, b, l ) send( s->sb_sd, b, (int) l, 0 )
#endif /* PCNFS */
#ifdef NCSA
#define read( s, b, l ) nread( s, b, l )
#define BerWrite( s, b, l ) netwrite( s->sb_sd, b, l )
#endif /* NCSA */
#ifdef WINSOCK
#define read( s, b, l ) recv( s, b, l, 0 )
#define BerWrite( s, b, l ) send( s->sb_sd, b, l, 0 )
#endif /* WINSOCK */
#else /* DOS */
#define BerWrite( sb, b, l )	write( sb->sb_sd, b, l )
#endif /* DOS */
#endif /* MACOS */

#define EXBUFSIZ	1024

int
ber_filbuf( sb )
Sockbuf	*sb;
{
	short	rc;

	if ( sb->sb_ber.ber_buf == NULL ) {
		if ( (sb->sb_ber.ber_buf = (char *) malloc( READBUFSIZ )) ==
		    NULL )
			return( -1 );
		sb->sb_ber.ber_ptr = sb->sb_ber.ber_buf;
		sb->sb_ber.ber_end = sb->sb_ber.ber_buf;
	}

#ifdef PCNFS
	if ( (rc = recv( sb->sb_sd, sb->sb_ber.ber_buf, READBUFSIZ, 0)) > 0 ) {
#else /* PCNFS */
	if ( (rc = read( sb->sb_sd, sb->sb_ber.ber_buf, READBUFSIZ )) > 0 ) {
#endif /* PCNFS */
		sb->sb_ber.ber_ptr = sb->sb_ber.ber_buf + 1;
		sb->sb_ber.ber_end = sb->sb_ber.ber_buf + rc;
		return( (unsigned char)*sb->sb_ber.ber_buf );
	}

	return( -1 );
}


long
BerRead( sb, buf, len )
Sockbuf	*sb;
char	*buf;
long	len;
{
	int	c;
	long	nread = 0;

	while ( len-- > 0 ) {
		if ( (c = bergetc( sb )) < 0 ) {
			return( c );
		}
		*buf++ = c;
		nread++;
	}

	return( nread );
}


long
ber_read( ber, buf, len )
BerElement	*ber;
char		*buf;
unsigned long	len;
{
	long	actuallen, nleft;

	nleft = ber->ber_end - ber->ber_ptr;
	actuallen = nleft < len ? nleft : len;

	(void) memcpy( buf, ber->ber_ptr, (size_t)actuallen );

	ber->ber_ptr += actuallen;

	return( actuallen );
}

long
ber_write( ber, buf, len, nosos )
BerElement	*ber;
char		*buf;
unsigned long	len;
int		nosos;
{
	if ( nosos || ber->ber_sos == NULL ) {
		if ( ber->ber_ptr + len > ber->ber_end ) {
			if ( ber_realloc( ber, len ) != 0 )
				return( -1 );
		}
		(void) memcpy( ber->ber_ptr, buf, (size_t)len );
		ber->ber_ptr += len;
		return( len );
	} else {
		if ( ber->ber_sos->sos_ptr + len > ber->ber_end ) {
			if ( ber_realloc( ber, len ) != 0 )
				return( -1 );
		}
		(void) memcpy( ber->ber_sos->sos_ptr, buf, (size_t)len );
		ber->ber_sos->sos_ptr += len;
		ber->ber_sos->sos_clen += len;
		return( len );
	}
}

static ber_realloc( ber, len )
BerElement	*ber;
unsigned long	len;
{
	unsigned long	need, have, total;
	Seqorset	*s;
	long		off;
	char		*oldbuf;

	have = (ber->ber_end - ber->ber_buf) / EXBUFSIZ;
	need = (len < EXBUFSIZ ? 1 : (len + (EXBUFSIZ - 1)) / EXBUFSIZ);
	total = have * EXBUFSIZ + need * EXBUFSIZ;

	oldbuf = ber->ber_buf;

	if ( ber->ber_buf == NULL ) {
		if ( (ber->ber_buf = (char *) malloc( (size_t)total )) == NULL )
			return( -1 );
	} else if ( (ber->ber_buf = (char *) realloc( ber->ber_buf,
	    (size_t)total )) == NULL )
		return( -1 );

	ber->ber_end = ber->ber_buf + total;

	/*
	 * If the stinking thing was moved, we need to go through and
	 * reset all the sos and ber pointers.  Offsets would've been
	 * a better idea... oh well.
	 */

	if ( ber->ber_buf != oldbuf ) {
		ber->ber_ptr = ber->ber_buf + (ber->ber_ptr - oldbuf);

		for ( s = ber->ber_sos; s != NULLSEQORSET; s = s->sos_next ) {
			off = s->sos_first - oldbuf;
			s->sos_first = ber->ber_buf + off;

			off = s->sos_ptr - oldbuf;
			s->sos_ptr = ber->ber_buf + off;
		}
	}

	return( 0 );
}

void ber_free( ber, freebuf )
BerElement	*ber;
int		freebuf;
{
	if ( freebuf && ber->ber_buf != NULL )
		free( ber->ber_buf );
	free( (char *) ber );
}

ber_flush( sb, ber, freeit )
Sockbuf		*sb;
BerElement	*ber;
int		freeit;
{
	int	rc;
	long	nwritten, towrite;

	towrite = ber->ber_ptr - ber->ber_buf;

#ifdef LDAP_DEBUG
	if ( lber_debug ) {
		fprintf( stderr, "ber_flush: %d bytes\n", towrite );
		if ( lber_debug > 1 )
			lber_bprint( ber->ber_buf, towrite );
	}
#endif

	nwritten = 0;
	do {
		if ( (rc = BerWrite( sb, ber->ber_buf + nwritten,
		    (size_t)towrite )) <= 0 )
			return( -1 );
		towrite -= rc;
		nwritten += rc;
	} while ( towrite > 0 );

	if ( freeit )
		ber_free( ber, 1 );

	return( 0 );
}

BerElement *ber_alloc()
{
	BerElement	*ber;

	if ( (ber = (BerElement *) calloc( 1, sizeof(BerElement) )) == NULLBER )
		return( NULLBER );
	ber->ber_tag = LBER_DEFAULT;

	return( ber );
}

BerElement *ber_dup( ber )
BerElement	*ber;
{
	BerElement	*new;

	if ( (new = ber_alloc()) == NULLBER )
		return( NULLBER );

	*new = *ber;

	return( new );
}

#ifdef LDAP_DEBUG

ber_dump( ber, inout )
BerElement	*ber;
int		inout;
{
	fprintf( stderr, "ber_dump: buf 0x%lx, ptr 0x%lx, end 0x%lx\n",
	    ber->ber_buf, ber->ber_ptr, ber->ber_end );
	if ( inout == 1 ) {
		fprintf( stderr, "          current len %ld, contents:\n",
		    ber->ber_end - ber->ber_ptr );
		lber_bprint( ber->ber_ptr, ber->ber_end - ber->ber_ptr );
	} else {
		fprintf( stderr, "          current len %ld, contents:\n",
		    ber->ber_ptr - ber->ber_buf );
		lber_bprint( ber->ber_buf, ber->ber_ptr - ber->ber_buf );
	}
}

ber_sos_dump( sos )
Seqorset	*sos;
{
	fprintf( stderr, "*** sos dump ***\n" );
	while ( sos != NULLSEQORSET ) {
		fprintf( stderr, "ber_sos_dump: clen %ld first 0x%lx ptr 0x%lx\n",
		    sos->sos_clen, sos->sos_first, sos->sos_ptr );
		fprintf( stderr, "              current len %ld contents:\n",
		    sos->sos_ptr - sos->sos_first );
		lber_bprint( sos->sos_first, sos->sos_ptr - sos->sos_first );

		sos = sos->sos_next;
	}
	fprintf( stderr, "*** end dump ***\n" );
}

#endif

ber_get_next( sb, len, ber )
Sockbuf		*sb;
unsigned long	*len;
BerElement	*ber;
{
	unsigned char	tag;
	unsigned char	lc;
	int		noctets, diff;
	unsigned long	netlen;

#ifdef LDAP_DEBUG
	if ( lber_debug )
		fprintf( stderr, "ber_get_next\n" );
#endif

	/*
	 * Any ber element looks like this: tag length contents.
	 * Assuming everything's ok, we return the tag byte (we
	 * can assume a single byte), return the length in len,
	 * and the rest of the undecoded element in buf.
	 *
	 * Assumptions:
	 *	1) small tags (less than 128)
	 *	2) definite lengths
	 *	3) primitive encodings used whenever possible
	 */

	/*
	 * First, we read the tag.
	 */

	if ( BerRead( sb, (char *) &tag, 1 ) != 1 )
		return( -1 );

	/*
	 * Next, read the length.  The first byte contains the length of
	 * the length.  If bit 8 is set, the length is the long form,
	 * otherwise it's the short form.  We don't allow a length that's
	 * greater than what we can hold in an unsigned long.
	 */

	*len = netlen = 0;
	if ( BerRead( sb, (char *) &lc, 1 ) != 1 )
		return( -1 );
	if ( lc & 0x80 ) {
		noctets = (lc & 0x7f);
		if ( noctets > sizeof(unsigned long) )
			return( -1 );
		diff = sizeof(unsigned long) - noctets;
		if ( BerRead( sb, (char *) &netlen + diff, noctets ) !=
		    noctets )
			return( -1 );
		*len = ntohl( netlen );
	} else {
		*len = lc;
	}

	/*
	 * Finally, malloc a buffer for the contents and read it in.
	 * It's this buffer that's passed to all the other ber decoding
	 * routines.
	 */

	if ( (ber->ber_buf = (char *) malloc( (size_t)*len )) == NULL )
		return( -1 );
	ber->ber_ptr = ber->ber_buf;
	ber->ber_end = ber->ber_buf + *len;

	if ( BerRead( sb, ber->ber_buf, *len ) <= 0 )
		return( -1 );

#ifdef LDAP_DEBUG
	if ( lber_debug ) {
		fprintf( stderr, "ber_get_next: tag 0x%x len %ld contents:\n",
		    tag, *len );
		if ( lber_debug > 1 )
			ber_dump( ber, 1 );
	}
#endif

	return( tag );
}


#ifndef NO_USERINTERFACE
/*
 * Print arbitrary stuff, for debugging.
 */

#define BPLEN	48

lber_bprint( data, len )
    char	*data;
    int		len;
{
    static char	hexdig[] = "0123456789abcdef";
    char	out[ BPLEN ];
    int		i = 0;

    memset( out, 0, BPLEN );
    for ( ;; ) {
	if ( len < 1 ) {
	    fprintf( stderr, "\t%s\n", ( i == 0 ) ? "(end)" : out );
	    break;
	}

#ifndef HEX
	if ( isgraph( (unsigned char)*data )) {
	    out[ i ] = ' ';
	    out[ i+1 ] = *data;
	} else {
#endif
	    out[ i ] = hexdig[ ( *data & 0xf0 ) >> 4 ];
	    out[ i+1 ] = hexdig[ *data & 0x0f ];
#ifndef HEX
	}
#endif
	i += 2;
	len--;
	data++;

	if ( i > BPLEN - 2 ) {
	    fprintf( stderr, "\t%s\n", out );
	    memset( out, 0, BPLEN );
	    i = 0;
	    continue;
	}
	out[ i++ ] = ' ';
    }
}
#endif /* NO_USERINTERFACE */
