/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/syslog.h>
#include <sys/wait.h>
#include <signal.h>

/*
 *************************************************************************
 * If you are not a U-M site, you probably want to tailor the following:
 *************************************************************************
 */

#define DAPUSER			"cn=Go500,ou=Miscellaneous Servers,o=University of Michigan,c=US"
#define DEFAULT_BASE		"ou=People,o=University of Michigan,c=US"
#define DEFAULT_FILTERFILE	"/usr/local/etc/ldapfilter.conf"

/*
 * only set and uncomment this if your hostname() does not return a
 * fully qualified hostname
 */

/* #define HOSTNAME	"fully.qualified.hostname.here" */

/*
 *************************************************************************
 * The rest of this stuff probably does not need to be changed
 *************************************************************************
 */

#define DEFAULT_LDAPHOST	"localhost"
#define DEFAULT_GO500PORT	5555

#ifndef FD_SET
#define NFDBITS         32
#define FD_SETSIZE      32
#define FD_SET(n, p)    ((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define FD_CLR(n, p)    ((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define FD_ISSET(n, p)  ((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#define FD_ZERO(p)      bzero((char *)(p), sizeof(*(p)))
#endif

int	debug;
int	dosyslog;
int	inetd;

char	*ldaphost = DEFAULT_LDAPHOST;
char	*base = DEFAULT_BASE;
int	strip = 3;
int	searchaliases = 1;
char	*filterfile = DEFAULT_FILTERFILE;

char	myhost[MAXHOSTNAMELEN];
int	myport;

static set_socket();
static wait4child();
static do_queries();
static do_error();
static do_search();
static do_read();

static usage( name )
char	*name;
{
	fprintf( stderr, "usage: %s [-d debuglevel] [-p port] [-l] [-x ldaphost] [-b searchbase] [-a]\n" );
	exit( 1 );
}

main (argc, argv)
int	argc;
char	**argv;
{
	int			s, ns, rc;
	int			port = -1;
	int			tblsize;
	int			i, pid;
	char			*myname;
	fd_set			readfds;
	struct hostent		*hp;
	struct sockaddr_in	from;
	int			fromlen;
	int			wait4child();
	extern char		*optarg;
	extern char		**Argv;
	extern int		Argc;

	/* for setproctitle */
        Argv = argv;
        Argc = argc;

	while ( (i = getopt( argc, argv, "ab:d:f:lp:s:x:I" )) != EOF ) {
		switch( i ) {
		case 'a':
			searchaliases = 0;
			break;

		case 'b':
			base = strdup( optarg );
			break;

		case 'd':
			debug = atoi( optarg );
			break;

		case 'f':
			filterfile = strdup( optarg );
			break;

		case 'l':
			dosyslog = 1;
			break;

		case 'p':
			port = atoi( optarg );
			break;

		case 's':
			strip = atoi( optarg );
			break;

		case 'x':
			ldaphost = strdup( optarg );
			break;

		case 'I':
			inetd = 1;
			break;

		default:
			usage( argv[0] );
		}
	}

#ifdef HOSTNAME
	strcpy( myhost, HOSTNAME );
#else
	if ( myhost[0] == '\0' && gethostname( myhost, sizeof(myhost) )
	    == -1 ) {
		perror( "gethostname" );
		exit( 1 );
	}
#endif

	/* detach if stderr is redirected or no debugging */
	if ( inetd == 0 )
		(void) detach( debug );

	if ( (myname = strrchr( argv[0], '/' )) == NULL )
		myname = strdup( argv[0] );
	else
		myname = strdup( myname + 1 );

	if ( dosyslog && openlog( myname, LOG_PID | LOG_NOWAIT, LOG_LOCAL3 )
	    < 0 ) {
		perror("openlog");
	}
	if ( dosyslog )
		syslog( LOG_INFO, "initializing" );

	/* set up the socket to listen on */
	if ( inetd == 0 ) {
		s = set_socket( port );

		/* arrange to reap children */
		(void) signal( SIGCHLD, wait4child );
	} else {
		myport = DEFAULT_GO500PORT;

		fromlen = sizeof(from);
		if ( getpeername( 0, (struct sockaddr *) &from, &fromlen )
		    == 0 ) {
			hp = gethostbyaddr( (char *) &(from.sin_addr.s_addr),
			    sizeof(from.sin_addr.s_addr), AF_INET );
			Debug( LDAP_DEBUG_ARGS, "connection from %s (%s)\n",
			    (hp == NULL) ? "unknown" : hp->h_name,
			    inet_ntoa( from.sin_addr ), 0 );

			if ( dosyslog ) {
				syslog( LOG_INFO, "connection from %s (%s)",
				    (hp == NULL) ? "unknown" : hp->h_name,
				    inet_ntoa( from.sin_addr ) );
			}

			setproctitle( hp == NULL ? inet_ntoa( from.sin_addr ) :
			    hp->h_name );
		}

		do_queries( 0 );

		exit( 0 );
	}

	tblsize = getdtablesize();
	for ( ;; ) {
		FD_ZERO( &readfds );
		FD_SET( s, &readfds );

		if ( (rc = select( tblsize, &readfds, 0, 0 ,0 )) == -1 ) {
			if ( debug ) perror( "select" );
			continue;
		} else if ( rc == 0 ) {
			continue;
		}

		if ( ! FD_ISSET( s, &readfds ) )
			continue;

		fromlen = sizeof(from);
		if ( (ns = accept( s, &from, &fromlen )) == -1 ) {
			if ( debug ) perror( "accept" );
			exit( 1 );
		}

		hp = gethostbyaddr( (char *) &(from.sin_addr.s_addr),
		    sizeof(from.sin_addr.s_addr), AF_INET );

		if ( dosyslog ) {
			syslog( LOG_INFO, "TCP connection from %s (%s)",
			    (hp == NULL) ? "unknown" : hp->h_name,
			    inet_ntoa( from.sin_addr ) );
		}

		switch( pid = fork() ) {
		case 0:		/* child */
			close( s );
			do_queries( ns );
			break;

		case -1:	/* failed */
			perror( "fork" );
			break;

		default:	/* parent */
			close( ns );
			if ( debug )
				fprintf( stderr, "forked child %d\n", pid );
			break;
		}
	}
	/* NOT REACHED */
}

static
set_socket( port )
int	port;
{
	int			s, one;
	struct sockaddr_in	addr;

	if ( port == -1 )
		port = DEFAULT_GO500PORT;
	myport = port;

	if ( (s = socket( AF_INET, SOCK_STREAM, 0 )) == -1 ) {
                perror( "socket" );
                exit( 1 );
        }

        /* set option so clients can't keep us from coming back up */
	one = 1;
        if ( setsockopt( s, SOL_SOCKET, SO_REUSEADDR, (char *) &one,
	    sizeof(one) ) < 0 ) {
                perror( "setsockopt" );
                exit( 1 );
        }

        /* bind to a name */
        addr.sin_family = AF_INET;
        addr.sin_addr.s_addr = INADDR_ANY;
        addr.sin_port = htons( port );
        if ( bind( s, &addr, sizeof(addr) ) ) {
                perror( "bind" );
                exit( 1 );
        }

	/* listen for connections */
        if ( listen( s, 5 ) == -1 ) {
                perror( "listen" );
                exit( 1 );
        }

        if ( debug ) printf("tcp socket allocated, bound, and listening\n");

	return( s );
}

static wait4child()
{
#if defined(SunOS) && SunOS < 40
        union wait status;
#else
        int     status;
#endif

        if ( debug ) printf( "parent: catching child status\n" );
        while ( wait3( &status, WNOHANG | WUNTRACED, 0 ) > 0 )
                ;       /* NULL */
}

static
do_queries( s )
int	s;
{
	char		buf[1024], *query;
	int		len;
	FILE		*fp;
	int		rc, tblsize;
	struct timeval	timeout;
	fd_set		readfds;
	LDAP		*ld;

	if ( (fp = fdopen( s, "a+")) == NULL ) {
		exit( 1 );
	}

	tblsize = getdtablesize();
	timeout.tv_sec = 300;
	timeout.tv_usec = 0;
	FD_ZERO( &readfds );
	FD_SET( fileno( fp ), &readfds );

	if ( (rc = select( tblsize, &readfds, 0, 0, &timeout )) <= 0 )
		exit( 1 );

	if ( fgets( buf, sizeof(buf), fp ) == NULL )
		exit( 1 );

	len = strlen( buf );
	if ( debug ) {
		fprintf( stderr, "got %d bytes\n", len );
#ifdef LDAP_DEBUG
		lber_bprint( buf, len );
#endif
	}

	/* strip of \r \n */
	if ( buf[len - 1] == '\n' )
		buf[len - 1] = '\0';
	len--;
	if ( buf[len - 1] == '\r' )
		buf[len - 1] = '\0';
	len--;

	query = buf;

	/* strip off leading white space */
	while ( isspace( *query )) {
		++query;
		--len;
	}

	rewind(fp);

	if ( *query != '~' && *query != '@' ) {
		if ( (ld = ldap_open( ldaphost, LDAP_PORT )) == NULL ) {
			fprintf(fp,
			    "0An error occurred (explanation)\t@%d\t%s\t%d\r\n",
			    LDAP_SERVER_DOWN, myhost, myport );
			fprintf( fp, ".\r\n" );
			rewind(fp);
			exit( 1 );
		}

		ld->ld_deref = LDAP_DEREF_ALWAYS;
		if ( !searchaliases )
			ld->ld_deref = LDAP_DEREF_FINDING;

		if ( (rc = ldap_simple_bind_s( ld, DAPUSER, NULL ))
		    != LDAP_SUCCESS ) {
			fprintf(fp,
			    "0An error occurred (explanation)\t@%d\t%s\t%d\r\n",
			    rc, myhost, myport );
			fprintf( fp, ".\r\n" );
			rewind(fp);
			exit( 1 );
		}
	}

	switch ( *query ) {
	case '~':
		fprintf( fp, "The query you specified was not specific enough, causing a size limit\r\n" );
		fprintf( fp, "to be exceeded and the first several matches found to be returned.\r\n" );
		fprintf( fp, "If you did not find the match you were looking for, try issuing a more\r\n" );
		fprintf( fp, "specific query, for example one that contains both first and last name.\r\n" );
		fprintf( fp, ".\r\n" );
		break;

	case '=':
		do_read( ld, fp, ++query );
		break;

	case '@':
		do_error( fp, ++query );
		break;

	default:
		do_search( ld, fp, query );
		break;
	}

	fprintf( fp, ".\r\n" );
	rewind(fp);

	exit( 1 );
	/* NOT REACHED */
}

static
do_error( fp, s )
FILE	*fp;
char	*s;
{
	int	code;

	code = atoi( s );

	fprintf( fp, "An error occurred searching X.500.  The error code was %d\r\n", code );
	fprintf( fp, "The corresponding error is: %s\r\n", ldap_err2string( code ) );
	fprintf( fp, "No additional information is available\r\n" );
	fprintf( fp, ".\r\n" );
}

static
do_search( ld, fp, buf )
LDAP	*ld;
FILE	*fp;
char	*buf;
{
	char		*dn, *rdn;
	char		**title;
	int		rc, matches = 0;
	struct timeval	tv;
	LDAPFiltInfo	*fi;
	LDAPFiltDesc	*filtd;
	LDAPMessage	*e, *res;
	static char	*attrs[] = { "title", 0 };

	if ( (filtd = ldap_init_getfilter( filterfile )) == NULL ) {
		fprintf( stderr, "Cannot open filter file (%s)\n", filterfile );
		exit( 1 );
	}

	tv.tv_sec = 60;
	tv.tv_usec = 0;
	for ( fi = ldap_getfirstfilter( filtd, "go500", buf ); fi != NULL;
	    fi = ldap_getnextfilter( filtd ) ) {
		if ( (rc = ldap_search_st( ld, base, LDAP_SCOPE_SUBTREE,
		    fi->lfi_filter, attrs, 0, &tv, &res )) != LDAP_SUCCESS
		    && rc != LDAP_SIZELIMIT_EXCEEDED ) {
			fprintf(fp,
			    "0An error occurred (explanation)\t@%d\t%s\t%d\r\n",
			    rc, myhost, myport );
			return;
		}

		if ( (matches = ldap_count_entries( ld, res )) != 0 )
			break;
	}

	if ( matches <= 0 ) {
		return;
	}

	for ( e = ldap_first_entry( ld, res ); e != NULL;
	    e = ldap_next_entry( ld, e ) ) {
		char	*s;

		dn = ldap_get_dn( ld, e );
		rdn = strdup( dn );
		if ( (s = strchr( rdn, ',' )) != NULL )
			*s = '\0';

		if ( (s = strchr( rdn, '=' )) == NULL )
			s = rdn;
		else
			++s;

		title = ldap_get_values( ld, e, "title" );

		if ( title != NULL ) {
			char	*p;

			for ( p = title[0]; *p; p++ ) {
				if ( *p == '/' )
					*p = '\\';
			}
		}

		fprintf( fp, "0%-20s    %s\t=%s\t%s\t%d\r\n", s,
		    title ? title[0] : "", dn, myhost, myport );

		if ( title != NULL )
			ldap_value_free( title );

		free( rdn );
		free( dn );
	}

	if ( ldap_result2error( ld, res, 1 ) == LDAP_SIZELIMIT_EXCEEDED ) {
		fprintf( fp, "0A size limit was exceeded (explanation)\t~\t%s\t%d\r\n",
		    myhost, myport );
	}
}

static
print_attr( ld, fp, label, attr, e, multiline )
LDAP		*ld;
FILE		*fp;
char		*label;
char		*attr;
LDAPMessage	*e;
int		multiline;
{
	char	**val;
	int	i, gotone = 0, firstline = 1;

	if ( (val = ldap_get_values( ld, e, attr )) == NULL )
		return;

	fprintf( fp, "%-19s\r\n", label );
	for ( i = 0; val[i] != NULL; i++ ) {
		if ( multiline ) {
			char	*s, *p;

			if ( gotone )
				fprintf( fp, "%-19s\r\n", label );
			p = s = val[i];
			while ( s = strchr( s, '$' ) ) {
				*s++ = '\0';
				while ( isspace( *s ) )
					s++;
				if ( firstline ) {
					fprintf( fp, "                    %s\r\n", p );
					firstline = 0;
				} else {
					fprintf( fp, "                    %s\r\n", p );
				}
				p = s;
			}
			if ( firstline ) {
				fprintf( fp, "                    %s\r\n", p );
			} else {
				fprintf( fp, "                    %s\r\n", p );
			}
			gotone = 1;
			firstline = 1;
		} else {
			if ( firstline ) {
				fprintf( fp, "                    %s\r\n", val[i] );
				firstline = 0;
			} else {
				fprintf( fp, "                    %s\r\n", val[i] );
			}
		}
	}
	ldap_value_free( val );
}

static
do_read( ld, fp, dn )
LDAP	*ld;
FILE	*fp;
char	*dn;
{
	char		*ufn;
	int		i;
	char		*s;
	char		**cn, **paddr, **tnumber, **uid, **title, **mail;
	struct timeval	tv;
	LDAPMessage	*res, *e;
	static char	*attrs[] = { "cn", "postalAddress", "uid", "title",
				     "mail", "telephoneNumber",
				     "facsimileTelephoneNumber", 0 };

	if ( ldap_search( ld, dn, LDAP_SCOPE_BASE, "objectclass=*", attrs, 0 )
	    == -1 ) {
		if ( debug ) ldap_perror( ld, "ldap_search" );
		exit( 1 );
	}
	tv.tv_sec = 60;
	tv.tv_usec = 0;
	if ( ldap_result( ld, LDAP_RES_ANY, 1, &tv, &res ) <= 0 ) {
		ldap_perror( ld, "ldap_result" );
		exit( 1 );
	}

	if ( (e = ldap_first_entry( ld, res )) == NULL ) {
		fprintf( fp, "No entry found.  This should not normally happen.\r\n" );
		fprintf( fp, ".\r\n" );
		exit( 1 );
	}
	if ( (dn = ldap_get_dn( ld, e )) == NULL ) {
		ldap_perror( ld, "ldap_get_dn" );
		exit( 1 );
	}
	if ( (ufn = ldap_dn2ufn( dn )) == NULL ) {
		ldap_perror( ld, "ldap_dn2ufn" );
		exit( 1 );
	}
	for ( i = 0; i < strip; i++ ) {
		if ( (s = strrchr( ufn, ',' )) == NULL )
			break;
		*s = '\0';
	}
	fprintf( fp, "\"%s\"\r\n", ufn );
	print_attr( ld, fp, "  Also known as:", "cn", e, 0 );
	print_attr( ld, fp, "  E-mail address:", "mail", e, 0 );
	print_attr( ld, fp, "  Fax number:", "facsimileTelephoneNumber", e, 0 );
	print_attr( ld, fp, "  Business phone:", "telephoneNumber", e, 0 );
	print_attr( ld, fp, "  Business address:", "postalAddress", e, 1 );
	print_attr( ld, fp, "  Title:", "title", e, 0 );
	print_attr( ld, fp, "  Uniqname:", "uid", e, 0 );

	free( dn );
	free( ufn );
}
