/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/************************************************************************
 * File:	beam3d.c						*
 *									*
 * Description:	This file contains the definition structure and		*
 *		stiffness function for a 3-dimensional beam element.	*
 *									*
 * History:	V1.0 by Jason Gobat and Darren C. Atkinson		*
 ************************************************************************/

# include <math.h>
# include <stdio.h>
# include "allocate.h"
# include "element.h"
# include "misc.h"


struct definition beam3dDefinition = {
    beam3dType, Linear, 2, 2, 6, {0, 1, 2, 3, 4, 5, 6}
};

Matrix Beam3dLocalK            ( );
Matrix Beam3dTransformMatrix   ( );
Vector Beam3dEquivNodalForces  ( );

int beam3dEltStiffness (element)
   Element	element;
{
   Matrix		T,
			ke;
   int			count;
   Vector		equiv;
   unsigned		i;
   static Matrix	Tt,
			temp = NullMatrix;

   if (temp == NullMatrix) {
      Tt = CreateMatrix (12,12);
      temp = CreateMatrix (12,12);

      if(Tt==NullMatrix || temp==NullMatrix)
         Fatal ("allocation error creating element stiffness matrix");
   }

   if (element -> material -> A == 0) {
      error ("3d beam element %d has 0.0 for cross-sectional area (A)", element -> number);
      return 1;
   }
   if (element -> material -> E == 0) {
      error ("3d beam element %d has 0.0 for Young's modulus (E)", element -> number);
      return 1;
   }
   if (element -> material -> Iy == 0) {
      error ("3d beam element %d has 0.0 for Iyy moment of inertia (Iy)", element -> number);
      return 1;
   }
   if (element -> material -> Iz == 0) {
      error ("3d beam element %d has 0.0 for Izz moment of inertia (Iz)", element -> number);
      return 1;
   }
   if (element -> material -> J == 0) {
      error ("3d beam element %d has 0.0 for polar moment of inertia (J)", element -> number);
      return 1;
   }
   if (element -> material -> G == 0) {
      error ("3d beam element %d has 0.0 for shear modulus (G)", element -> number);
      return 1;
   }

   ke = Beam3dLocalK (element);
   if (ke == NullMatrix)
      return 1;
    
   T = Beam3dTransformMatrix (element);
   if (T == NullMatrix)
      return 1; 

   Tt = TransposeMatrix (Tt,T);

   element -> K = CreateMatrix (12,12);
   if (element -> K == NullMatrix) 
      Fatal ("allocation error creating element %d stiffness matrix",element -> number);

   temp = MultiplyMatrices (temp,Tt,ke); 
   element -> K = MultiplyMatrices (element -> K,temp,T);

   if (element -> numdistributed > 0) {
      equiv = Beam3dEquivNodalForces (element, &count);
      if (equiv == NullMatrix)
         return count;

      for (i = 1 ; i <= 6 ; i++) {
         element -> node[1] -> eq_force[i] += VectorData (equiv) [i];
         element -> node[2] -> eq_force[i] += VectorData (equiv) [i+6];
      }
   }

   return 0;
}

int beam3dEltStress (element)
   Element	element;
{
   unsigned		i;
   int			count;
   static Vector	f,
			eq_local,
			dlocal,
			d = NullMatrix;
   Vector		equiv;
   Matrix		T,
			ke;

   if (d == NullMatrix) {
      d = CreateVector (12);
      f = CreateVector (12);
      dlocal = CreateVector (12);

      if (d == NullMatrix || f == NullMatrix || dlocal == NullMatrix)
         Fatal ("allocation error in element %d stresses",element -> number);
   } 

   for (i = 1 ; i <= 6 ; i++) {
      VectorData (d) [i] = element -> node[1] -> dx[i];
      VectorData (d) [i+6] = element -> node[2] -> dx[i]; 
   }

   T = Beam3dTransformMatrix (element);
   if (T == NullMatrix)
      return 1;

   ke = Beam3dLocalK (element);
   if (ke == NullMatrix)
      return 1;

   dlocal = MultiplyMatrices (dlocal, T, d);
   f = MultiplyMatrices (f, ke, dlocal);
   
   if (element -> numdistributed > 0) {
      equiv = Beam3dEquivNodalForces (element, &count);
      if (equiv == NullMatrix)
         return count;

      eq_local = MultiplyMatrices (eq_local, T, equiv);

      for (i = 1 ; i <= 12 ; i++)
         VectorData (f) [i] -= VectorData (eq_local) [i];
   } 

   element -> stress = Allocate (double, 12);
   if (element -> stress == NULL)
      Fatal ("allocation error creating element %d stress vector",element -> number);

   UnitOffset (element -> stress);
   element -> numstresses = 12;

   for (i = 1 ; i <= 12 ; i++)
      element -> stress [i] = VectorData (f) [i];

   return 0;          
} 

Matrix Beam3dLocalK (element)
   Element	element;
{
   double		L,
			L3,
			L2;
   double		EIz,
			EIy,
			GJ,
			AEonL;
   static Matrix	ke = NullMatrix;

   if (ke == NullMatrix) {
      ke = CreateMatrix (12,12);

      if (ke == NullMatrix)
         Fatal ("allocation error in local element %d stiffness",element -> number);
   }

   ZeroMatrix (ke);

   L = ElementLength (element, 3);

   if (L <= TINY) {
      error ("length of element %d is zero to machine precision",element -> number);
      return NullMatrix;
   } 

   L2 = L*L;
   L3 = L2*L;

   EIy = element -> material -> E * element -> material -> Iy;
   EIz = element -> material -> E * element -> material -> Iz;
   GJ = element -> material -> J * element -> material -> G;
   AEonL = (element -> material -> E * element -> material -> A)/L;

   MatrixData (ke) [1][1]  = MatrixData (ke) [7][7]   = AEonL;
   MatrixData (ke) [2][2]  = MatrixData (ke) [8][8]   = 12*EIz/L3;
   MatrixData (ke) [3][3]  = MatrixData (ke) [9][9]   = 12*EIy/L3;
   MatrixData (ke) [4][4]  = MatrixData (ke) [10][10] = GJ/L;
   MatrixData (ke) [5][5]  = MatrixData (ke) [11][11] = 4*EIy/L;
   MatrixData (ke) [6][6]  = MatrixData (ke) [12][12] = 4*EIz/L;
   MatrixData (ke) [1][7]  = -AEonL;
   MatrixData (ke) [2][6]  = MatrixData (ke) [2][12]  = 6*EIz/L2;
   MatrixData (ke) [2][8]  = -12*EIz/L3;
   MatrixData (ke) [3][5]  = MatrixData (ke) [3][11]  = -6*EIy/L2;
   MatrixData (ke) [3][9]  = -12*EIy/L3;
   MatrixData (ke) [4][10] = -GJ/L;
   MatrixData (ke) [5][9]  = MatrixData (ke) [9][11]  = 6*EIy/L2;
   MatrixData (ke) [5][11] = 2*EIy/L;
   MatrixData (ke) [6][8]  = MatrixData (ke) [8][12]  = -6*EIz/L2;
   MatrixData (ke) [6][12] = 2*EIz/L;

   MirrorMatrix (ke,ke);
   return ke;
}

Matrix Beam3dTransformMatrix (element)
   Element	element;
{
   unsigned		i;
   double	   	cl,	
			cm,
			cn,
			d,	
			L;
   static Matrix	T = NullMatrix;

   if (T == NullMatrix) {
      T = CreateMatrix (12,12);

      if (T == NullMatrix)
         Fatal ("allocation error in element %d transform matrix",element -> number);
   }

   ZeroMatrix (T);

   L = ElementLength (element, 3);

   if (L <= TINY) {
      error ("length of element %d is zero to machine precision",element -> number);
      return NullMatrix;
   } 

   cl = (element -> node[2] -> x - element -> node[1] -> x)/L;
   cm = (element -> node[2] -> y - element -> node[1] -> y)/L;
   cn = (element -> node[2] -> z - element -> node[2] -> z)/L;

   d = sqrt (cl*cl + cm*cm);

   for (i = 0 ; i <= 9 ; i += 3) {
      MatrixData (T) [i+1][i+1] = cl;
      MatrixData (T) [i+1][i+2] = cm;
      MatrixData (T) [i+1][i+3] = cn;
      if (d <= TINY) {
         MatrixData (T) [i+2][i+1] = 0;
         MatrixData (T) [i+2][i+2] = 1;
         MatrixData (T) [i+3][i+1] = 1;
         MatrixData (T) [i+3][i+2] = 0;
      }
      else {
         MatrixData (T) [i+2][i+1] = -cm/d;
         MatrixData (T) [i+2][i+2] = -cl/d;
         MatrixData (T) [i+3][i+1] = -cl*cn/d;
         MatrixData (T) [i+3][i+2] = -cm*cn/d;
      }
      MatrixData (T) [i+3][i+3] = d;
   }

   return T;
}

Vector Beam3dEquivNodalForces (element, err_count)
   Element	element;
   int		*err_count;
{
   double		L;
   double		wa,wb;
   double		force1,
			force2,
			moment1,
			moment2;
   int			count;
   unsigned		i,j;
   Matrix		T;
   static Matrix	Tt;
   static Vector 	equiv = NullMatrix;
   static Vector	result;
 
   if (equiv == NullMatrix) {
      equiv = CreateVector (12);
      result = CreateVector (12);
      Tt = CreateMatrix (12,12);

      if (equiv == NullMatrix || result == NullMatrix || Tt == NullMatrix)
         Fatal ("error creating equivalent force vector for element %d",element -> number);
   }

   ZeroMatrix (equiv);

   count = 0;
 
   if (element -> numdistributed != 1) {
      error ("beam3d elt %d can only have one distributed load", element -> number);
      count++;
   }
 
   if (element -> distributed[1] -> nvalues != 2) {
      error ("beam3d elt %d does not have 2 nodal values for a distributed load",
              element -> number);
      count++;
   }

   L = ElementLength (element, 3);

   if (L <= TINY) {
      error ("length of element %d is zero to machine precision",element -> number);
      count ++;
   } 

   if (element -> distributed[1] -> direction == Perpendicular ||
       element -> distributed[1] -> direction == DirX) {
       error ("invalid direction specified for beam3d elt %d distributed load",
              element -> number);
       count++;
   }

   for (i = 1 ; i <= element -> distributed[1] -> nvalues ; i++) {
      if (element -> distributed[1] -> value[i].node < 1 || 
          element -> distributed[1] -> value[i].node > 2) {

          error ("incorrect node numbering for beam3d elt %d distributed load",
                  element -> number);
          count++;
      }
   }

   if (element -> distributed[1] -> value[1].node == 
       element -> distributed[1] -> value[2].node) {

       error ("incorrect node numbering for elt %d distributed load", element -> number);
       count++;
   }

	/* 
	 * Thats all the error checking, bail out if we've had any
	 */

   if (count) {
      *err_count = count;
      return NullMatrix;
   }

   if (element -> distributed[1] -> value[1].node == 1) {
      wa = element -> distributed[1] -> value[1].magnitude;
      wb = element -> distributed[1] -> value[2].magnitude;
   }
   else if (element -> distributed[1] -> value[1].node == 2) {
      wb = element -> distributed[1] -> value[1].magnitude;
      wa = element -> distributed[1] -> value[2].magnitude;
   }

   if (element -> distributed[1] -> direction != Parallel) {
      if (wa == wb) {		     /* uniform distributed load    */
         force1 = force2 = -wa*L/2.0;
         moment1 = -wa*L*L/12.0;
         moment2 = -moment1;
      } 
      else if (fabs(wa) > fabs(wb)) {     /* load sloping node1 to node2 */
         force1 = -wb*L/2.0 - 7.0*(wa - wb)*L/20.0; 
         force2 = -wb*L/2.0 - 3.0*(wa - wb)*L/20.0;
         moment1 = -wb*L*L/12.0 - (wa - wb)*L*L/20.0;
         moment2 = wb*L*L/12.0 + (wa - wb)*L*L/30.0;
      }
      else if (fabs(wa) < fabs(wb)) {     /* load sloping node2 to node1 */
         force1 = -wa*L/2.0 - 3.0*(wb - wa)*L/20.0; 
         force2 = -wa*L/2.0 - 7.0*(wb - wa)*L/20.0;
         moment1 = -wa*L*L/12.0 - (wb - wa)*L*L/30.0;
         moment2 = wa*L*L/12.0 + (wb - wa)*L*L/20.0;
      } 

      if (element -> distributed [1] -> direction == DirY) {
         VectorData (equiv) [2] = force1;
         VectorData (equiv) [8] = force2;
         VectorData (equiv) [6] = moment1;
         VectorData (equiv) [12] = moment1;
      }
      else if (element -> distributed [1] -> direction == DirZ) {
         VectorData (equiv) [3] = force1;
         VectorData (equiv) [9] = force2;
         VectorData (equiv) [5] = moment1;
         VectorData (equiv) [11] = moment1;
      }

   } else {
      if (wa == wb) 
         force1 = force2 = wa*L/2;
      else if (fabs (wa) > fabs (wb)) {
         force1 = wb*L/2 + (wa - wb)*L/3;
         force2 = wb*L/2 + (wa - wb)*L/6;
      }
      else if (fabs (wb) > fabs (wa)) {
         force1 = wa*L/2 + (wb - wa)*L/6;
         force2 = wa*L/2 + (wb - wa)*L/3;
      }
      VectorData (equiv) [1] = force1;
      VectorData (equiv) [7] = force2;
   }

	/*
	 * Now that we know all is okay, allocate some memory if we
	 * haven't already done so for some other element
	 */

   for (i = 1 ; i <= element -> definition -> numnodes ; i++) {
      if (element -> node[i] -> eq_force == NULL) {
         element -> node[i] -> eq_force = Allocate (double, 6);

         if (element -> node[i] == NULL)
            Fatal ("unable to allocate memory");

         UnitOffset (element -> node[i] -> eq_force);
         for (j = 1 ; j <= 6 ; j++)
            element -> node[i] -> eq_force[j] = 0.0;
      }
   }

   T = Beam3dTransformMatrix (element);
   Tt = TransposeMatrix (Tt, T);
   result = MultiplyMatrices (result, Tt, equiv);

   *err_count = 0;
   return result; 
}
