/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/************************************************************************
 * File:	matrix.h						*
 *									*
 * Description:	This file contains the public type, macro, and function	*
 *		declarations for the matrix package.			*
 ************************************************************************/

# ifndef _MATRIX_H
# define _MATRIX_H
# include <stdio.h>

typedef struct matrix {
    unsigned nrows;
    unsigned ncols;
    double  *vdata;
    double **mdata;
} *Matrix, *Vector;

# define MatrixRows(m)	((m) -> nrows)
# define MatrixCols(m)	((m) -> ncols)
# define MatrixData(m)	((m) -> mdata)
# define VectorSize(v)	((v) -> nrows)
# define VectorData(v)	((v) -> vdata)

# define NullMatrix	((Matrix) 0)
# define NullVector	((Vector) 0)


# if defined (__STDC__) || defined (__GNUC__)

    extern Matrix CreateMatrix     (unsigned, unsigned);
    extern Matrix CreateSubmatrix  (unsigned, unsigned);
    extern Vector CreateVector     (unsigned);
    extern void   DestroyMatrix    (Matrix);
    extern void   DestroySubmatrix (Matrix);
    extern void   DestroyVector    (Vector);
    extern void   PrintMatrix      (Matrix, FILE *);
    extern void   PrintVector      (Vector, FILE *);

    extern Matrix MatrixAlgebra    (char *, ...);

    extern Matrix SetSubmatrix     (Matrix, Matrix, unsigned, unsigned);
    extern Matrix ScaleMatrix      (Matrix, Matrix, double, double);
    extern Matrix AssignMatrix     (Matrix, Matrix);
    extern Matrix InvertMatrix     (Matrix, Matrix);
    extern Matrix TransposeMatrix  (Matrix, Matrix);
    extern Matrix MirrorMatrix     (Matrix, Matrix);
    extern Matrix ZeroMatrix       (Matrix);

    extern Matrix AddMatrices      (Matrix, Matrix, Matrix);
    extern Matrix SubtractMatrices (Matrix, Matrix, Matrix);
    extern Matrix MultiplyMatrices (Matrix, Matrix, Matrix);

    extern Matrix LUDecomposition  (Matrix, Vector, Matrix);
    extern Vector LUBackSolve      (Vector, Matrix, Vector, Vector);
    extern Vector EigenSolve       (Vector, Matrix, Matrix);

# else

    extern Matrix CreateMatrix     ( );
    extern Matrix CreateSubmatrix  ( );
    extern Vector CreateVector     ( );
    extern void   DestroyMatrix    ( );
    extern void   DestroySubmatrix ( );
    extern void   DestroyVector    ( );
    extern void   PrintMatrix      ( );
    extern void   PrintVector      ( );

    extern Matrix MatrixAlgebra    ( );

    extern Matrix SetSubmatrix     ( );
    extern Matrix ScaleMatrix      ( );
    extern Matrix AssignMatrix     ( );
    extern Matrix InvertMatrix     ( );
    extern Matrix TransposeMatrix  ( );
    extern Matrix MirrorMatrix     ( );
    extern Matrix ZeroMatrix       ( );

    extern Matrix AddMatrices      ( );
    extern Matrix SubtractMatrices ( );
    extern Matrix MultiplyMatrices ( );

    extern Matrix LUDecomposition  ( );
    extern Vector LUBackSolve      ( );
    extern Vector EigenSolve       ( );

# endif

# endif /* _MATRIX_H */
