/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/************************************************************************
 * File:	fe.h							*
 *									*
 * Description:	This file contains the primary type and function	*
 *		declarations for the finite element package.		*
 *									*
 * History:	V1.0 by Jason Gobat and Darren C. Atkinson		*
 ************************************************************************/

# ifndef _FE_H
# define _FE_H
# include "matrix.h"


# define TINY 1.0e-60
# define StoreEltStiffnesses   1
# define DestroyEltStiffnesses 2


/* Element shapes */

typedef enum {
    Linear = 1,
    Planar = 2,
    Solid  = 3
} Shape;


/* Load directions */

typedef enum {
    DirX          = 1,
    DirY          = 2,
    DirZ          = 3,
    Parallel      = 4,
    Perpendicular = 5
} Direction;


/* Degrees of freedom */

typedef enum {
    Tx = 1, Fx = 1,			/* translation/force along x axis */
    Ty = 2, Fy = 2,			/* translation/force along y axis */
    Tz = 3, Fz = 3,			/* translation/force along z axis */
    Rx = 4, Mx = 4,			/* rotation/moment about x axis   */
    Ry = 5, My = 5,			/* rotation/moment about y axis   */
    Rz = 6, Mz = 6			/* rotation/moment about z axis   */
} DOF;


/* A node-magnitude pair */

typedef struct pair {
    unsigned node;			/* node number */
    double   magnitude;			/* magnitude   */
} Pair;


/* A reaction force */

typedef struct reaction {
    double   force;			/* reaction force             */
    unsigned node;			/* node number                */
    unsigned dof;			/* affected degree of freedom */
} *Reaction;


/* An element definition */

typedef struct definition {
    int      type;			/* element type                       */
    Shape    shape;			/* element dimensional shape          */
    unsigned numnodes;			/* number of nodes in element         */
    unsigned shapenodes;		/* number of nodes which define shape */
    unsigned numdofs;			/* number of degrees of freedom       */
    unsigned dofs [7];			/* degrees of freedom                 */
} *Definition;


/* A distributed load */

typedef struct distributed {
    char     *aux;			/* auxillary data pointer   */
    char     *name;			/* name of distributed load */
    Direction direction;		/* direction of load        */
    unsigned  nvalues;			/* number of values         */
    Pair     *value;			/* nodes and magnitudes     */
} *Distributed;


/* A force */

typedef struct force {
    char  *aux;				/* auxillary data pointer */
    char  *name;			/* name of force          */
    double force [7];			/* force vector           */
} *Force;


/* A constraint */

typedef struct constraint {
    char *aux;				/* auxillary data pointer */
    char *name;				/* name of constraint     */
    char  constraint [7];		/* constraint vector      */
} *Constraint;


/* A material */

typedef struct material {
    char  *aux;				/* auxillary data pointer           */
    char  *name;			/* name of material                 */
    double E;				/* Young's modulus                  */
    double Ix;				/* moment of inertia about x-x axis */
    double Iy;				/* moment of inertia about y-y axis */
    double Iz;				/* moment of inertia about z-z axis */
    double A;				/* cross-sectional area             */
    double J;				/* polar moment of inertia          */
    double G;				/* bulk (shear) modulus             */
    double t;				/* thickness                        */
    double rho;				/* density                          */
    double nu;				/* Poisson's ratio                  */
    double kappa;			/* shear force correction           */
} *Material;


/* A node */

typedef struct node {
    char      *aux;			/* auxillary data pointer         */
    unsigned   number;			/* node number                    */
    Constraint constraint;		/* constrained degrees of freedom */
    Force      force;			/* force acting on node           */
    double    *eq_force;		/* equivalent force               */
    double     dx [7];			/* displacement                   */
    double     x;			/* x coordinate                   */
    double     y;			/* y coordinate                   */
    double     z;			/* z coordinate                   */
} *Node;


/* An element */

typedef struct element {
    char       *aux;			/* auxillary data pointer */
    unsigned    number;			/* element number         */
    Node       *node;			/* array of nodes         */
    Matrix      K;			/* stiffness matrix       */
    Material    material;		/* material               */
    Definition  definition;		/* definition of element  */
    Distributed distributed [4];	/* distributed loads      */
    unsigned    numdistributed;		/* number of loads        */
    double     *stress;			/* element stresses       */
    unsigned    numstresses;		/* number of stresses     */
} *Element;


/* Function declarations */

# if defined (__STDC__) || defined (__GNUC__)

extern unsigned FindDOFS               (Element *, unsigned, unsigned *);
extern Matrix   AssembleStiffness      (Element *, unsigned, unsigned,
                                        unsigned, unsigned *, int *);
extern Matrix   ZeroConstrainedDOFS    (Matrix, Node *, unsigned, unsigned,
                                        unsigned *);
extern Vector   ZeroFixedDisplacements (Vector, Node *, unsigned, unsigned,
                                        unsigned *);
extern unsigned SolveForReactions      (Matrix, Vector, Node *, unsigned, 
                                        unsigned, unsigned *, Reaction **);
extern Matrix   ZeroRowCol             (Matrix, unsigned);
extern Vector   ConstructForceVector   (Node *, unsigned, unsigned, unsigned *);
extern Vector   SolveForDisplacements  (Matrix, Vector, Node *, unsigned,
                                        unsigned, unsigned *);
extern void     DrawStructure          (Element *, Node *, unsigned, unsigned,
                                        char *);
extern int      EltStiffness           (Element, unsigned);
extern int      WriteGraphicsFile      (char *, Element *, unsigned);
extern Vector	MakeCompactColumns     (Matrix, unsigned *);
extern double   DotProduct	       (double *, double *, unsigned);
extern int      CroutFactorization     (Vector, unsigned *, unsigned);
extern int      CompactForwardBack     (Vector, Vector, unsigned *, unsigned);
extern void     ClearNodes             (Node *, unsigned);
extern double   ElementArea	       (Element, unsigned);
extern int      WriteFeltFile	       (FILE *, Element *, Node *, unsigned,
					unsigned, char *);
extern int      EltStresses	       (Element *, unsigned);
extern int      WriteOutput	       (FILE *, char *, Element *, Node *,
					Reaction *, unsigned, unsigned,
					unsigned, unsigned);

# else

extern unsigned FindDOFS               ( );
extern Matrix   AssembleStiffness      ( );
extern Matrix   ZeroConstrainedDOFS    ( );
extern Vector   ZeroFixedDisplacements ( );
extern unsigned SolveForReactions      ( );
extern Matrix   ZeroRowCol             ( );
extern Vector   ConstructForceVector   ( );
extern Vector   SolveForDisplacements  ( );
extern void     DrawStructure          ( );
extern int      EltStiffness           ( );
extern Vector	MakeCompactColumns     ( );
extern double   DotProduct             ( );
extern int	CroutFactorization     ( );
extern int      CompactForwardBack     ( );
extern int      WriteGraphicsFile      ( );
extern void     ClearNodes  	       ( );
extern double   ElementArea	       ( );
extern int      WriteFeltFile	       ( );
extern int      EltStresses	       ( );
extern int      WriteOutput	       ( );

# endif

# endif /* _FE_H */
