/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <unistd.h>
# include <X11/Intrinsic.h>
# include <X11/StringDefs.h>
# include <X11/Xaw/Simple.h>
# include "dialog.h"
# include "forms.h"
# include "fe.h"
# include "Tree.h"
# include "problem.h"
# include "globals.h"
# include "Drawing.h"
# include "vfe.h"
# include "procedures.h"
# include "error.h"


/****************************************************************************
 *
 * Function:	SolveProblem
 *
 * Description:	Basically main from the Felt application ... pretty 
 *		straightforward
 *
 ***************************************************************************/

int SolveProblem (element, node, numelts, numnodes, title)
    Element	*element;
    Node	*node;
    unsigned	numelts,
		numnodes;
    char	*title;
{
    int		 i;			/* loop variable		*/
    unsigned	 count;			/* number of affected dofs	*/
    Matrix	 K,			/* global stiffness matrix	*/
		 Kcond;			/* condensed stiffness matrix	*/
    Vector	 F,			/* force vector			*/
		 Fcond,			/* condensed force vector	*/
		 d;			/* displacement vector		*/
    unsigned	 dofs [7];		/* degrees of freedom		*/
    int		 status;		/* return status		*/
    Reaction	 *R;			/* reaction force vector	*/
    unsigned	 numreactions;		/* the number of reactions	*/
   
    char	*temp_out;
    FILE	*output;

        /*
         * Do everything
         */

    SetShellTitle (error_dialog -> shellwidget, "Solve Problem");

    ClearNodes (node, numnodes);

    count = FindDOFS (element, numelts, dofs);

    status = DestroyEltStiffnesses;

    K = AssembleStiffness (element, numelts, numnodes, count, dofs, &status);

    if (status) {
	error ("%d fatal errors in element stiffness definitions.", status);

        DestroyMatrix (K);
        return 1;
    }

    Kcond = ZeroConstrainedDOFS (K, node, numnodes, count, dofs);
        
    F = ConstructForceVector (node, numnodes, count, dofs);

    Fcond = ZeroFixedDisplacements (F, node, numnodes, count, dofs);

    d = SolveForDisplacements (Kcond, Fcond, node, numnodes, count, dofs);

    if (d == NullVector) {
       DestroyMatrix (Kcond);
       DestroyMatrix (K);
       DestroyVector (F);
       DestroyVector (Fcond);

       return 1;
    }    

    numreactions = SolveForReactions (K, d, node, numnodes, count, dofs, &R);

    status = EltStresses (element, numelts);
    if (status) {
       error ("%d errors found computing element stresses.",status);
       return 1;
    } 

    temp_out = tempnam (NULL,"velv");
    output = fopen (temp_out,"w");
    if (output == NULL) {
       error ("Could not open temporary file for output.");
       return 1;
    }

    WriteOutput (output, title, element, node, R, numelts, numnodes, numreactions, 0);
    fclose (output);

    SetShellTitle (outputShell, title);
    UpdateOutputWindow (temp_out);

    unlink (temp_out);
    XtFree ((char *) temp_out);

    DestroyMatrix (K);
    DestroyMatrix (Kcond);
    DestroyVector (F);
    DestroyVector (Fcond);

    if (numreactions) {
	for (i = 1; i <= numreactions; i ++)
	    XtFree ((char *) R [i]);

	XtFree ((char *) (R + 1));
    }

    return 0;
}


static unsigned count;
static FigureAttributes attr;
static char buffer [10];
static Boolean renumber_redraw;


static int SetNodeArray (item)
    Item item;
{
    Node n = (Node) item;
    Drawn drawn = (Drawn) n -> aux;


    node [++ count] = n;

    if (n -> number != count) {
	n -> number = count;
	if (drawn -> label != NULL) {
	    if (renumber_redraw == True)
		DW_SetAutoRedraw (drawing, renumber_redraw = False);
	    sprintf (attr.text = buffer, " %d", count);
	    DW_SetAttributes (drawing, drawn -> label, DW_FigureText, &attr);
	}
    }

    return 0;
}


static int SetElementArray (item)
    Item item;
{
    Element e = (Element) item;


    element [++ count] = e;

    if (e -> number != count)
	e -> number = count;

    return 0;
}


void CompactOperation (output)
    FILE *output;
{
    numnodes = TreeSize (problem.node_tree);
    numelts = TreeSize (problem.element_tree);

    if (numnodes == 0 || numelts == 0)
	return;

    node = (Node *) XtMalloc (sizeof (Node) * numnodes);
    element = (Element *) XtMalloc (sizeof (Element) * numelts);
    node --;
    element --;

    (void) TreeSetIterator (problem.node_tree, SetNodeArray);
    (void) TreeSetIterator (problem.element_tree, SetElementArray);

    count = 0;
    renumber_redraw = True;
    (void) TreeIterate (problem.node_tree);
    count = 0;
    (void) TreeIterate (problem.element_tree);

    if (renumber_redraw == False && number_status == True)
	DW_SetAutoRedraw (drawing, True);

    if (title == NULL)
	title = XtNewString ("Unnamed");

    if (output == NULL) {
	BufferErrors (True);
	SolveProblem (element, node, numelts, numnodes, title);
	BufferErrors (False);
	if (XtIsRealized (nodeShell))
	    GetNodeInformation (displayed_node);
	if (XtIsRealized (elementShell))
	    GetElementInformation (displayed_element);
    } else
	WriteFeltFile (output, element, node, numelts, numnodes, title);
	

    node ++;
    element ++;
    XtFree ((char *) node);
    XtFree ((char *) element);
}


void DefineProblem ()
{
    unsigned selected;
    String   ans;
    static char suggestion [256];

    if (title != NULL)
	strcpy (suggestion, title);
    else
	suggestion [0] = 0;

    SetShellTitle (file_dialog -> shellwidget, "Define Problem");
    selected = PopupDialog (file_dialog, "Problem title:", suggestion, &ans);
    strcpy (suggestion, ans);

    if (selected == Okay) {
	if (title != NULL)
	    XtFree (title);

	title = XtNewString (ans);
	changeflag = True;
    }
}
