/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <X11/Intrinsic.h>
# include "fe.h"
# include "forms.h"
# include "dialog.h"
# include "problem.h"
# include "globals.h"
# include "objects.h"
# include "allocate.h"
# include "procedures.h"
# include "error.h"
# include "Drawing.h"
# include "vfe.h"

static int count;


static int SetMaterialNames (item)
    Item item;
{
    if (active_material == (Material) item)
	active_material_number = count;

    material_names [count ++] = ((Material) item) -> name;
    return 0;
}


void GatherMaterialNames ( )
{
    int size;


    if ((size = TreeSize (problem.material_tree)) <= 0)
	return;

    count = 0;
    size *= sizeof (String);
    material_names = (String *) XtRealloc ((char *) material_names, size);
    (void) TreeSetIterator (problem.material_tree, SetMaterialNames);
    (void) TreeIterate (problem.material_tree);
}


void SetActiveMaterial ()
{
   int             	result, size;
   struct material	dummy;


   SetShellTitle (material_list -> shellwidget, "Set Active Material");
   if ((size = TreeSize (problem.material_tree)) <= 0) {
      error ("No materials defined.");
      return;
   }

   SetListItems (material_list, material_names, size);
   result = GetListFormValue (material_list, active_material_number);

   if (result > -1) {
      dummy.name = material_names [result];
      active_material = (Material) TreeSearch (problem.material_tree, &dummy);
      active_material_number = result;
      changeflag = True;
   }
}


void GetMaterialDefinition ()
{
   String 			result[12];
   extern EntryRecord		material_entries;
   unsigned			status;
   Material			material;
   Item			        found;

   SetShellTitle (material_form -> shellwidget, "Define Material");
   status = GetEntryFormValues (material_form, material_entries, result);
   if (status) {
      if (strlen (result[0]) > 0)  {
	 material = CreateMaterial (XtNewString (result [0]));
	 material -> E = atof (result [1]);
	 material -> A = atof (result [2]);
	 material -> Ix = atof (result [3]);
	 material -> Iy = atof (result [4]);
	 material -> Iz = atof (result [5]);
	 material -> J = atof (result [6]);
	 material -> G = atof (result [7]);
	 material -> nu = atof (result [8]);
	 material -> kappa = atof (result [9]);
	 material -> t = atof (result [10]);
	 material -> rho = atof (result [11]);

	 found = TreeInsert (problem.material_tree, (Item) material);
	 if (found != (Item) material) {
	    error ("Material %s already exists.", material -> name);
	    DestroyMaterial (material);
	    return;
	 }

	 active_material = material;
	 GatherMaterialNames ( );
	 changeflag = True;
      } else
	 error ("No material name specified.");
   } 
}


void EditMaterialInfo ()
{
   int             	i, status, size;
   String		result[12];
   extern EntryRecord   material_entries;
   Material		material;
   static String        suggestion [12];
   static int           flag = 0;
   struct material	dummy;
   Item			found;

   if (!flag) {
	flag = 1;
	for (i = 0; i < 12; i ++)
	    suggestion [i] = (String) XtMalloc (sizeof (char) * 80);
   }

   SetShellTitle (material_list -> shellwidget, "Edit Material");
   if ((size = TreeSize (problem.material_tree)) <= 0) {
      error ("No materials defined.");
      return;
   }

   SetListItems (material_list, material_names, size);
   status = GetListFormValue (material_list, active_material_number);

   if (status == -1)
	return;

   dummy.name = material_names [status];
   material = (Material) TreeSearch (problem.material_tree, &dummy);

   strcpy  (suggestion [0], material -> name);
   sprintf (suggestion [1], "%g", material -> E);
   sprintf (suggestion [2], "%g", material -> A);
   sprintf (suggestion [3], "%g", material -> Ix);
   sprintf (suggestion [4], "%g", material -> Iy);
   sprintf (suggestion [5], "%g", material -> Iz);
   sprintf (suggestion [6], "%g", material -> J);
   sprintf (suggestion [7], "%g", material -> G);
   sprintf (suggestion [8], "%g", material -> nu);
   sprintf (suggestion [9], "%g", material -> kappa);
   sprintf (suggestion [10], "%g", material -> t);
   sprintf (suggestion [11], "%g", material -> rho);
   FillTextBuffers (material_form, material_entries, suggestion);

   SetShellTitle (material_form -> shellwidget, "Edit Material");
   status = GetEntryFormValues (material_form, material_entries, result);

   if (status) {
      if (strlen (result[0]) > 0)  {
	 if (strcmp (result [0], material -> name))
	    material = CreateMaterial (XtNewString (result [0]));

	 material -> E = atof (result [1]);
	 material -> A = atof (result [2]);
	 material -> Ix = atof (result [3]);
	 material -> Iy = atof (result [4]);
	 material -> Iz = atof (result [5]);
	 material -> J = atof (result [6]);
	 material -> G = atof (result [7]);
	 material -> nu = atof (result [8]);
	 material -> kappa = atof (result [9]);
	 material -> t = atof (result [10]);
	 material -> rho = atof (result [11]);

	 found = TreeInsert (problem.material_tree, (Item) material);
	 if (found != (Item) material) {
	    error ("Material %s already exists.", material -> name);
	    DestroyMaterial (material);
	    return;
	 }

	 active_material = material;
	 GatherMaterialNames ( );
	 changeflag = True;
      } else
	 error ("No material name specified.");
   }
}


static Material material;
static int      number;

static int MaterialRef (item)
    Item item;
{
    if (((Element) item) -> material == material) {
	number = ((Element) item) -> number;
	return 1;
    }

    return 0;
}


void DeleteMaterial ()
{
   int             	result, size;
   struct material	dummy;


   SetShellTitle (material_list -> shellwidget, "Delete Material");
   if ((size = TreeSize (problem.material_tree)) <= 0) {
      error ("No materials defined.");
      return;
   }

   SetListItems (material_list, material_names, size);
   result = GetListFormValue (material_list, -1);

   if (result > -1) {
      count = 0;
      dummy.name = material_names [result];
      material = (Material) TreeSearch (problem.material_tree, &dummy);
      (void) TreeSetIterator (problem.element_tree, MaterialRef);
      if (TreeIterate (problem.element_tree)) {
	error ("Material %s is still in use by element %d.", dummy.name, number);
	return;
      }

      DestroyMaterial ((Material) TreeDelete (problem.material_tree, &dummy));
      if (active_material_number == result)
	active_material_number = -1;

      GatherMaterialNames ( );
      changeflag = True;
   }
}


Problem     saved;


static int UpdateMaterial (item)
    Item item;
{
    Item     found;
    Material old;
    Material new;


    found = TreeInsert (saved.material_tree, item);
    if (found != item) {
	new = (Material) item;
	old = (Material) found;
	old -> E = new -> E;
	old -> A = new -> A;
	old -> Ix = new -> Ix;
	old -> Iy = new -> Iy;
	old -> Iz = new -> Iz;
	old -> J = new -> J;
	old -> G = new -> G;
	old -> nu = new -> nu;
	old -> t = new -> t;
	old -> rho = new -> rho;
	old -> kappa = new -> kappa;
	DestroyMaterial (new);
    }
    return 0;
}


void OpenMaterialFile ( )
{
    FILE       *fp;
    String	ans;
    String      title;
    static char suggestion [256] = "";
    unsigned    selected;
    int         status;
    int         usecpp = 1;
    char       *cpp = "/lib/cpp";


    SetShellTitle (file_dialog -> shellwidget, "Open Material Database");
    selected = PopupDialog (file_dialog, "Database to load:", suggestion, &ans);
    (void) strcpy (suggestion, ans);

    if (selected != Okay || !ans [0])
	return;


    if (freopen (ans, "r", stdin) == NULL) {
	error ("Could not access file %s.", ans);
	return;
    }

    if (usecpp) {
	if ((fp = popen (cpp, "r")) == NULL) {
	    error ("Could not execute %s.", cpp);
	    return;
	}
    } else
	fp = stdin;

    saved = problem;
    BufferErrors (True);
    status = GetProblem (fp, &element, &node, &numelts, &numnodes, &title);
    BufferErrors (False);
    (void) pclose (fp);

    if (status)
	DestroyProblem ((ItemDestructor) DestroyMaterial);
    else {
	(void) TreeSetIterator (problem.material_tree, UpdateMaterial);
	(void) TreeIterate (problem.material_tree);
    }

    DestroyProblem (NULL);
/*
    ZeroOffset (element);
    Deallocate (element);
    ZeroOffset (node);
    Deallocate (node);
*/
    problem = saved;

    if (active_material == NULL)
	active_material = TreeMinimum (problem.material_tree);

    GatherMaterialNames ( );
    changeflag = True;
}


static FILE *fp;

static int WriteMaterial (item)
    Item item;
{
    Material material = (Material) item;


    if (strpbrk (material -> name, "- \t\n=[]\",+*/()#"))
	fprintf (fp, "\"%s\"  ", material -> name);
    else
	fprintf (fp, "%s  ", material -> name);

    fprintf (fp, "E=%g A=%g ", material -> E, material -> A);
    fprintf (fp, "Ix=%g Iy=%g ", material -> Ix, material -> Iy);
    fprintf (fp, "Iz=%g J=%g ", material -> Iz, material -> J);
    fprintf (fp, "G=%g nu=%g ", material -> G, material -> nu);
    fprintf (fp, "t=%g rho=%g ", material -> t, material -> rho);
    fprintf (fp, "kappa=%g\n", material -> kappa);
    return 0;
}


void WriteMaterialFile ( )
{
    String 	ans;
    static char suggestion [256] = "";
    unsigned    selected;


    SetShellTitle (file_dialog -> shellwidget, "Update Material Database");
    selected=PopupDialog (file_dialog, "Database to update:", suggestion, &ans);
    (void) strcpy (suggestion, ans);

    if (selected != Okay || !ans [0])
	return;

    if ((fp = fopen (ans, "w")) == NULL) {
	error ("Unable to update file %s.\n", ans);
	return;
    }

    fprintf (fp, "material properties\n");
    (void) TreeSetIterator (problem.material_tree, WriteMaterial);
    (void) TreeIterate (problem.material_tree);
    fprintf (fp, "\nend\n");
    (void) fclose (fp);
}
