/* parse.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1995 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 */


/*
 * Code to parse a fairly general filter which allows one level of parenthesis,
 * AND, OR, and ~, EG:
 *   TO Helene OR (FROM Helene AND TO yeager) AND ~DELETED 
 * The evaluation is left to right. 
 *     Bill Yeager -- February 19, 1993 ... 
 *
 *                 
 */

#include "ml.h"

static LEAF default_petals[]= {
  "DELETED",				/* predicate */
  NO_OP,				/* predeceding operator  */
  simple,				/* expression type */
  no_binding,				/* NOT binding with predicate */
  no_binding,				/* NOT binding with expression */
  pred_simple,				/* simple, field, dynamic */
  local,				/* local or server search */
  NULL,					/* next - leaf in tree */

  "ANSWERED", 
  NO_OP, 
  simple, 
  no_binding, 
  no_binding, 
  pred_simple,
  local,				/* local or server search */
  NULL,


  "FLAGGED", 
  NO_OP, 
  simple, 
  no_binding, 
  no_binding, 
  pred_simple,
  local,				/* local or server search */
  NULL,

  "UNSEEN", 
  NO_OP, 
  simple, 
  no_binding, 
  no_binding, 
  pred_simple,
  local,				/* local or server search */
  NULL,

  "OLD", 
  NO_OP, 
  simple, 
  no_binding, 
  no_binding, 
  pred_simple,
  local,				/* local or server search */
  NULL,

  "RECENT", 
  NO_OP, 
  simple, 
  no_binding, 
  no_binding, 
  pred_simple,
  local,				/* local or server search */
  NULL,

  "NEW", 
  NO_OP, 
  simple, 
  no_binding, 
  no_binding, 
  pred_simple,
  local,				/* local or server search */
  NULL,


  NULL, NO_OP, simple, no_binding, no_binding, pred_simple, local, NULL
};



FLAG_TO_PARSE_TYPE default_types[]= {
  "DELETED",  TYPE_DELETED,
  "ANSWERED", TYPE_ANSWERED,
  "FLAGGED",  TYPE_FLAGGED,
  "UNSEEN",   TYPE_UNSEEN,
  "OLD",      TYPE_OLD,
  "RECENT",   TYPE_RECENT,
  "NEW",      TYPE_NEW,
  NULL,       0,  
};





/* Note: Strings are UPPERCASE:
 * "Token", Field required, field attributes, parenthesis, NOT binding*/

static STOKEN search_tokens[]= {
  "TO",					/* Token string */
  TRUE,					/* True if field required */
  APPLYSEARCH,					/* attributes of field */
  zero_paren,				/* dynamic - set during parsing */
  no_binding,				/* dynamic - how "not" binds token */
  no_binding,				/* dynamic - how "not" bind express. */
  TYPE_NULL,				/* token search types */
  local,				/* search type: local / server */
  
  "FROM",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "CC",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  CORRESPONDENT_MACRO,
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  RECIPIENT_MACRO,
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "SUBJECT",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "TEXT",
  TRUE,
  NONE,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "HEADER",
  TRUE,
  NONE,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "BODY",
  TRUE,
  NONE,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,


  "MESSAGE-ID",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,


  "SENDER",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "REPLY-TO",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "IN-REPLY-TO",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "RETURN-PATH",
  TRUE,
  APPLYSEARCH,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,
/*
 *  "REMAIL",
 *  TRUE,
 *  NONE,
 *  zero_paren,
 *  no_binding,
 *  no_binding,
 *  TYPE_NULL,
 *  local,
 */
  "KEYWORD",
  TRUE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_KEYWORD,
  local,

  "SINCE",
  TRUE,
  LE_TEMPS,				/* TIME attribute */
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "BEFORE",
  TRUE,
  LE_TEMPS,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,

  "ON",
  TRUE,
  LE_TEMPS,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NULL,
  local,


  ">", 
  TRUE,
  ATOMIC+NUMERIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_UNFLAGGED,
  local,

  /* Now the flags - NO field which is "vacuously" ATOMIC*/
  "NEW",
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_NEW,
  local,

  "OLD", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_OLD,
  local,

  "DELETED", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_DELETED,
  local,

  "UNDELETED", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_UNDELETED,
  local,

  "SEEN", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_SEEN,
  local,

  "UNSEEN", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_UNSEEN,
  local,

  "RECENT", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_RECENT,
  local,

  "ANSWERED", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_ANSWERED,
  local,

  "UNANSWERED", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_UNANSWERED,
  local,


  "FLAGGED", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_FLAGGED,
  local,

  "UNFLAGGED", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_UNFLAGGED,
  local,

  "REMAILED", 
  FALSE,
  ATOMIC,
  zero_paren,
  no_binding,
  no_binding,
  TYPE_UNFLAGGED,
  local,

  NULL, 
  FALSE, 
  0L, 
  zero_paren, 
  no_binding, 
  no_binding, 
  TYPE_NULL,
  undefined,
};

BLIST bool_list[]= {
  /* The boolean tokens */
  "AND", AND_OP,
  "OR", OR_OP,
  NULL, NO_OP,
};



ACTIONREC actlist[] = {
  "DELETE",   DELETE_ACT,   FALSE, FALSE, TRUE,
  "EXPUNGE",  EXPUNGE_ACT,  FALSE, FALSE, TRUE,
  "COPY",     COPY_ACT,     FALSE, TRUE,  TRUE,
  "MOVE",     MOVE_ACT,     FALSE, TRUE,  TRUE,
  "SAVE",     SAVE_ACT,     TRUE,  TRUE,  TRUE,
  "SHELL",    SHELL_ACT,    TRUE,  TRUE,  TRUE,
  "FLAG",     FLAG_ACT,     FALSE, FALSE, TRUE,
  "KEYWORD",  KEYWORD_ACT,  FALSE, TRUE,  FALSE,
  "REMAIL",   REPLY_ACT,    FALSE, TRUE,  FALSE,
  "PRINT",    PRINT_ACT,    TRUE,  FALSE, TRUE,
  "SELECT",   SELECT_ACT,   FALSE, FALSE, TRUE,
  "NEW",      NEW_ACT,      FALSE, FALSE, TRUE,
  "UNSELECT", UNSELECT_ACT, FALSE, FALSE, TRUE,
  "READ",     READ_ACT,     FALSE, FALSE, TRUE,
  NULL,       NO_ACT,       FALSE, FALSE, FALSE,
};



HEADERREC hdrlist[] = {
  "NOHEADER", HEADER_NONE,
  "PARTHEADER", HEADER_PART,
  "FULLHEADER", HEADER_FULL,
  NULL, HEADER_UNKNOWN,
};


ACTIONREC *find_action(s)
     char *s;
{

  ACTIONREC *a;

  for(a = actlist; (a->name != NULL); a ++ ) {
    if(strncasecmp(s,a->name,strlen(a->name)) == 0)
      return(a);
  }
  return(NULL);

}

HEADERREC *get_hdrlist(s)
     char *s;
{
  HEADERREC *h;
  for(h = hdrlist; (h->name != NULL); h ++ ) {
    if(strncasecmp(s,h->name,strlen(h->name)) == 0)
      return(h);
  }

  return(NULL);

}


void free_actions(ss)
     char ** ss;
{
  int n;
  for(n = 0; ss[n] != NULL; n ++)
    fs_give((void **) &(ss[n]));
  fs_give((void **) ss);

}


char **get_actions(s) 
     char *s;
{


  int len, n;
  char *ptr = s;
  char *endptr;
  char *zeroptr;
  char **ret = NULL;
  char buf[FILEBUFFLEN];

  len = count_separators(s);
  if(!len)
    return(NULL);
  ret = (char **) fs_get((len + 1) * sizeof(char *));
  n = 0;
  while(*ptr != '\0') {
    while(isspace(*ptr))
      ptr ++;
    endptr = strchr(ptr,ACTION_SEPARATOR);
    if(endptr == NULL)
      break;
    strncpy(buf,ptr,(int) (endptr - ptr));
    buf[endptr-ptr] = '\0';
    for(zeroptr = &buf[strlen(buf)-1]; (isspace(*zeroptr)); zeroptr --)
      *zeroptr = '\0';
    if(! strlen(buf))
      break;
    ret[n] = cpystr(buf);
    n ++;
    ptr = (++endptr);
  }
  ret[n] = NULL;

  return(ret);
}


int count_separators(s)
     char *s;
{
  int n = 0;
  char *p;
  for(p = s; (*p) ; p ++ )
    if(*p == ACTION_SEPARATOR)
      n ++;
  return(n);
}



/* Warning: we dispose of the calling string in this call */

char *clean_and_parse_action(s)
     char *s;
{
  char buffer[FILEBUFFLEN];
  char warn[FILEBUFFLEN];
  char *ptr = s;
  char *ret = NULL;
  char *end_of_arg = NULL;
  int errors = 0;
  char *qptr = NULL;
  ACTIONREC *a = NULL;
  HEADERREC *h = NULL;
  *buffer = '\0';

  while(*ptr) {
    while(isspace(*ptr))
      ptr ++;
    a = find_action(ptr);
    if(a == NULL || a->implemented == FALSE) {
      if(strlen(ptr)) {
	end_of_arg = strpbrk(ptr," \t\n;");
	if(end_of_arg)
	  *end_of_arg = NUL_TERM;
	sprintf(warn,"Unknown action: \"%s\"",ptr);
	mm_log(warn,WARN);
	errors ++;
      }
      buffer[strlen(buffer)] = NUL_TERM;
      break;
    }
    else {
      ptr += strlen(a->name);
      strcat(buffer,a->name);
      strcat(buffer," ");
      if( a->arg == FALSE && a->option == FALSE ) {
	ptr = strchr(ptr,(int) ACTION_SEPARATOR);
	if(ptr) {
	  strcat(buffer,"; ");
	  ptr ++;
	  continue;
	}
	else {
	  strcat(buffer, "; ");
	  break;
	}
      }
      if(a->option == TRUE ) {
	while(isspace(*ptr))
	  ptr ++;
	h = get_hdrlist(ptr);
	if(h) {
	  ptr += strlen(h->name);
	  strcat(buffer,h->name);
	  strcat(buffer," ");
	}
	else
	  strcat(buffer,"PARTHEADER ");
      }
      if(a->arg == FALSE) {
	ptr = strchr(ptr,(int) ACTION_SEPARATOR);
	if(ptr) {
	  strcat(buffer,"; ");
	  ptr ++;
	  continue;
	}
	else {
	  strcat(buffer, "; ");
	  break;
	}
      }
      while(isspace(*ptr))
	ptr ++;
      if((! *ptr) || (*ptr == ACTION_SEPARATOR)) {
	sprintf(warn,"ARGUMENT required for %s",a->name);
	mm_log(warn,WARN);
	errors ++;
	break;
      }
      if(*ptr == '\"') {
	qptr = strchr(ptr+1,'\"');
	if(! qptr) {
	  mm_log("Quote requires matching end quote.",WARN);
	  errors ++;
	  break;
	}
	end_of_arg = strpbrk(qptr," \t\n;");
      }
      else
	end_of_arg = strpbrk(ptr," \t\n;");
      if(! end_of_arg) {
	strcat(buffer,ptr);
	strcat(buffer," ; ");
	break;
      }
      else {
	strncat(buffer,ptr,(int) (end_of_arg - ptr));
	strcat(buffer," ; ");
	ptr = end_of_arg;
	while((isspace(*ptr)) || (*ptr == ACTION_SEPARATOR))	
	  ptr ++;
	continue;
      }
    }
  }

  fs_give((void **) &s);
  if(errors) {
    ret = cpystr(EMPTYSTR);
    return(ret);
  }
  
  if(strlen(buffer))
    ret = cpystr(buffer);
  else
    ret = cpystr(EMPTYSTR);
  return(ret);
  
}



/* return pointer to last char in a string */
char *
last(str)
     char *str;
{
  int len = strlen(str);

  return &str[len - 1];
}

void
strip_not(str, tptr)
     char **str;
     STOKEN *tptr;
{
  if(strncasecmp((*str),"not-",4) == STRMATCH)
    (*str) += 4;
}

/* remove parens from a string */
void
strip_parens(str, ptype)
     char **str;
     PARENS_TYPE ptype;
{
  char *cp;

  switch (ptype) {
  case zero_paren:
    break;
  case left_paren:
    ++*str;				/* skip this char */
    break;
  case right_paren:
    cp = last(*str);
    *cp = '\0';				/* null it out */
    break;
  default:				/* left and right */
    ++*str;				/* skip both parens */
    cp = last(*str);
    *cp = '\0';
    break;
  }
}

/* return paren type of a string. It may begin and end in a parenthesis */
PARENS_TYPE
which_parens(str)
     char *str;
{
  PARENS_TYPE ptype;
  char *cp;

  ptype = zero_paren;
  if (*str == '(') {
    ptype = left_paren;
  }
  if (*(cp = last(str)) == ')') 
    if (ptype == left_paren)
      ptype = both_paren;
    else
      ptype = right_paren;
  return ptype;
}
/*
 * num_parens returns 0, 1, or 2 as a function of ptype */
int
num_parens(ptype)
     PARENS_TYPE ptype;
{
  switch (ptype) {
  case zero_paren:
    return 0;
  case left_paren:
  case right_paren:
    return 1;
    break;
  default:				/* left and right */
    break;
  }
  return 2;
}
/* Match str to a token in tlist.
 * Return a STOKEN pointer, or NULL */

STOKEN *
found_token(tlist, token, err)
     STOKEN tlist[];
     char *token;
     parse_errors *err;
{
  int i;
  PARENS_TYPE ptype;
  char *cp;
  char str[PARSEBUFLEN], *strptr= str;
  Boolean not;
  Boolean before, after;

  strcpy(str, token);			/* local copy */

  if (strncasecmp(str,"not-",4) == STRMATCH) {
    before = TRUE;
    not = TRUE;
    strptr += 4;
  } else {
    not = FALSE;
    before = FALSE;
  }
  ptype = which_parens(strptr);		/* now parens check */
  switch (ptype) {
  case zero_paren:
    break;
  case left_paren:
    ++strptr;				/* skip this char */
    break;
  case right_paren:			
    cp = last(strptr);
    *cp = '\0';				/* null it out */
    break;
  default:				/* left and right */
    ++strptr;				/* skip both parens */
    cp = last(strptr);
    *cp = '\0';
    break;
  }

  if (strncasecmp(strptr,"not-",4) == STRMATCH) {
    after = not = TRUE;
    strptr += 4;
  } else
    after = FALSE;

  for (i = 0; tlist[i].token; ++i)
    if (strcasecmp(strptr, tlist[i].token) == 0) {
      tlist[i].parens = ptype;
      if (tlist[i].field_required == TRUE && ptype == right_paren) {
	*err = illegal_right_paren_token;
	return NULL;			/* <token>) <field> illegal */
      }
      /* fix our not binding for this token */
      if (not) {
	if (before) { /* ~exp OR ~(..) */
	  if (ptype == left_paren)
	    tlist[i].expression_not = bind_expression; /* ~( */
	  else {
	    tlist[i].expression_not = no_binding; /* ~exp */
	    tlist[i].atomic_not = bind_atom;
	  }
	} else				/* NOT before */
	    tlist[i].expression_not = no_binding; /* (<exp> */
	if (after) {			/* ~(~ OR (~ */
	  tlist[i].atomic_not = bind_atom;
	}
	/* The weird case of ~(~ATOMIC_EXPRESSION) == ATOMIC_EXPRESSION */
	if (ptype == both_paren && before && after) {
	  tlist[i].atomic_not = no_binding;
	}
      } else
	tlist[i].atomic_not = tlist[i].expression_not = no_binding;
      return &tlist[i];
    }
  *err = illegal_token;
  return NULL;
}

BOOL
found_boolean(blist, str)
     BLIST blist[];
     char *str;
{
  int i;
  
  for (i = 0; blist[i].operator; ++i)
    if (strcasecmp(str, blist[i].operator) == STRMATCH) {
      return blist[i].boole;
    }
  return NO_OP;
}
     
/* make <token> or <token> "<field>" as a string */

char *
make_predicate(token, field, needs_field, attributes, wild, wcdone)
     char *token, *field;
     Boolean needs_field;
     unsigned long attributes;
     Boolean wild;			/* ?insert wild card FLAG in field */
     Boolean *wcdone;
{
  int len;
  char *string;
  Boolean insert_wc;
  Boolean quoted;

  if (!needs_field) {
    string = cpystr(token);
    *wcdone = FALSE;
    return string;
  }
  /* need space for NULL SPACE ^C and possibly "" */
  len = strlen(token) + strlen(field) + 8; /* we may add '"''s */
  string = (char *) fs_get(len);
  insert_wc = (wild && (attributes & APPLYSEARCH) && *field != SAUVAGE);
  *wcdone = insert_wc;

  quoted = (field[0] == '"');
  if (quoted || attributes & ATOMIC) 
    sprintf(string,"%s %s", token, field);
  else 
    sprintf(string,"%s \"%s\"", token, field);
  return string;
}

/* place null at end of field, and return pointer to beginnig
   of the field which may skip a '('.
   n_excess will count surrounding parenthesis and any whitespace
   we step on to terminate the field .*/
char *
set_form_end(field, n_excess, err)
     char *field;
     int *n_excess;
     parse_errors *err;
{
  char c, *cp= field, *cp0= field;

  if (*cp == '(') {
    ++cp;				/* skip parens */
    ++cp0;
    *n_excess = 1;
  } else
    *n_excess = 0;

  if (*cp == '"') {			/* search for '"' */
    ++cp;				/* skip initial '"' */
    while (c = *cp++) {
      if (c == '"') {
	if (*cp == ')') {		/* "..") is legal */
	  ++cp;
	  ++*n_excess;			/* parenthesis are excess */
	}
	*cp = '\0';			/* tie it off */
	return cp0;			/* found our matching " */
      }
    }
    /*
     * We should never reach this error because "" parsing is done
     * before we parse for structure detail */
    *err = missing_dquote;
    return NULL;			/* This should never happen */
  }
  /* OK, search for whitespace or a terminating ')' */
  while (c = *cp) {
    if (c == ' ' || c == '\t' || c == ')') {
      if (c == ')') {
	++cp;				/* don't step on it. */
	++*n_excess;
	if (*cp != ' ' && *cp != '\t' && *cp != '\0') {
	  /* MUST have ') ' or end of string */
	  *err = illegal_right_paren_termination;
	  return NULL;
	}
      }
      if (*cp != '\0') {		/* end of field here */
	*cp = '\0';			/* tie it off */
	++n_excess;			/* stepped on white-space */
      }
      break;
    }
    ++cp;
  }
  return cp0;
}

Boolean
numeric_field(field, err)
     char *field;
     parse_errors *err;
{
  int i;

  if (*field == '\"') ++field;
  i = atoi(field);
  if (i <= 0) {
    *err = illegal_numeric_field;
    return FALSE;
  }
  return TRUE;
}

/* select the field following a token. 
   o (<field>) and <field>) are legal.
   o (<field>  is not legal.
   If parenthesisa are included in a field, then the field must be of
   the form "<expression>". */

Boolean
select_field(fptr, tptr, field, len_trouve, err)
     SFIELD *fptr;
     STOKEN *tptr;
     char *field;
     int *len_trouve;
     parse_errors *err;
{
  char c;
  int loc_len= 0;
  char *loc_str, *str= field;
  PARENS_TYPE ptype;
  char *cp;
  
  /* Anything left? */
  if (!*field) {
    *err = missing_field;
    return FALSE;
  }

  /* check the parenthesis type. loc_str may be adjusted past a
   * leading "(" or leading ^C */
  loc_str = set_form_end(str, &loc_len, err); 
  if (!loc_str) 
    return FALSE;
  ptype = which_parens(str);		/* must include all parens */
  switch (ptype) {
  case zero_paren:
    fptr->parens = zero_paren;
    break;
  case left_paren:
    *err = illegal_left_paren_field;
    return FALSE;
  case right_paren:
    fptr->parens = right_paren;		/* <token> <field>) terminates */
    break;
  case both_paren:			/* left and right */
    fptr->parens = both_paren;		/* nullifly one another */
    break;
  }

  /* Is it a \". Note that any surrounding parenthesis have
     already been counted in set_form_end(). */
  if (*loc_str == '"') {			/* quoted field */
    ++loc_str;
    loc_len += 1;
    while (c = *loc_str++) {		/* cherche '"' termination */
      loc_len += 1;
      if (c == '"') {
	*len_trouve = loc_len;
	fptr->field = str;
	return TRUE;
      }
    }
    *err = missing_dquote;
    return FALSE;
  }
      
  /* Take next variable, ie, break on white-space or end-of-string */
  while (c = *loc_str++)
    if (c == ' ' || c == '\t' || c == ')') break;
    else 
      loc_len += 1;
  *len_trouve = loc_len;
  fptr->field = str;
  return TRUE;
}


/* Don't allow our sauvage caractere! */
Boolean
all_chars_ok(filter)
     char *filter;
{
  return TRUE;
}


/*
 * Here one searches for properly balanced [] for wild matching,
 * Ah oui, on est bien dechaine!!
 *
 * Given "[" one must find a terminating "]". 
 * Note that [[[[[] is legal but [[[[[ is not.
 * Also, [\[] is legal as is [^[] */
Boolean
  brackets_balanced(filter)
char *filter;
{
  char c, *f= filter;
  int ng= 0;
  Boolean ignore= FALSE;
  
  while (c = *f++) {
    switch (c) {
    case '\\': 
      ignore = TRUE;
      break;
    case '^': 
      if (ng) ignore = TRUE;
      break;
    case '[':
      if (!ignore)
	++ng;
      ignore = FALSE;
      break;
    case ']':
      if (!ignore)
	ng = 0;
      ignore = FALSE;
      break;
    default:
      ignore = FALSE;
      break;
    }
  }
  if (ng) return FALSE;
  else
    return TRUE;
}
 
/* Here is a dumb paranthesis check that makes sure we have:
   o balanced right and left parenthesis
   o () is illegal
   o Only a depth of 1 is allowed, ie (..( .. ) ..) is illegal.
   o skip "<expression>"  */
Boolean
parenthesis_balanced(filter)
     char *filter;
{
  int pcnt= 0;
  char c;
  int n_not= 0;

  while (c = *filter++) {
    switch (c) {
    case '"':				/* scan until '"' */
      ++n_not;
      while (c = *filter++) {
	++n_not;			/* not a parenthesis */
	if (c == '"') {			/* skipped "<expression>" */
	  break;
	}
      }
      if (!c) --filter;			/* '"' not found, terminate the loop */
      break;
    case '(':
      ++pcnt;
      if (pcnt > 1)
	return FALSE;			/* too deep */
      n_not = 0;
      break;
    case ')':
      --pcnt;
      if (pcnt < 0)
	return FALSE;			/* too many rights */
      else
	if (n_not == 0)	     
	  return FALSE;			/* () NULLE is illegal */
      n_not = 0;
      break;
    default:
      ++n_not;				/* not one of the above */
      break;
    }
  }
  if (pcnt != 0)
    return FALSE;			/* not balanced */
  else
    return TRUE;
}


void fix_nots(str)
     char *str;
{

#ifdef NEVER

  char *src, *dst, c;
  int n_nots= 0;

  src = dst = str;

  while (*src) {
    if (*src == '"') {
      *dst++ = *src++;

      while (*src) {	  /* copy quoted strings */
	*dst++ = *src++;
	if (*src == '"')
	  break;
      }
      if (*src == NUL_TERM) 
	break;			  /* all done  */
      else
	continue;
    }

    if (strncasecmp(src,"not",3) == STRMATCH) {
      src += 3;
      *dst = ' '; dst ++;
      *dst = 'N'; dst ++;
      *dst = 'O'; dst ++;
      *dst = 'T'; dst ++;
      *dst = ' '; dst ++;
      while((*src != NUL_TERM) && (isspace(*src)))
	src ++;
      continue;
    }
    
    *dst++ = *src;
  }

  *dst = NUL_TERM;
#endif /* NEVER */

}

/* fix_whitespace:
 *   Consectutive whitespace chars made into a ' ',
 *   and "( " becomes "(" and " )" becomes ")" 
 */

int fix_whitespace(str)
     char *str;
{
  char *src, *dst;
  char c;
  int ws= 1;				/* to clear leading whitespace */
  Boolean gp;				/* gauche (left) parenthesis */
  int parens= 0;
  int cnt;

  src = dst = str;
  /* first compact all whitespace */
  while (c = *src++) {
    if (c == '\n')
      continue;				/* skip RETURN */
    if (c == ' ' || c == '\t') {
      ws += 1;				/* count whitespace */
      if (ws > 1) continue;		/* skip it */
      c = ' ';				/* use SPACE */
    } else
      ws = 0;
    *dst++ = c;
    if (c == ')' || c == '(')
      ++parens;
  }
  *dst = '\0';
  if (parens == 0)
    return 0;
  /* Now check for "( " or " )" */
  src = dst = str;
  gp = FALSE;
  cnt = 0;
  while (c = *src++) {
    ++cnt;
    switch (c) {
    case '"':
      *dst++ = c;
      while (c = *src++) {
	*dst++ = c;			/* cueiller these chars. */
	if (c == ')' || c == '(')
	  --parens;			/* does not count within ".." */
	else if (c == '"') {		/* skipped "<expression>" */
	  break;
	}
      }
      if (!c) --str;			/* '"' not found, terminate the loop */
      break;
    case '(': 
      gp = TRUE; *dst++ = c;
      break;
    case ')': 
      if (cnt >= 2 && *(src - 2) == ' ') { /* then " )" */
	--dst;				/* write over the SPACE */
      } 	
      *dst++ = c;			/* keep the guy */
      break;
    default:
	if (gp) {			/* previous "(" */
	  gp = FALSE;
	  if (c == ' ')			/* Found "( " */
	    continue;			/* skip the SPACE */
	}
      *dst++ = c;			/* keep the CHAR */
      break;
    }
   }
  *dst = '\0';
  return parens;
}

/* refuse blank names */
Boolean whitespace_name(str)
     char *str;
{
  char c;

  while (c = *str++)
    if (c != ' ' && c != '\t')
      return FALSE;
  return TRUE;
}
/*
 * set leaf state given parenthesis type of token and field
    <token>            -->                 SIMPLE_EXPRESSION
   (<token>)           -->                 SIMPLE_EXPRESSION
   (<token>) (<field>) --> <token> <field> SIMPLE_EXPRESSON 
   (<token>   <field>) --> <token> <field> SIMPLE_EXPRESSON 
   <token>   (<field>) --> <token> <field> SIMPLE_EXPRESSON 

   (<token>   <field>  -->                 START_EXPRESSION
    <token>   <field>) -->                 END_EXPRESSION 

   *** Others illegal and illiminated in parse ***
 */
expression_type
set_feuille_state(token_ptype, field_ptype)
     PARENS_TYPE token_ptype,
                 field_ptype;
{
  switch (token_ptype) {
  case both_paren:			/* (<expression>) */
  case zero_paren:			/*  <expression> */
    break;				/* field determines result */
  case left_paren:
    if (field_ptype == right_paren)
      return simple;
    else
      return start;
  case right_paren:			/* <token>) with no field */
    return end;
  }
  /* "case left_paren:" is impossible */
  switch (field_ptype) {
  case both_paren:			/* (<expression>) */
  case zero_paren:			/*  <expression> */
    return simple;
  case right_paren:
    return end;
  }
}

/*
 * add a new leaf to our tree. */

Boolean
add_leaf_to_tree(arbe, ctoken, cfield, operator, tptr, fptr, sauvage)
     LEAF **arbe;
     char *ctoken, *cfield;
     BOOL operator;
     STOKEN *tptr;
     SFIELD *fptr;
     Boolean sauvage;
{
  LEAF *flle= (LEAF *)fs_get(sizeof(LEAF));
  LEAF *vert;
  Boolean need_end;
  Boolean wc_inserted;

  flle->predicate = make_predicate(ctoken, cfield, 
				   tptr->field_required,
				   tptr->field_attributes,
				   sauvage, &wc_inserted);
  flle->operator = operator;
  flle->expression = set_feuille_state(tptr->parens, fptr->parens);
  flle->atomic_not = tptr->atomic_not;
  flle->expression_not = tptr->expression_not;

  /* Set the predicate type to simple (no field required) OR field */
  flle->predicate_type = (tptr->field_required ? pred_field : pred_simple);

  /* some predicate fields are dynamic at search time */
  if (flle->predicate_type == pred_field &&
      tptr->field_attributes & LE_TEMPS)
    flle->predicate_type = pred_dynamic_field;
  flle->search_type = tptr->search_type;

  /* Adjoin the leaf to the tree */
  if (!*arbe) {
    *arbe = flle;			/* the tete (head) */
  } else {				/* find last leaf */
    vert = *arbe;
    if (vert->expression == start)	/* current expression type */
      need_end = TRUE;
    else
      need_end = FALSE;
    while (vert->next) {
      vert = vert->next;
      switch (vert->expression) {
      case simple:
      case intermediate:
	break;
      case start:
	need_end = TRUE;
	break;
      case end:
	need_end = FALSE;
	break;
      }
    }
    /* ajouter cette feuille - to add this leaf */
    vert->next = flle;	
    /* if flle is in a chain with a start and no end,
     * then flle is intermediate */
    if (need_end && flle->expression != end)
      flle->expression = intermediate;
  }
  /* terminer l'arbe - to terminate the tree */	
  flle->next = NULL;		
  return wc_inserted;
}

/*
 * dispose of the tree */
void
cut_down_the_tree(leafs)
     LEAF *leafs;
{
  LEAF *feuille;

  /* dispose of the tree */
  while (leafs) {
    feuille = leafs;
    fs_give((void **) &feuille->predicate);	/* the string */
    leafs = feuille->next;		/* next leaf */
    fs_give((void **) &feuille);			/* the structure */
  }
}
/*
 * create IMAP search filter from the tree's leaf's */
void
move_leafs_to_filter(search_tree, search_text, search_types,
		     leafs, 
		     text, 
		     token_types)
     LEAF **search_tree;
     char **search_text;
     unsigned long *search_types;
     LEAF *leafs;
     char *text;
     unsigned long token_types;
{
  /* the search_filter.name is set up by the caller to lv_parse_filter */
  *search_tree = leafs;
  *search_text = text;
  *search_types = token_types;
}    

/* augment the filter by len */    
char *
advance_filter(filter, filter_len, len)
     char *filter;
     int  *filter_len, len;
{
  *filter_len -= len;			/* decrement by string length  */
  if (*filter_len > 1) {			/* strtok NULLS out one char */
    filter += len + 1;			/* next list item */
    *filter_len -= 1;			/* advance one char more */
  } else
    filter += len;			/* points to NULL termination */
  return filter;
}

/*
 * Here we parse the filter text provided by the user
 *
 * Note: embedded ^C in strings coerces wildcard searching on
 * that string ...
 * Returns: parse_success if parse does not fail. Otherwise an ERROR */

#define TOKEN_STOPS " \t"
#define ERRMSGLEN 32

parse_errors
lv_parse_filter(name_text, filter_text, 
		search_tree, 
		search_text, 
		search_types,
		errmsg)
     char *name_text, *filter_text;
     LEAF **search_tree;
     char **search_text;
     unsigned long  *search_types;
     char *errmsg;			/* for returning extra info */
{
  char *search_string;			/* destination  */
  STOKEN *tokens= search_tokens;	/* our tokens */
  char *filter= filter_text;		/* source to parse */
  char *ctoken;				/* current token */
  STOKEN *tptr;				/* token block  */
  SFIELD field;				/* for a field */
  int filter_len;			/* chars in filter */
  int field_len;
  int other_len;
  parse_errors err, rval;		/* in places, used for error return */
  BOOL operator= NO_OP;			/* AND_OP / OR_OP / NO_OP */
  LEAF *arbe= NULL;			/* the tree which grows on C street */
  int next_term= TOKEN;			/* initially a TOKEN */
  int parsing;				/* counts terms */
  Boolean looping= TRUE;		/* keeps it going */
  int n_parens;				/* counts parens found */
  unsigned long token_types;		/* marks tokens in search_text */

  /* Make sure the name is not white-space */
  if (whitespace_name(name_text)) {
    return invalid_name;
  }

  fix_nots(filter);			/* clean up ~'s */
  n_parens = fix_whitespace(filter);	/* clear unnecessary whitespace */
  if (n_parens > 0 && !parenthesis_balanced(filter)) /* check parantheticals */
    return parenthesis_bad;
  /* Make sure the "[]" balance for wild card matching */
  if (!brackets_balanced(filter))
    return brackets_bad;
  filter_len = strlen(filter);		/* what's is there */
  operator = NO_OP;
  search_string = cpystr(filter);	/* save a copy */
  /* The string may be prefixed for the WIZARD ... */
   if (*filter == SAUVAGE) ++filter;   /* Le Wizard! Ah bon? */

  *errmsg = '\0';			/* no message yet */
  token_types = TYPE_NULL;		/* none yet */
  for (parsing = 0; looping; ++parsing) {
    /*
     * Here we expect to find a predicate like "FROM" or "DELETED" */
    if((strncasecmp(filter,"not ",4)) == STRMATCH)
      filter[3] = '-';
    ctoken = strtok(filter, TOKEN_STOPS); /* first token */
    if (!ctoken) {
      if (parsing == 0) {		/* First time through */
	err =  empty_filter;		/* no filter provided */
      } else
	if (next_term == GEORGES_BOOLE) { /* Final term a predicate */
	  err = no_error;
	} else
	  err =  missing_predicate;	/* No predicate after AND/OR */
      break;
    }
    if (strlen(ctoken) > PARSEBUFLEN) {	/* token too long */
      err = trop_de_texte;
      strncpy(errmsg, ctoken, ERRMSGLEN);
      errmsg[ERRMSGLEN] = '\0';
      strcat(errmsg, " ...");
      looping = FALSE;
      continue;
    }
    /* check for token or Boolean operator */
    switch (next_term) {
    case TOKEN:
      /* select a token from our tokens[] */
      if (!(tptr = found_token(tokens, ctoken, &err))) {
	looping = FALSE;
	strcpy(errmsg, ctoken);
	continue;
      } else {
	other_len = strlen(ctoken);	/* includes parenthesis, not */
	strip_not(&ctoken);		/* strip not */
	strip_parens(&ctoken, tptr->parens); /* remove em. */
	strip_not(&ctoken);		/* remove possible 2nd not */
	token_types |= tptr->token_type; /* collect the token types */
 	next_term = GEORGES_BOOLE;	/* Boolean next en haut.*/
      }
      filter = advance_filter(filter, &filter_len, other_len);
      break;
    case GEORGES_BOOLE:
      /* We find the operator which binds the NEXT predicate with
       * it predecessors */
      operator = found_boolean(bool_list, ctoken);
      if (operator == NO_OP) {
	err =  expected_boolean;
	looping = FALSE;
	strcpy(errmsg, ctoken);
	continue;
      } else {
	other_len = strlen(ctoken);
	next_term = TOKEN;		/* token on next pass */
      }
      filter = advance_filter(filter, &filter_len, other_len);
      continue;
    }
    /* check for <field> */
    if (tptr->field_required) {
      /* On a besoin de comprendre les parenthesis ici:
       * (FIELD) OR FIELD) are legal - tuesday! */
      if (!select_field(&field, tptr, filter, &field_len, &err)) {
	looping = FALSE;		/* err set in select_field */
	strcpy(errmsg, ctoken);
	continue;
      }
      if (strlen(field.field) > PARSEBUFLEN) {	/* token too long */
	err = trop_de_texte;
	strncpy(errmsg, field.field, ERRMSGLEN);
	errmsg[ERRMSGLEN] = '\0';
	strcat(errmsg, " ...");
	looping = FALSE;
	continue;
      }
      strip_parens(&field.field, field.parens); /* remove em. */
      if (tptr->field_attributes & NUMERIC &&
	  !numeric_field(field.field, &err)) {
	looping = FALSE;
	strcpy(errmsg, field.field);
	continue;
      }
      filter_len -= field_len;		/* decrement chars remaining */
      filter += field_len;		/* skip field in string */
      *filter = '\0';		 	/* terminate predicate */
      if (filter_len > 0) {		/* then not at end */
	++filter;			/* to next term */
	--filter_len;			/* one less character */
      }
    } else {
      field.parens = zero_paren;	/* no parenthesis */
      field.field = "";			/* No field - filter is OK. */
    }
    /* place in our parsing tree */
    add_leaf_to_tree(&arbe, ctoken, field.field, operator, tptr, &field,
		     FALSE);
    tptr->parens = zero_paren;
  }
  /* chase down the tree and build the various search filters 
   *
   */
  if (err == no_error) {
    move_leafs_to_filter(search_tree, search_text, search_types,
			 arbe, 
			 search_string,
			 token_types);
    rval = parse_success;
  } else {
    cut_down_the_tree(arbe);		/* dispose of the tree */
    fs_give((void **) &search_string);		/* And the string */
    rval = err;
  }
  return rval;
}

/*
 * Identical too above except we are either editing or creating
 * a filter - We need this distinction for wildcard/substring
 * parsing, and to avoid twisted logic */

typedef struct _edit_list_element_ {
  char *cpos;
  struct _edit_list_element_ *next;
} EDIT_ELEMENT;

typedef struct _edit_list_ptr {
  int count;
  EDIT_ELEMENT *head;
  EDIT_ELEMENT *tail;
} EDIT_LPTR;


char *
force_sauvage(search_string)
     char *search_string;
{
  char *src= search_string;
  char *dst=  (char *)fs_get(strlen(src) + 2);
  char *nouveau= dst;
  char c;

  *dst++ = SAUVAGE;

  while (c = *src++) *dst++ = c;

  *dst = NIL;
  fs_give((void **) &search_string);

  return nouveau;
}
char *
edit_search_string(search_string, elptr, search_state)
     char *search_string;
     EDIT_LPTR *elptr;
     int search_state;
{
  EDIT_ELEMENT *e;
  char *dst, *nouveau, *src, c;

  src = search_string;
  nouveau = dst = (char *)fs_get(strlen(src) + elptr->count + 4);

  for (e = elptr->head; e; e = e->next) {
    while (e->cpos != src)		/* copy until cpos */
      *dst++ = *src++;
    *dst++ = SAUVAGE;			/* insert SAUVAGE */
    *dst++ = *src++;			/* Now the source char */
  }
  /* now what ever remains is inserted */
  while (c = *src++) *dst++ = c;
  *dst = NIL;

  /* Free things ... */
  fs_give((void **) &search_string);
  for (e = elptr->head; e; ) {
    EDIT_ELEMENT *tmp= e->next;

    fs_give((void **) &e);
    e = tmp;
  }
  return nouveau;
}
parse_errors
lv_edit_parse_filter(name_text, filter_text, 
		     search_tree, 
		     search_text, 
		     search_types,
		     sauvage,
		     search_state,
		     update,
		     errmsg)
     char *name_text, *filter_text;
     LEAF **search_tree;
     char **search_text;
     unsigned long  *search_types;
     Boolean sauvage;			/* TRUE if wildcard is applied. */
     int search_state;			/* Substring, Unix or Wizard */
     char **update;			/* for updated filter_text */
     char *errmsg;			/* for returning extra info */
{
  char *search_string;			/* destination  */
  STOKEN *tokens= search_tokens;	/* our tokens */
  char *filter= filter_text;		/* source to parse */
  char *ctoken;				/* current token */
  STOKEN *tptr;				/* token block  */
  SFIELD field;				/* for a field */
  int filter_len;			/* chars in filter */
  int field_len;
  int other_len;
  parse_errors err, rval;		/* in places, used for error return */
  BOOL operator= NO_OP;			/* AND_OP / OR_OP / NO_OP */
  LEAF *arbe= NULL;			/* the tree which grows on C street */
  int next_term= TOKEN;			/* initially a TOKEN */
  int parsing;				/* counts terms */
  Boolean looping= TRUE;		/* keeps it going */
  int n_parens;				/* counts parens found */
  unsigned long token_types;		/* marks tokens in search_text */
  EDIT_LPTR elptr;			/* list of post edit positions */
  char *filter0;			/* place holder */
  /* Make sure the name is not white-space */
  if (whitespace_name(name_text)) {
    return invalid_name;
  }

  fix_nots(filter);			/* clean up ~'s */
  n_parens = fix_whitespace(filter);	/* clear unnecessary whitespace */
  if (n_parens > 0 && !parenthesis_balanced(filter)) /* check parantheticals */
    return parenthesis_bad;
  /* Make sure the "[]" balance for wild card matching */
  if (!brackets_balanced(filter))
    return brackets_bad;
  if (!all_chars_ok(filter))
    return unprintable_chars;

  /*
   * For post rebuilding of the filter text */
  elptr.head = elptr.tail = NIL;
  elptr.count = 0;
  filter0 = filter;			/* keep scanning base */
  filter_len = strlen(filter);		/* what's is there */
  operator = NO_OP;
  search_string = cpystr(filter);	/* save a copy */
  *errmsg = '\0';			/* no message yet */
  token_types = TYPE_NULL;		/* none yet */
  for (parsing = 0; looping; ++parsing) {
    /*
     * Here we expect to find a predicate like "FROM" or "DELETED" */
    ctoken = strtok(filter, TOKEN_STOPS); /* first token */
    if (!ctoken) {
      if (parsing == 0) {		/* First time through */
	err =  empty_filter;		/* no filter provided */
      } else
	if (next_term == GEORGES_BOOLE) { /* Final term a predicate */
	  err = no_error;
	} else
	  err =  missing_predicate;	/* No predicate after AND/OR */
      break;
    }
    if (strlen(ctoken) > PARSEBUFLEN) {	/* token too long */
      err = trop_de_texte;
      strncpy(errmsg, ctoken, ERRMSGLEN);
      errmsg[ERRMSGLEN] = '\0';
      strcat(errmsg, " ...");
      looping = FALSE;
      continue;
    }
    /* check for token or Boolean operator */
    switch (next_term) {
    case TOKEN:
      /* select a token from our tokens[] */
      if (!(tptr = found_token(tokens, ctoken, &err))) {
	looping = FALSE;
	strcpy(errmsg, ctoken);
	continue;
      } else {
	other_len = strlen(ctoken);	/* includes parenthesis, not */
	strip_not(&ctoken);		/* strip not */
	strip_parens(&ctoken, tptr->parens); /* remove em. */
	strip_not(&ctoken);		/* remove possible 2nd not */
	token_types |= tptr->token_type; /* collect the token types */
 	next_term = GEORGES_BOOLE;	/* Boolean next en haut.*/
      }
      filter = advance_filter(filter, &filter_len, other_len);
      break;
    case GEORGES_BOOLE:
      /* We find the operator which binds the NEXT predicate with
       * it predecessors */
      operator = found_boolean(bool_list, ctoken);
      if (operator == NO_OP) {
	err =  expected_boolean;
	looping = FALSE;
	strcpy(errmsg, ctoken);
	continue;
      } else {
	other_len = strlen(ctoken);
	next_term = TOKEN;		/* token on next pass */



      }
      filter = advance_filter(filter, &filter_len, other_len);
      continue;
    }
    /* check for <field> */
    if (tptr->field_required) {
      /* On a besoin de comprendre les parenthesis ici:
       * (FIELD) OR FIELD) are legal - tuesday! */
      if (!select_field(&field, tptr, filter, &field_len, &err)) {
	looping = FALSE;		/* err set in select_field */
	strcpy(errmsg, ctoken);
	continue;
      }
      if (strlen(field.field) > PARSEBUFLEN) {	/* token too long */
	err = trop_de_texte;
	strncpy(errmsg, field.field, ERRMSGLEN);
	errmsg[ERRMSGLEN] = '\0';
	strcat(errmsg, " ...");
	looping = FALSE;
	continue;
      }
      strip_parens(&field.field, field.parens); /* remove em. */
      if (tptr->field_attributes & NUMERIC &&
	  !numeric_field(field.field, &err)) {
	looping = FALSE;
	strcpy(errmsg, field.field);
	continue;

      }
      filter_len -= field_len;		/* decrement chars remaining */
      filter += field_len;		/* skip field in string */
      *filter = '\0';		 	/* terminate predicate */
      if (filter_len > 0) {		/* then not at end */
	++filter;			/* to next term */
	--filter_len;			/* one less character */
      }
    } else {
      field.parens = zero_paren;	/* no parenthesis */
      field.field = "";			/* No field - filter is OK. */
    }
    /* place in our parsing tree:
     * Returns TRUE if a wildcard char was forced into the field */
    if (add_leaf_to_tree(&arbe, ctoken, field.field, operator, tptr, &field, 
			 sauvage)) {
      /* Push a char position on our list for later reconstruction of
       * the filter text */
      EDIT_ELEMENT *element= (EDIT_ELEMENT *)fs_get(sizeof(EDIT_ELEMENT));
      
      /* Save char position in search_string */
      element->cpos = search_string + (field.field - filter0);
      element->next = NIL;
      if (elptr.count == 0) 
	elptr.head = elptr.tail = element;

      else {
	elptr.tail->next = element;	/* link last with this one */
	elptr.tail = element;		/* new tail (toujours amusant) */
      }
      elptr.count += 1;			/* number edited */
    }
      
    tptr->parens = zero_paren;
  }
  /* chase down the tree and build the various search filters 
   *
   */
  if (err == no_error) {
    /* First, if necessary, remake our search string editing in
     * wild-card search flags where necessary */
    if (elptr.count > 0) {
      search_string = edit_search_string(search_string, &elptr, search_state);
      *update = search_string;
    }
    move_leafs_to_filter(search_tree, search_text, search_types,
			 arbe, 
			 search_string,
			 token_types);
    rval = parse_success;

  } else {
    cut_down_the_tree(arbe);		/* dispose of the tree */
    fs_give((void **) &search_string);		/* And the string */
    rval = err;
  }
  return rval;
}

