/* msgfncs.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1995 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

void update_view_line();


/* 
 * This takes a message header, extracts the subject, and builds a
 * "reply" subject out of it. It does these things after
 * looking for a beginning "re" string.
 * something              - add a "re" -> "Re: something"
 * Re: something          - up the level "Re[2]: something"
 * Re[n]: something       - keep upping it "Re[3]: something" etc.
 * Re: Re[n]: something   - They're not playing ball. Strip the initial
 *                          Re: and up the level by 2. Otherwise we'll
 *                          end up in a "Re:" war in a protracted 
 *                          conversation.
 *
 * Return string is allocated and must be free'd when no longer needed.
 */

#ifdef __STDC__
char *get_reply_subject(char *header)
#else
char *get_reply_subject(header)
     char *header;
#endif
{
  char *ret;
  char *subject;
  char *first_text;
  char *ptr;
  int replyno;
  Boolean not_playing = FALSE;

  if(header == NULL)
    return(cpystr(EMPTYSTR));

  subject = get_header_field_contents("subject", header);

  if(! subject)
    return(cpystr(DEFAULT_REPLY_SUBJECT));

  first_text = first_nonwhite(subject);
  if(*first_text == NUL_TERM) 
    return(cpystr(DEFAULT_REPLY_SUBJECT));
    
  ret = (char *) fs_get(strlen(first_text) + 32);

  if(strncasecmp(first_text,"re: re[",7) == STRMATCH) {
    not_playing = TRUE;
    first_text += 4;
  }


  if(strncasecmp(first_text,"re[",3) == STRMATCH) {
    replyno = atoi(first_text + 3);
    replyno ++;
    if(not_playing)
      replyno ++;
    ptr = strchr(first_text + 3,']');
    if(ptr) {
      ptr ++;
      sprintf(ret,"Re[%d]%s",replyno,ptr);
    }
    else {
      sprintf(ret,"Re: %s",first_text);
    }
  }
  else {
    if(strncasecmp(first_text,"re:",3) == STRMATCH) {
      sprintf(ret,"Re[2]:%s",first_text + 3);
    }
    else
      sprintf(ret,"Re: %s", first_text);
  }

  fs_give((void **) &subject);
  return(ret);

}

#ifdef __STDC__
char *get_forward_subject(char *header)
#else
char *get_forward_subject(header)
     char *header;
#endif
{
  char *ret;
  char *subject;

  if(header == NULL)
    return(cpystr(EMPTYSTR));

  subject = get_header_field_contents("subject", header);

  if(! subject)
    subject = cpystr(EMPTYSTR);

  ret = (char *) fs_get(strlen(subject) + 64);

  sprintf(ret,"Forwarded: [%s]",subject);
  ret[COMPOSEWIDTH - 1] = NUL_TERM;

  fs_give((void **) &subject);
  return(ret);

}

#ifdef __STDC__
char *get_short_forward_line(char *header, int count)
#else
char *get_short_forward_line(header, count)
     char *header;
     int count;
#endif
{
  char *ret;
  char *subject;
  char *from;
  if(header == NULL)
    return(cpystr(EMPTYSTR));

  from = get_header_field_contents("from", header);

  if(! from)
    from = cpystr(EMPTYSTR);

  subject = get_header_field_contents("subject", header);

  if(! subject)
    subject = cpystr(EMPTYSTR);

  ret = (char *) fs_get(strlen(from) + strlen(subject) + 64);

  sprintf(ret,"\t\t%3d %-20.20s [%-36.36s]\n",count, from, subject);

  fs_give((void **) &from);
  fs_give((void **) &subject);
  return(ret);

}





#ifdef __STDC__
char *get_forward_text(char *text)
#else
char *get_forward_text(text)
     char *text;
#endif
{
  char *ret;

  if(text == NULL)
    return(cpystr(EMPTYSTR));

  ret = (char *) fs_get(strlen(text) + 256);

  strcpy(ret,BEGIN_FORWARD);
  strcat(ret,text);
  strcat(ret,END_FORWARD);

  return(ret);

}


char *get_short_header(header)
     char *header;
{
  char *from;    
  char *to; 
  char *subject;
  char *date;
  char *ret;

  if(! header)
    return(cpystr(EMPTYSTR));
  from = get_formatted_header_field("from",header);
  if(from == NULL)
    from = cpystr(EMPTYSTR);
  to = get_formatted_header_field("to",header);
  if(to == NULL)
    to = get_formatted_header_field("newsgroups", header);
  if(to == NULL)
    to = cpystr(EMPTYSTR);
  subject = get_formatted_header_field("subject",header);
  if(subject == NULL)
    subject = cpystr(EMPTYSTR);
  date = get_formatted_header_field("date",header);
  if(date == NULL)
    date = cpystr(EMPTYSTR);

  ret = (char *) fs_get(strlen(from) + strlen(to) 
			+ strlen(subject) + strlen(date) + 1);
  *ret = NUL_TERM;
  strcat(ret,from);
  strcat(ret,to);
  strcat(ret,subject);
  strcat(ret,date);

  fs_give((void **) &from);
  fs_give((void **) &to);
  fs_give((void **) &subject);
  fs_give((void **) &date);

  return(ret);

}


/*
 * Matches "field" (case-insensitive, and no colon on input) in the
 * header, and returns the contents of the field, including continuation
 * lines, but without any LF's. The field name is not returned, only the 
 * contents.
 */

#ifdef __STDC__
char *get_header_field_contents(char *field, char *header)
#else
char *get_header_field_contents(field,header)
     char *field;
     char *header;
#endif
{
  char *ptr = header;
  unsigned long len   = strlen(field);
  unsigned long fieldlen = 0;
  Boolean match = FALSE;
  char *ret = NULL;
  char *firstwhite;


  if(! header)
    return(ret);

  while(*ptr) {
    if(((strncasecmp(ptr,field,len)) == STRMATCH) &&
       ((*(ptr + len)) == COLONCHAR)) {
      match = TRUE;
      ptr += len + 1;
      while(isspace(*ptr))
	    ptr ++;
      break;
    }
    else {
      while(*ptr) {
	if(*ptr == LFCHAR) {
	  ptr ++;
	  break;
	}
	ptr ++;
      }
    }
  }
  if(match == TRUE) {
    fieldlen = header_field_len(ptr);
    ret = (char *) fs_get(fieldlen + 1);
    strncpy(ret,ptr,fieldlen);
    ret[fieldlen] = NUL_TERM;
    striplf(ret);
  }
  return(ret);
}


/* 
 * Finds a given field in the header (case insensitive, no colon on input),
 * and produces a suitable for display representation, with a pseudo tab
 * stop after the field name is printed, but this is ignored if the field
 * name is bigger than the tab length.
 * The field contents includes continuation lines and all linefeeds. the only
 * formatting done is the initial pseudo tab.
 */ 

#ifdef __STDC__
char *get_formatted_header_field(char *field, char *header)
#else
char *get_formatted_header_field(field,header)
     char *field;
     char *header;
#endif
{
  char *ptr = header;
  unsigned long len   = strlen(field);
  unsigned long fieldlen = 0;
  Boolean match = FALSE;
  char *ret = NULL;
  char *firstwhite;

  if(! header)
    return(ret);

  while(*ptr) {
    if(((strncasecmp(ptr,field,len)) == STRMATCH) &&
       ((*(ptr + len)) == COLONCHAR)) {
      match = TRUE;
      break;
    }
    else {
      while(*ptr) {
	if(*ptr == LFCHAR) {
	  ptr ++;
	  break;
	}
	ptr ++;
      }
    }
  }

  if(match == TRUE) {
    fieldlen = header_field_len(ptr);
    ret = (char *) fs_get(fieldlen + (2 * HEADER_TAB_LEN));
    firstwhite = strchr(ptr,COLONCHAR);
    firstwhite ++;
    if((firstwhite - ptr) < HEADER_TAB_LEN) {
      strncpy(ret,ptr,firstwhite - ptr);
      ret[firstwhite - ptr] = NUL_TERM;
      strncat(ret,BLANK_STR,HEADER_TAB_LEN - (firstwhite - ptr));
      ret[HEADER_TAB_LEN] = NUL_TERM;
      strncat(ret,firstwhite,fieldlen - (firstwhite - ptr));
      ret[fieldlen + (HEADER_TAB_LEN - (firstwhite - ptr))] = NUL_TERM;
    }
    else {
      strncpy(ret,ptr,fieldlen);
      ret[fieldlen] = NUL_TERM;
    }
  }
  return(ret);
}


/* 
 * Assumes that str points to the beginning of a header field.
 * It finds the end of the field (allowing for continuation lines)
 * and returns the length. This is used by the functions to extract
 * particular header fields.
 */


#ifdef __STDC__
unsigned long header_field_len(char *str)
#else
unsigned long header_field_len(str)
  char *str;
#endif
{
  unsigned long n = 0;
  char *ptr;

  for(ptr = str; *ptr ; ptr ++, n ++) {
    if(*ptr == LFCHAR) {
      if((*(ptr + 1) == TABCHAR) || (*(ptr + 1) == SPACECHAR))
	continue;
      else
	break;
    }
  }
  return(n + 1);

}

#ifdef __STDC__
XmString make_view_line(Message *message)
#else
XmString make_view_line(message)
     Message *message;
#endif
{
  char mnum[32];
  char date[64];
  char from[40];
  char buffer[FILEBUFFLEN];
  char kwds[FILEBUFFLEN];
  unsigned long i;
  char flags[32];
  char *subject;
  MESSAGECACHE *elt;
  XmString ret;


  /* We have the info ?? */

  if(message && message->longcache && message->envelope) {

    elt = &(message->longcache->elt);

    sprintf(mnum,"%5lu",message->msgno);
    flags[0] = 
      (elt->recent) 
	? ((elt->seen) ? 'R' : 'N'):
	  ((elt->seen) ? ' ' : 'U');
    flags[1] = (elt->flagged)  ? 'F' : ' ';
    flags[2] = (elt->answered) ? 'A' : ' ';
    flags[3] = (elt->deleted)  ? 'D' : ' ';
    flags[4] = NUL_TERM;
    
    /*  only use "dd-mmm" from date */
    mail_date(date,elt);
    date[6] = NUL_TERM;

    mail_fetchfrom(from, message->mailstream, message->msgno, 20);

    kwds[0] = NUL_TERM;
    if(i = elt->user_flags) {
      kwds[0] = '[';
      kwds[1] = NUL_TERM;
      while(i) {
	strcat(kwds,message->mailstream->user_flags[find_rightmost_bit(&i)]);
	if(i)
	  strcat(kwds," ");
      }
      strcat(kwds,"] ");
    }

    subject = message->envelope->subject;
    if(subject == NULL)
      subject = EMPTYSTR;

    sprintf (buffer,"%s %s %s %s%-.80s (%lu chars)",flags, date, 
	     from, (*kwds != NUL_TERM) ? kwds : EMPTYSTR, subject,
	     elt->rfc822_size);
  }
  else     /* We don't have the info */
    sprintf(buffer,"<Message Not Yet Loaded>");

  ret = XmStringCreateSimple(buffer);
  return(ret);

}



void get_message_info(mailbox,lview)
     Mailbox *mailbox;
     Lview *lview;
{

  Arg args[ARGLISTSIZE];
  int n = 0;
  Widget shell, scroll;
  int x, y;
  Message *message;
  Message_List *message_list;
  Message_List *prev = NULL;
  double percentage = 0.0;
  int int_percent = 0;

  if(mailbox->nmsgs == 0 || lview->count == 0)
    return;

  shell = XtCreateWidget("Loading Mailbox",topLevelShellWidgetClass,	 
			 top, NULL, 0);

  XtSetArg(args[n], XmNwidth,            300);          n ++;
  XtSetArg(args[n], XmNorientation,      XmHORIZONTAL); n ++;
  XtSetArg(args[n], XmNshowArrows,       FALSE);        n ++;
  XtSetArg(args[n], XmNmaximum,          100);          n ++;
  XtSetArg(args[n], XmNminimum,          0);            n ++;
  XtSetArg(args[n], XmNvalue,            0);            n ++;
  XtSetArg(args[n], XmNsliderSize,       1);            n ++;
  scroll = XmCreateScrollBar(shell,"bargraph",args,n); n = 0;

  XtManageChild(scroll);
  XtManageChild(shell);

  

  for(x = 1; x <= lview->count; x ++ ) {
    
    percentage = ((double) x / (double) lview->count) * 100.0 ;
    int_percent = (int) percentage;
    if(int_percent > 0 && int_percent <= 100) {
      XtSetArg(args[n], XmNsliderSize, int_percent); n ++;
      XtSetValues(scroll,args,n); n = 0;
      XFlush(display);
    }

    message_list = new_message_list();

    if(prev) {
      prev->next = message_list;
      message_list->prev = prev;
    }
    else
      lview->message_list = message_list;

    message_list->number = x;
    message = new_message();
    message_list->message = message;
    message->msgno = x;
    message->mailbox = mailbox;
    message->mailstream = mailbox->mailstream;

    message->envelope = mail_fetchstructure(message->mailstream,
					    message->msgno,
					    &message->body);
    message->longcache = mail_lelt(message->mailstream, message->msgno);
    message->viewline = make_view_line(message);
    prev = message_list;
  }

  /* make sure we get that last little bit in the scrollbar */

  XtSetArg(args[n], XmNsliderSize, 100); n ++;
  XtSetValues(scroll,args,n); n = 0;
  XFlush(display);

  mailbox->fetched = TRUE;
  XtDestroyWidget(shell);

}

void get_new_message_info(mailbox)
     Mailbox *mailbox;
{

  int n = 0;
  Lview *lview;
  Message *message;
  Message_List *message_list;
  Message_List *prev = NULL;


  lview = get_all_lview(mailbox->mailstream);

  if(mailbox->nmsgs == 0 || lview == NULL)
    return;

  if(lview->message_list) {
    for(prev = lview->message_list; prev->next; prev = prev->next)
      ;
  }

  for(n = lview->count + 1; n <= mailbox->nmsgs; n ++) {

    message_list = new_message_list();

    if(prev == NULL)
      lview->message_list = message_list;
    else
      prev->next = message_list;

    message_list->prev = prev;
    message_list->number = n;

    message = new_message();
    message_list->message = message;

    message->msgno = n;
    message->mailbox = mailbox;
    message->mailstream = mailbox->mailstream;

    message->envelope = mail_fetchstructure(message->mailstream,
					    message->msgno,
					    &message->body);
    message->longcache = mail_lelt(message->mailstream, message->msgno);
    message->viewline = make_view_line(message);
    if(session.view->current == lview)
      XmListAddItemUnselected(session.view->list,message->viewline,0);

    prev = message_list;
  }
  lview->count = mailbox->nmsgs;

  if((has_new_messages(lview,FALSE)) == TRUE)
    lview->has_new = TRUE;
  else
    lview->has_new = FALSE;

  update_logical_views(mailbox,FALSE);

}


/* Called from the mm_flags() callback */

void update_view_line_stream_msgno(mailstream,msgno)
     MAILSTREAM *mailstream;
     unsigned long msgno;
{
  Message *message;
  Mailbox *mailbox;

  mailbox = find_mailbox_from_mailstream(mailstream);
  
  if(mailbox && mailbox->fetched) {
    message = get_message_from_mailbox(mailbox,msgno);
    mailbox->flags_changed ++;
    if(message)
      update_view_line(mailbox,message);
  }

}	

void update_view_line(mailbox,message)
     Mailbox *mailbox;
     Message *message;
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Message_List *message_list;
  XmString xstr = NULL;
  Boolean different = FALSE;


  xstr = make_view_line(message);

  if(xstr == NULL)
    return;
  
  /* we only need to update things if the string has actually changed. */
  if((message->viewline != NULL)
     && ((XmStringCompare(xstr,message->viewline)) == FALSE))
    different = TRUE;
  
  if(message->viewline == NULL) /* just in case wasn't set already */
    different = TRUE;
  
  if(different == TRUE) {       /* Set the new structure value */
    if(message->viewline)
      XmStringFree(message->viewline);
    message->viewline = xstr;
  }
  else {                       /* No changes needed. Go home. */
    XmStringFree(xstr);
    return;
  }


  
  /* 
   * If the current view contains the message, update it.
   * Preserve highlight. If not in the view, we've already changed
   * the structure value, and it will take effect on the next
   * context switch. One of the reasons for checking "sameness" above
   * is that highlighting gets messed up if we re-highlight the same
   * string value, which toggles it off and doesn't tell us.
   */

  if((session.view != NULL) 
     && (session.view->current != NULL) && session.view->is_realized 
     && (session.view->current->mailstream == message->mailstream)) {
    for(message_list = session.view->current->message_list; 
	message_list; message_list = message_list->next) {
      if(message_list->message == message) {
	mailbox->refresh_needed ++;
	/*	
	   XmListReplaceItemsPos(session.view->list,&message->viewline,
	   1,message_list->number);
	   if(message_list->selected == TRUE) {
	   XtSetArg(args[n], XmNselectionPolicy, XmMULTIPLE_SELECT);   n ++;
	   XtSetValues(session.view->list, args, n);                   n = 0;
	   XmListSelectPos(session.view->list, message_list->number, FALSE);
	   XtSetArg(args[n], XmNselectionPolicy, XmEXTENDED_SELECT);   n ++;
	   XtSetValues(session.view->list, args, n);                   n = 0;
	   }
	   */
	break;
      }
    }
  }

  mailbox->update_needed ++;
  session.update_needed ++;

  return;
}


Message *get_message_from_mailbox(mailbox,msgno)
     Mailbox *mailbox;
     unsigned long msgno;
{

  Lview *lview;
  Message_List *message_list;

  lview = get_all_lview(mailbox->mailstream);

  if(lview) {
    for(message_list = lview->message_list;
	message_list; message_list = message_list->next) 
      if((message_list->message) && (message_list->message->msgno == msgno))
	return(message_list->message);
  }

  return(NULL);
}



/* 
 * Looks at the current highlighted messages and generates an IMAP
 * sequence string. This can be individual messages "n,nn,nnn,[etc.]" 
 * and/or message ranges denoted by "start_n:end_n". The range notation
 * cuts down considerable network bandwidth, but won't work well, (if at all)
 * if we have an alternate representation, like reverse order or sorted views.
 * We don't even try to optimize those. Hopefully the c-client won't crash
 * if we feed it an entire selected mailbox in reverse order.
 */

char *generate_sequence(lview)
     Lview *lview;
{
    Message_List *message_list;
    char *buffer;
    char *sequence;
    unsigned long found = 0L;
    unsigned long range_begin = 0L;
    unsigned long last_range = 0L;
    Boolean range = FALSE;
    

    if(lview == NULL || lview->num_selected == 0L)
      return(NULL);

    /*
     * Overkill (worst case) allocation. 9 million messages out of
     * sequence starting at 1000000 (plus a separator char) is unlikely.
     * The program will have crashed elsewhere or would have hit system
     * limits if somebody ever overflows this buffer.
     */

    buffer = (char *) fs_get(8 * lview->num_selected);
    sequence = buffer;
    *sequence = NUL_TERM;		

    for(message_list = lview->message_list; message_list; 
	message_list = message_list->next) {
      if(message_list->selected == FALSE)
	continue;

      /* 
       * if we're in a range sequence and haven't reached the end of it,
       * update the last_range variable and keep going
       */

      if((found != 0) 
	 && (range == TRUE) 
	 && (message_list->next != NULL) 
	 && (message_list->next->selected == TRUE)
	 && (message_list->message)
	 && (message_list->next->message)
	 && (message_list->next->message->msgno == (last_range +1))) {
	last_range = message_list->next->message->msgno;
	found ++;
	continue;
      }

      /* 
       * See if this qualifies as a range sequence.
       */

      if((range == FALSE)
	 && (message_list->next != NULL)
	 && (message_list->next->selected == TRUE)
	 && (message_list->message)
	 && (message_list->next->message)
	 && (message_list->next->message->msgno 
	     == (message_list->message->msgno + 1))) {
	range = TRUE;
	range_begin = message_list->message->msgno;
	last_range = message_list->next->message->msgno;
	if(found != 0) {
	  strcat(sequence,",");
	  sequence += 1;
	}
	sprintf(sequence,"%d", message_list->message->msgno);
	sequence += strlen(sequence);
	found ++;
	continue;
      }

      /* We've found the last in a range. Flush it. */

      if((range == TRUE) 
	 && (range_begin != 0)) {
	sprintf(sequence,":%d",last_range);
	range = FALSE;
	range_begin = 0L;
	last_range = 0L;
	found ++;
	sequence += strlen(sequence);
	continue;
      }

      /* 
       * Not a range. If the first message, start the string.
       * Otherwise, add a comma, then the current number.
       */
	
      if(found == 0)  
	sprintf(sequence,"%d", message_list->message->msgno);
      else
	sprintf(sequence,",%d", message_list->message->msgno);
      found ++;
      sequence += strlen(sequence);
    }

    return(buffer);			
}



/* 
 * This is the same as the above function except that it builds the list
 * from the current lview, ignoring highlight state. Every message in the 
 * lview becomes part of the sequence.
 */

char *generate_sequence_unselected(lview)
     Lview *lview;
{
    Message_List *message_list;
    char *buffer;
    char *sequence;
    unsigned long found = 0L;
    unsigned long range_begin = 0L;
    unsigned long last_range = 0L;
    Boolean range = FALSE;
    

    if(lview == NULL || lview->count == 0L)
      return(NULL);

    /*
     * Overkill (worst case) allocation. 9 million messages out of
     * sequence starting at 1000000 (plus a separator char) is unlikely.
     * The program will have crashed elsewhere or would have hit system
     * limits if somebody ever overflows this buffer.
     */

    buffer = (char *) fs_get(8 * lview->count);
    sequence = buffer;
    *sequence = NUL_TERM;		

    for(message_list = lview->message_list; message_list; 
	message_list = message_list->next) {

      /* 
       * if we're in a range sequence and haven't reached the end of it,
       * update the last_range variable and keep going
       */

      if((found != 0) 
	 && (range == TRUE) 
	 && (message_list->next != NULL) 
	 && (message_list->message)
	 && (message_list->next->message)
	 && (message_list->next->message->msgno == (last_range +1))) {
	last_range = message_list->next->message->msgno;
	found ++;
	continue;
      }

      /* 
       * See if this qualifies as a range sequence.
       */

      if((range == FALSE)
	 && (message_list->next != NULL)
	 && (message_list->message)
	 && (message_list->next->message)
	 && (message_list->next->message->msgno 
	     == (message_list->message->msgno + 1))) {
	range = TRUE;
	range_begin = message_list->message->msgno;
	last_range = message_list->next->message->msgno;
	if(found != 0) {
	  strcat(sequence,",");
	  sequence += 1;
	}
	sprintf(sequence,"%d", message_list->message->msgno);
	sequence += strlen(sequence);
	found ++;
	continue;
      }

      /* We've found the last in a range. Flush it. */

      if((range == TRUE) 
	 && (range_begin != 0)) {
	sprintf(sequence,":%d",last_range);
	range = FALSE;
	range_begin = 0L;
	last_range = 0L;
	found ++;
	sequence += strlen(sequence);
	continue;
      }

      /* 
       * Not a range. If the first message, start the string.
       * Otherwise, add a comma, then the current number.
       */
	
      if(found == 0)  
	sprintf(sequence,"%d", message_list->message->msgno);
      else
	sprintf(sequence,",%d", message_list->message->msgno);
      found ++;
      sequence += strlen(sequence);
    }

    return(buffer);			
}



/* 
 * A partial misnomer. The "asking" is done during the file select.
 * The routine just saves a binary buffer or returns.
 */


void ask_save(w,binary_buffer)
     Widget w;
     Binary_Buffer *binary_buffer;
{

  FILE *fp;
  Boolean append_it;
  char *filename;
  int errors = 0;

  filename = file_select(w, NULL, NULL, NULL, TRUE, &append_it);
  if(filename == NULL)
    return;

  set_watch_cursors();
  if((fp = fopen(filename,(append_it == TRUE) ? "a" : "w")) == NULL)
    errors ++;

  if(! errors) { 
    if(fwrite(binary_buffer->data,binary_buffer->length,1,fp) != 1)
      errors ++;
    if(fclose(fp))
      errors ++;
  }

  if(errors)
    mm_log("Save failed.", WARN);
  else
    mm_log("Save successful.", NIL);

  fs_give((void **) &filename);

  reset_cursors();

  return;
}



