/* ml.c 
 *
 *
 * $Author: mtm $
 * $Date: 1994/11/07 22:06:46 $
 * $Id: ml.c,v 1.1 1994/11/07 22:06:46 mtm Exp mtm $
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1994 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log: ml.c,v $
 * Revision 1.1  1994/11/07  22:06:46  mtm
 * Initial revision
 * 
 */

#include "ml.h"



/* global structures and variables */

Widget top;
Widget status;
XtAppContext context;
Preferences preferences;
Local_Auth local_auth;
Session session;
Display *display;

/* used in compose address text field widgets */

XtActionsRec global_actions[] = {
  { "complete_address", (XtActionProc) complete_address }
};

/* IMAP registered drivers */

extern DRIVER 
imapdriver,   bezerkdriver,
tenexdriver,    nntpdriver, 
newsdriver,    dummydriver,
pop3driver,    mtxdriver,
mhdriver,      mmdfdriver,
philedriver;


/* 
 *     Local function definitions 
 */

void start_up();
void create_main_window();
void main_loop();
void no_op();
void bye();
void main_open();
void main_close();
void main_compose();
void main_address_book();
void make_prefs_window();
void make_prefs2_window();
void main_help();
void timed_check();
void auto_open();
void open_mailbox();
void main_raise_view();
void main_raise_read();
void main_raise_compose();
void trap_stderr();



static String fallback_resources[] = { 
"*textFontList:                 -*-courier-medium-r-normal--12-*",
"*buttonFontList:	        -*-courier-bold-r-normal--12-*",
"*labelFontList:                -*-courier-bold-r-normal--12-*",
"*XmCascadeButton.marginHeight: 1",
"*XmCascadeButton.marginWidth:  8",
"*marginHeight:                 2",
"*marginWidth:                  3",
"*XmList*highlightThickness:    0",
"*background:                   #d3d3d3",    /* light grey */
"*foreground:                   #000000",    /* black      */
"*XmCascadeButton*foreground:   #0000ff",    /* blue       */
"*XmLabel*foreground:           #000000",    /* black      */
"*doubleClickInterval:          500",
NULL,
};


Menu main_file_menu[] = {

  { "Open Mailbox", "main_open", 'O', 
      main_open, NULL, 0, NULL, NULL, BTN_ON },
  { "Address_Book", "main_address_book", 'A',
      main_address_book, NULL, 0, NULL, NULL, BTN_ON },
  { "Compose New Message", "main_compose", 'N',
      main_compose, NULL, 0, NULL, NULL, BTN_ON },
  { "Close All Mailboxes", "main_close", 'C', 
      main_close, NULL, 0, NULL, NULL, BTN_NOMAILBOXESOPEN },
  { "Exit Program", "main_exit", 'x', 
      bye,   NULL, 0, NULL, NULL, BTN_ON },

};


Menu main_window_menu[] = {
  { "View Window", "main_raise_view", 'V', 
      main_raise_view, NULL, 0, NULL, NULL, BTN_NOMAILBOXESOPEN },
  { "Read Window", "main_raise_read", 'R', 
      main_raise_read, NULL, 0, NULL, NULL, BTN_NOMAILBOXESOPEN },
  { "Compose Window", "main_raise_compose", 'C', 
      main_raise_compose, NULL, 0, NULL, NULL, BTN_ON },

};

Menu main_option_menu[] = {
  { "Preferences", "make_prefs_window", 'P',
      make_prefs_window, NULL, 0, NULL, NULL, BTN_ON },
  { "More Preferences", "make_prefs2_window", 'M',
      make_prefs2_window, NULL, 0, NULL, NULL, BTN_ON },

};



Menu main_menu[] = {
  { "File", "main_file_menu", 'F', 
      NULL, main_file_menu, XtNumber(main_file_menu), 
      NULL, NULL, BTN_ON },
  { "Option", "main_option_menu", 'O',
      NULL, main_option_menu, XtNumber(main_option_menu),
      NULL, NULL, BTN_ON },

  { "Window", "main_window_menu", 'W',
      NULL, main_window_menu, XtNumber(main_window_menu), 
      NULL, NULL, BTN_ON },

  { "Help", "main_HELP!", 'H', 
      main_help, NULL, 0,
      NULL, NULL, BTN_ON },
};



/* Begin Program */

#ifdef __STDC__
void main(int argc,char **argv, char **envp)
#else
void main(argc,argv,envp)
     int argc;
     char ** argv;
     char ** envp;
#endif
{
  Arg args[ARGLISTSIZE];
  int n = 0;
  Session *debug;
  FILE *fp;
  char filename[MAXPATHLEN];
  char buffer[FILEBUFFLEN];

  /***********************************************************************
   *              Create top level application shell - start X           *
   ***********************************************************************/

  XtSetArg(args[n], XtNtitle, PROGRAM);                    n ++;
  top = XtAppInitialize((XtAppContext*) &context,PROGRAM,
			NULL, 0, &argc,argv, 
			fallback_resources, args, n);      n = 0;

  (void) start_up();

  (void) create_main_window();

  XtRealizeWidget(top);
  display = XtDisplay(top);


  XtAppAddActions(context, global_actions, XtNumber(global_actions));

  (void) create_cursors(display);

  sprintf(buffer,"Welcome to %s, Version %s\n",PROGRAM,VERSION);
  AppendText(status,buffer);
  

  if(preferences.auto_open == TRUE)
    auto_open();


  trap_stderr();

  XFlush(display);


  /* This is only for debugging convenience, it has no real function. */
  debug = &session;

  /*************************************************************************
   *           Main application loop - get and process input forever       *
   *************************************************************************/

  main_loop();

}

#ifdef __STDC__
void start_up(void)
#else
void start_up()
#endif
{

  /* find out who is running the program. */

  if((get_user()) == SYSCALL_FAILURE) 
    exit(0);

  (void) chdir(local_auth.homedir);

  mail_link(&imapdriver);
  mail_link(&bezerkdriver);
  mail_link(&tenexdriver);
  mail_link(&nntpdriver);
  mail_link(&newsdriver);
  mail_link(&pop3driver);
  mail_link(&mtxdriver);
  mail_link(&mhdriver);
  mail_link(&mmdfdriver);
  mail_link(&philedriver);
  mail_link(&dummydriver);

  session.address_book = NULL;
  session.filters = NULL;
  session.mime_handlers = NULL;
  session.busy = FALSE;
  session.button_state = (unsigned long) 0;
  session.authorizations = NULL;
  session.compose = NULL;
  session.view = NULL;
  session.read = NULL;
  session.prefst = NULL;
  session.pref2st = NULL;
  session.server_list = NULL;
  session.update_needed = 0;
  session.filter_map = NULL;
  (void) get_resources(top);
  (void) load_defaults();
  (void) sane_defaults();

  /* Load these after the preferences, so we know where to look */

  (void) load_mime_types();
  (void) load_mime_handlers();

  /* Initialize the c-client's sense of identity. Ignore results. */

  (void) myusername();
  


}


#ifdef __STDC__
void create_main_window(void)
#else
void create_main_window()
#endif
{
  Arg args[ARGLISTSIZE];
  int n = 0;

  Widget form, menubar;
  XtTranslations translations;


  /**********************************************************************
   *                     Create status/telemetry widget                 *
   **********************************************************************/


  form = XmCreateForm(top, "form", args, n); n = 0;
  
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form,"menubar", args, n); n = 0;


  create_menu_buttons(NULL, menubar, main_menu, XtNumber(main_menu), 
		      BTN_NOMAILBOXESOPEN, NULL);

  XtSetArg(args[n], XmNeditable,         TRUE);                  n ++;
  XtSetArg(args[n], XmNeditMode,         XmMULTI_LINE_EDIT);     n ++;
  XtSetArg(args[n], XmNwordWrap,         TRUE);                  n ++;
  XtSetArg(args[n], XmNrows,             STATUSWIN_HEIGHT);      n ++;
  XtSetArg(args[n], XmNcolumns,          STATUSWIN_WIDTH);       n ++;
  XtSetArg(args[n], XmNtopAttachment,    XmATTACH_WIDGET);       n ++;
  XtSetArg(args[n], XmNtopWidget,        menubar);               n ++;
  XtSetArg(args[n], XmNleftAttachment,   XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNrightAttachment,  XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNscrollVertical,   TRUE);                  n ++;
  XtSetArg(args[n], XmNscrollHorizontal, FALSE );                n ++;
  status = XmCreateScrolledText(form, "status", args, n);        n = 0;

  translations = XtParseTranslationTable(GLOBAL_text_translations);
  XtOverrideTranslations(status, translations);
  
  XtManageChild(form);
  XtManageChild(menubar);
  XtManageChild(status);
  return;
}



Boolean add_server_to_session(str,type)
     char *str;
     int type;
{
  Server_List *server_list;

  if((str == NULL) || (*str == NUL_TERM))
    return(FALSE);

  for(server_list = session.server_list; 
      server_list; server_list = server_list->prev) {
    if((strcasecmp(server_list->name,str) == STRMATCH) 
       && (server_list->type == type))
      return(FALSE);
  }

  server_list = new_server_list();
  server_list->name = cpystr(str);
  server_list->type = type;
  if(session.server_list)
    session.server_list->next = server_list;
  server_list->prev = session.server_list;
  session.server_list = server_list;
  return(TRUE);

}


/* 
 * If an auto-open operation failed or was cancelled, we haven't obtained 
 * any mailbox list info. Here we check for that condition, and if that's 
 * the case, remove the server from session.server_list, so that the next 
 * open will load it correctly. 
 */

void invalidate_server_list(server_list)
     Server_List *server_list;
{

  if(server_list->mailbox_list)
    return;

  if(server_list == session.server_list)
    session.server_list = server_list->prev;
  if(server_list->prev)
    server_list->prev->next = server_list->next;
  if(server_list->next)
    server_list->next->prev = server_list->prev;

  if(server_list->name)
    fs_give((void **) &server_list->name);
  fs_give((void **) &server_list);
}




void main_open(w,cb,xp)
     Widget w;
     XtPointer cb;
     XtPointer xp;
{

  char buff[FILEBUFFLEN];
  Mailbox *mailbox = new_mailbox();

  mailbox->has_new_mail = FALSE;
  mailbox->imapname = get_mailbox_name(top, mailbox, FALSE,
				       preferences.imap_server,
				       preferences.default_mailbox);

  if((mailbox->imapname != NULL) && (*mailbox->imapname != NUL_TERM))
    open_mailbox(mailbox, FALSE);
  else
    free_mailbox(mailbox);

}


/* 
 * auto_open has to initialize a few things in the mailbox structure
 * which would have otherwise been initialized by the mailbox list widget.
 */


void auto_open()
{
  char buff[FILEBUFFLEN];
  Mailbox *mailbox = new_mailbox();
  Boolean new_server = FALSE;

  mailbox->has_new_mail = FALSE;
  mailbox->host = get_qualified_host_name(preferences.imap_server);
  mailbox->mailboxname = fix_mailboxpath(preferences.default_mailbox);
  mailbox->imapname = 
    format_mailbox_name(mailbox->host,mailbox->mailboxname,MAILBOX_TYPE_MAIL);
  mailbox->type = MAILBOX_TYPE_MAIL;

  if((mailbox->imapname == NULL) || (*mailbox->imapname == NUL_TERM)) {
    free_mailbox(mailbox);
    return;
  }

  if(add_server_to_session(mailbox->host, MAILBOX_TYPE_MAIL))
    new_server = TRUE;

  open_mailbox(mailbox, new_server);
}


void open_mailbox(mailbox, new_server)
     Mailbox *mailbox;
     Boolean new_server;
{

  char log[FILEBUFFLEN];
  Server_List *server_list;
  int openflags = 0;

  set_watch_cursors();

  if((mailbox->imapname == NULL) || (*mailbox->imapname == NUL_TERM)) {
    free_mailbox(mailbox);
    return;
  }

  sprintf(log,"Opening mailbox '%s'.",mailbox->imapname);
  mm_log(log,NIL);

  if(((mailbox->mailstream = 
      mail_open(mailbox->mailstream,mailbox->imapname,openflags)) == NIL) 
     || (mailbox->mailstream->halfopen)) {
    server_list = find_server_list(mailbox->host,mailbox->type);
    invalidate_server_list(server_list);
    if(mailbox->mailstream)
      mail_close(mailbox->mailstream);
    free_mailbox(mailbox);
    mm_log("Failed to open mailbox.",ERROR);
    reset_cursors();
    return;
  }
  else {
    session.button_state &= (~BTN_NOMAILBOXESOPEN);

  }

  if(new_server) {
    server_list = find_server_list(mailbox->host,mailbox->type);
    server_list->active = TRUE;
    mail_find(mailbox->mailstream,"*");
    server_list->active = FALSE;
  }

  mailbox->nmsgs = mailbox->mailstream->nmsgs;
  if(session.mailboxes)
    session.mailboxes->prev = mailbox;
  mailbox->next = session.mailboxes;
  session.mailboxes = mailbox;

  mailbox->timer = 
    XtAppAddTimeOut(context,
		    ((unsigned long) preferences.check_interval * 1000L),
		    (XtTimerCallbackProc) timed_check, mailbox);

  create_all_lview(mailbox);


  check_buttons(main_menu, XtNumber(main_menu), session.button_state);
  reset_cursors();

}


void check_mailbox(mailbox)
     Mailbox *mailbox;
{
  Message_List *message_list = NULL;
  char msg[512];
  char sequence[32];

  set_watch_cursors();

  if(mail_ping(mailbox->mailstream)) {

    if(mailbox->has_new_mail == TRUE) {
      get_new_message_info(mailbox);
      mailbox->has_new_mail = FALSE;
      XBell(display,1000);
    }
    if(mailbox->flags_changed != 0L) {
      mail_check(mailbox->mailstream);
      mailbox->flags_changed = 0L;
    }

  }
  else {
    sprintf(msg,"Connection to server '%s' broken.",mailbox->host); 
    mm_log(msg,ERROR);
  }
  reset_cursors();
}

void timed_check(mailbox,id)
     Mailbox *mailbox;
     XtIntervalId *id;
{

  unsigned long next_timer;
 
  /* 
   * We've been called by a timer (or user) to check the mailbox.
   * Unfortunately, we might be doing something critical.
   * If we are, set the timer interval to a short period (15 seconds)
   * so we'll keep coming back until things are free again.
   * Otherwise, check the mailbox, and reset to the normal timer interval.
   */

  if(session.busy == FALSE) { 
    check_mailbox(mailbox);
    next_timer = (unsigned long) preferences.check_interval * 1000L;
  }
  else
    next_timer = (unsigned long) 15000L;

  mailbox->timer = XtAppAddTimeOut(context,next_timer,
				   (XtTimerCallbackProc) timed_check,
				   mailbox);
  return;
}


void main_close(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{

  Mailbox *mailbox;
  Lview *lview;
  Read_Info *read_info;
  Boolean found = FALSE;

  mm_log("Closing all mailboxes.", NIL);

  for(mailbox = session.mailboxes; mailbox ; mailbox = mailbox->next) {
    XtRemoveTimeOut(mailbox->timer);
    if(mail_ping(mailbox->mailstream)) {
      mail_check(mailbox->mailstream);
      mail_close(mailbox->mailstream);
    }

    if(session.view) {
      do {
	for(lview = session.view->lview; lview ; lview = lview->prev)
	  if(lview->mailstream == mailbox->mailstream) {
	    found = TRUE;
	    view_unlink_destroy_lview(session.view,lview,TRUE);
	  }
	  else
	    found = FALSE;
      } while((lview) && (found == TRUE));
    }
 
    if(session.read)
      for(read_info = session.read->read_info; read_info; 
	  read_info = read_info->prev)
	if(read_info->mailstream == mailbox->mailstream) {
	  if((session.read->is_realized) 
	     && (read_info == session.read->current))
	    read_close_kill_current(session.read->shell,session.read,NULL);
	  else
	    kill_read_info(read_info);
	}
    compose_mailbox_close(mailbox->mailstream);
    free_mailbox(mailbox);
  }

  session.mailboxes = NULL;
  session.button_state |= (BTN_NOMAILBOXESOPEN);

  check_buttons(main_menu, XtNumber(main_menu), session.button_state);

}

void main_compose(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{
  (void) compose(NEW_MESSAGE_STR, MAIL_BEHAVIOUR, COMPOSE_NEW, 
		 NULL, NULL, NULL);
}


void main_address_book(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{
  (void) create_address_book_window(top);
}

void main_raise_view(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{
  if(session.view && session.view->lview) {
    create_view_window();
    view_context_switch(session.view,NULL,session.view->lview);
  }
  else
    mm_log("Window open failed. Please open a mailbox.", WARN);
}

void main_raise_read(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{
  if(session.read && session.read->read_info) {
    create_read_window();
    read_context_switch(session.read,NULL,session.read->read_info);
  }
  else
    mm_log("Window open failed. No messages in read queue.", WARN);

}

void main_raise_compose(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{
  if(session.compose && session.compose->compose_info) {
    create_compose_window(session.compose,MAIL_BEHAVIOUR);
    XRaiseWindow(display,XtWindow(session.compose->shell));
    compose_context_switch(session.compose,NULL,session.compose->compose_info);
  }
  else
    mm_log("Window open failed. No current compositions.", WARN);

}


void no_op(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{

  mm_log("Function not implemented.", WARN);

  /* globally accessible absolutely useless function */
}

void main_help(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{

  help(top,"ML Help","mainhelp.help");

}


void bye(w,data,xp)
     Widget w;
     XtPointer data;
     XtPointer xp;
{
  if(session.mailboxes)
    main_close(w,NULL,NULL);
  exit(0);
}

void update_views()
{
  Mailbox *mailbox;
  for(mailbox = session.mailboxes; mailbox; mailbox = mailbox->next) {
    if(mailbox->refresh_needed) {
      update_current_view(mailbox);
      mailbox->refresh_needed = 0;
    }
    if(mailbox->update_needed) {
      update_logical_views(mailbox,FALSE);
      mailbox->update_needed = 0;
    }
  }
}

/* 
 * This loop is called from our modal pop-ups -- those that wait
 * for you to do something before continuing. The loop is exited
 * when the condition (usually an int) becomes non-zero. This usually
 * happens when a modal widget gets what it needs and relinquishes
 * exclusive control of the keyboard and mouse by killing itself.
 * In general we set "pirate" cursors for other windows to indicate
 * that they are non-functional until the active window is dismissed.
 * Yes, it's awful design strategy, but necessary in this program,
 * since something you are doing elsewhere (like expunging messages 
 * or closing mailboxes) could invalidate the entire state of the 
 * current data structures.
 */

void modal_main_loop (exit_condition)
     int *exit_condition;
{
  XEvent event;

  while(!(*exit_condition)) {
    XtAppNextEvent(context, &event);
    XtDispatchEvent(&event);
  }
}

void main_loop ()
{
  XEvent event;

  while(1) {
    XtAppNextEvent(context, &event);
    XtDispatchEvent(&event);
    if(session.update_needed) {
      update_views();
      session.update_needed = 0;
    }
  }
}




/* 
 * A function to make our main menu buttons tweakable from other files,
 * since they don't know the size of our menu. 
 */

#ifdef __STDC__
void update_main_buttons(void)
#else
void update_main_buttons()
#endif
{
  check_buttons(main_menu, XtNumber(main_menu), session.button_state);
}








void redirect_stderr(client_data, fd, id)
     XtPointer client_data;
     int *fd;
     XtInputId *id;
{

#define ERRBUFLEN 512
  char buf[ERRBUFLEN + 1];

  int nbytes;

  if((nbytes = read(*fd,buf,ERRBUFLEN)) == (EOF))
    return;
  buf[nbytes] = NUL_TERM;
  if(nbytes) {
    XRaiseWindow(display,XtWindow(top));
    AppendText(status,buf);
  }
}


int errfd[2];

void trap_stderr()
{

  (void) signal(SIGPIPE,SIG_IGN);

  if(pipe(errfd))
    return;

  dup2(errfd[1], fileno(stderr));
  close(errfd[1]);
    

  XtAppAddInput(context, errfd[0], (caddr_t) XtInputReadMask, 
		(XtInputCallbackProc) redirect_stderr, NULL);


}


