/* lvfncs.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1995 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

void remote_search();


/*
 * Macro definitions. These are filter criteria which are expanded
 * at search time and contain more than one filter rule. for instance,
 * "conversant john" is the same as "from john or to john or cc john". 
 */


static LEAF recipient_macro[]= {
  "CC", NO_OP, simple, no_binding, no_binding, pred_field, local, NULL,
  "TO", OR_OP, simple, no_binding, no_binding, pred_field, local, NULL,
  NULL, NO_OP, simple, no_binding, no_binding, pred_simple, undefined, NULL
};  
  
static LEAF correspondent_macro[]= {
  "FROM", NO_OP, simple, no_binding, no_binding, pred_field, local, NULL,
  "TO",   OR_OP, simple, no_binding, no_binding, pred_field, local, NULL,
  NULL,   NO_OP, simple, no_binding, no_binding, pred_simple, undefined, NULL
};  

static LEAF conversant_macro[]= {
  "FROM", NO_OP, simple, no_binding, no_binding, pred_field, local, NULL,
  "TO",   OR_OP, simple, no_binding, no_binding, pred_field, local, NULL,
  "CC",   OR_OP, simple, no_binding, no_binding, pred_field, local, NULL,
  NULL,   NO_OP, simple, no_binding, no_binding, pred_simple, undefined, NULL
};  



static LEAFMACRO macro_petals[]= {

  RECIPIENT_MACRO,			/* name of macro */
  recipient_macro,			/* macro table */
  2,					/* n table entries */

  CORRESPONDENT_MACRO,			/* name of macro */
  correspondent_macro,			/* macro table */
  2,					/* n table entries */

  CONVERSANT_MACRO,
  conversant_macro,
  3,

  NULL, NULL, 0,			/* end of table */
};


Message_List *apply_op_to_operands(mailbox,op,list1,list2)
     Mailbox *mailbox;
     BOOL op;
     Message_List **list1;
     Message_List **list2;
{
  Message_List *result = NULL;

  switch(op) {
  case AND_OP:
    conjunct_and(mailbox,*list1,*list2);
    result = mailbox->found;
    mailbox->found = NULL;
    break;

    /* 
     * A no_op is treated as an or_op because we don't know which of the
     * two lists to preserve -- one of them will be empty. The other needs
     * to be returned. Or'ing the two gives us the right result, regardless.
     */

  case OR_OP:
  case NO_OP:
    conjunct_or(mailbox,*list1,*list2);
    result = mailbox->found;
    mailbox->found = NULL;
    break;

  default:
    /* this shouldn't ever happen. */
    break;
  }

  free_message_list(*list1,TRUE,FALSE);
  *list1 = NULL;
  free_message_list(*list2,TRUE,FALSE);
  *list2 = NULL;

  return(result);
}


LEAFMACRO *macro_expansion_required(leaf)
     LEAF *leaf;
{
  int i;
  LEAFMACRO *lm = macro_petals;
  char tmp[FILEBUFFLEN];

  strcpy(tmp, leaf->predicate);

  /*
   * We have either 
   *  predicate "FIELD"
   *      or
   *  predicate
   * In the former case we NULL terminate the predicate
   * for our comparsion to our macro predicates 
   */

  if (leaf->predicate_type == pred_field ||
      leaf->predicate_type == pred_dynamic_field) {
    /* select the predicate only - it is terminated by a space */
    strtok(tmp, " ");			/* NULL out the SPACE */
  }
  for (i = 0; lm->macro_name; ++i, ++lm)
    if (strcasecmp(lm->macro_name, tmp) == 0)
      return lm;
  return NIL;
}


void do_imap_macro_mail_search(mailbox, parent, lm, petal)
     Mailbox *mailbox;
     Lview *parent;
     LEAFMACRO *lm;			/* The expansion rules */
     LEAF *petal;			/* The macro which we expand */
     
{
  LEAF *macro= lm->leaves;
  int n_leaves= lm->n_leaves;
  int i;
  char field[FILEBUFFLEN], *fptr= field;
  Boolean local_ok= TRUE;
  Message_List *result = NULL;
  Message_List *current = NULL;

  /* see if our MACRO predicate requires a field */

  if (petal->predicate_type == pred_field ||
      petal->predicate_type == pred_dynamic_field) {
    /* extract the field */
    strcpy(field, petal->predicate);	/* "pred field" */
    fptr = strchr(field,(int) ' ') + 1;	/* skip the space */
  } else
    fptr = NULL;


  /* EXPAND our macro here one leaf at a time ... */

  for (i = 0; i < n_leaves; ++i, ++macro) {
    char pred[FILEBUFFLEN];		/* for expanded predicate */

    /* 
     * Contruct a search predicate with a field if necessary.
     * Our macros NEVER include them since they are dynamic 
     */

    strcpy(pred, macro->predicate);	/* get the MACRO predicate */
    if ((petal->predicate_type == pred_field ||
	 petal->predicate_type == pred_dynamic_field) &&
	macro->predicate_type == pred_field) { /* double check OK */
      strcat(pred, " ");		/* adjunct the FIELD */
      strcat(pred, fptr);
    }

    /*
     * Dispatch the search
     */

    free_message_list(mailbox->found,TRUE,FALSE);
    mailbox->found = NULL;

    if (petal->search_type == local)
      local_ok = local_search(mailbox, parent, pred);
    if (petal->search_type == server || !local_ok)
      remote_search(mailbox, parent, pred);

    /*
     * Save the current results.
     */

    current = mailbox->found;
    mailbox->found = NULL;

    /*
     * Now accumulate the results with any operands
     */


    result = apply_op_to_operands(mailbox, petal->operator,
				    &result, &current);

  }    

  /* 
   * Our accumulated results are in "result".  We'll make it appear
   * as though this was a simple mailbox search, (results accumulated
   * into "mailbox->found") since that's what the dispatcher is expecting. 
   * We're done with the intermediate list. Free its memory.
   */

  free_message_list(current, TRUE, FALSE);
  mailbox->found = result;

}


void remote_search(mailbox, parent, predicate)
     Mailbox *mailbox;
     Lview *parent;
     char *predicate;
{

  Message_List *result;

  free_message_list(mailbox->found,TRUE,FALSE);
  mailbox->found = NULL;

  mail_search(mailbox->mailstream,predicate);

  result = mailbox->found;
  mailbox->found = NULL;

  conjunct_and(mailbox,result,parent->message_list);
  free_message_list(result,TRUE,FALSE);

}





/*
 * Do the search on the given mail box 
 */


void do_imap_mail_search(mailbox, parent, leaf)
     Mailbox *mailbox;
     Lview *parent;
     LEAF *leaf;
{
  Boolean local_ok = TRUE;

  /*
   * Some searches are dispatched to the server,
   * and others we do locally from our cache */

  free_message_list(mailbox->found,TRUE,FALSE);
  mailbox->found = NULL;

  if (leaf->search_type == local)
    local_ok = local_search(mailbox, parent, leaf->predicate);

  /* If a server search we do it. Also if for some reason
   * the local search failed, we abdicate to the server */

  if (leaf->search_type == server || !local_ok)
    remote_search(mailbox, parent, leaf->predicate);

}



/*
 * Does a search on all search petals, and accumulates the results
 * into result 
 */

Message_List *do_imap_search(mailbox, parent, petal)
     Mailbox *mailbox;
     Lview *parent;
     LEAF *petal;
{
  LEAF *leaf= petal;
  Boolean not_at_end= FALSE;
  BOOL exp_op;
  char *old_pred;
  char *new_pred;
  Message_List *current = NULL;
  Message_List *result = NULL;

  /* run down the petals accumulating the results of
   * each search */

  while (leaf) {
    LEAFMACRO *lm;

    /* Search and set the search mask.
     *   1. See if the predicate's field is dynamic,
     *   2. check for macro expansion. 
     */

    new_pred = update_dynamic_predicates(leaf);
    if(new_pred) {
      /* Temporarily switch the predicates for the search */
      old_pred = leaf->predicate;
      leaf->predicate = new_pred;
    }
    else
      old_pred = NULL;

      
    /* 2. See if we must expand a macro */

    if (lm = macro_expansion_required(leaf))
      do_imap_macro_mail_search(mailbox, parent, lm, leaf);
    else
      do_imap_mail_search(mailbox, parent, leaf);

    /* replace the switched predicates if necessary */

    if (old_pred) {
      fs_give((void **) &new_pred);   /* Toss the dynamic predicate */
      leaf->predicate = old_pred;	      /* restore the static one */
    }


    /* If we have found something, hold on to it */

    current = mailbox->found;

    /* reset the search results */

    mailbox->found = NULL;


    /* Look for atomic NOT binding */

    switch (leaf->atomic_not) {
    case no_binding: 
      break;
    case bind_atom:
      negate_search(mailbox,current,parent->message_list);
      current = mailbox->found;
      mailbox->found = NULL;
      break;
    }

    /* Examine if an expression is being NOT bound */

    switch (leaf->expression_not) {
    case no_binding: 
      break;
    case bind_expression:
      not_at_end = TRUE;
      break;
    }

    /* Look for operation binding */

    switch (leaf->expression) {

    case simple:

      /* apply operation to the result, EG: a OR b */

      result = apply_op_to_operands(mailbox, leaf->operator,
				    &result, &current);
      break;

    case start:
      exp_op = leaf->operator;		/* save for the end */
      result = current;
      break;

    case intermediate:

      result = apply_op_to_operands(mailbox,leaf->operator,
				    &result,&current);
      break;

    case end:
      /* end-of-expression. Do leaf->operation, "not_at_end", and
	 exp_op */
      result = apply_op_to_operands(mailbox, leaf->operator,
				    &result,&current);

      if (not_at_end) {		
	negate_search(mailbox,current,parent->message_list);
	current = mailbox->found;
	mailbox->found = NULL;
	current = NULL;
	not_at_end = FALSE;
      }

      /* Now apply the expression operand, result_mask OP search_mask: */
      result = apply_op_to_operands(mailbox, exp_op, 
				    &result,&current);
      break;
    }
    leaf = leaf->next;
  }


  if(current)
    free_message_list(current,TRUE,FALSE);
  return(result);

}


