/* addressbook.c
 *
 *
 * $Author$
 * $Date$
 * $Id$
 *
 * Mike Macgirvin <Mike_Macgirvin@CAMIS.Stanford.EDU>
 *
 * Copyright 1994 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL 
 * WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE, AND IN NO EVENT SHALL THE LELAND STANFORD JUNIOR 
 * UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 */

/*
 * $Log$
 * 
 */

#include "ml.h"

void address_book_quit();
void address_book_select();
void stuff_address_book();
void address_save();
void address_add();
void address_remove();
void address_reset();
void address_book_context_switch();
void address_search();
void address_modify();
Boolean address_searchit();
void address_sort();
void address_help();

Menu address_book_file_menu[] = {
  { "Add Address", "ADDRESS_ADD", 'A',
      address_add, NULL, 0, NULL, NULL, BTN_SELECTION },
  { "Remove Address", "ADDRESS_REMOVE", 'R',
      address_remove, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Modify Address", "ADDRESS_MODIFY", 'M',
      address_modify, NULL, 0, NULL, NULL, BTN_NOSELECTION },
  { "Close Window", "ADRESS_CLOSE", 'C',
      address_book_quit, NULL, 0, NULL, NULL, BTN_ON },

};


Menu address_book_menu[] = {
  { "File", "ADDRESS_FILE", 'F',
      NULL, address_book_file_menu, XtNumber(address_book_file_menu),
      NULL, NULL, BTN_ON },
  { "Search", "ADDRESS_SEARCH", 'S',
      address_search, NULL, 0, NULL, NULL, BTN_ON },
  { "Reset", "ADDRESS_RESET", 'R',
      address_reset, NULL, 0, NULL, NULL, BTN_ON },
  { "Help", "ATTACH_HELP!", 'H',
      address_help, NULL, 0, NULL, NULL, BTN_ON },

};





Boolean create_address_book_window(w)
     Widget w;
{

  Arg args[ARGLISTSIZE];
  int n = 0;
  Address_Book *address_book;
  Position x, y;
  Widget form, menubar;
  XtTranslations translations;


  if(session.address_book == NULL) {
    address_book = (Address_Book *) fs_get(sizeof(Address_Book));
    address_book->is_realized = FALSE;
    session.address_book = address_book;
  }
  else
    address_book = session.address_book;

  if(session.address_book->is_realized == TRUE) {
    XRaiseWindow(display,XtWindow(address_book->shell));
    return;
  }


  address_book->current = NULL;
  address_book->button_state = BTN_ON | BTN_NOSELECTION;


  get_pointer_position(w, &x, &y);

  XtSetArg(args[n], XtNx, x); n ++;
  XtSetArg(args[n], XtNy, y); n ++;

  XtSetArg(args[n], XtNallowShellResize, TRUE); n ++;
  address_book->shell = XtCreateWidget("Address Book",
					  topLevelShellWidgetClass, w,
					  args, n); n = 0;

  XmAddTabGroup(address_book->shell);

  form = XmCreateForm(address_book->shell, "form", args, n ); n = 0;

  XtSetArg(args[n], XmNtopAttachment,   XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNleftAttachment,  XmATTACH_FORM); n ++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n ++;
  menubar = XmCreateMenuBar(form, "menubar", args, n); n = 0;
  XtManageChild(menubar);

  create_menu_buttons(NULL, menubar, address_book_menu, 
		      XtNumber(address_book_menu),
		      address_book->button_state, (XtPointer) address_book);

  address_book->name =
    create_text_field(form, menubar, 
		      "Name   :", NULL,40, NULL, NULL);

  address_book->address =
    create_text_field(form, address_book->name, 
		      "Address:", NULL, 40, NULL, NULL);

  address_book->comment =
    create_text_field(form, address_book->address,
		      "Comment:", NULL, 40, NULL, NULL);

  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC);             n ++;
  XtSetArg(args[n], XmNlistSizePolicy,         XmCONSTANT);           n ++;
  XtSetArg(args[n], XmNvisibleItemCount,       10 );                  n ++; 
  XtSetArg(args[n], XmNselectionPolicy,        XmSINGLE_SELECT);      n ++;
  address_book->list = XmCreateScrolledList(form,"list",args,n);      n = 0;

  XtAddCallback(address_book->list, XmNsingleSelectionCallback, 
		address_book_select, address_book);
  XtAddCallback(address_book->list, XmNdefaultActionCallback, 
		address_book_select, address_book);

  XtSetArg(args[n], XmNtopAttachment,          XmATTACH_WIDGET);       n ++;
  XtSetArg(args[n], XmNtopWidget,              address_book->comment); n ++;
  XtSetArg(args[n], XmNleftAttachment,         XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNrightAttachment,        XmATTACH_POSITION);     n ++;
  XtSetArg(args[n], XmNrightPosition,          50);                    n ++;
  XtSetArg(args[n], XmNbottomAttachment,       XmATTACH_FORM);         n ++;

  XtSetValues(XtParent(address_book->list), args, n);                  n = 0;

  XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmSTATIC);              n ++;
  XtSetArg(args[n], XmNlistSizePolicy,         XmCONSTANT);            n ++;
  XtSetArg(args[n], XmNvisibleItemCount,       10 );                   n ++; 
  XtSetArg(args[n], XmNselectionPolicy,        XmSINGLE_SELECT);       n ++;
  address_book->search_list = 
    XmCreateScrolledList(form,"search_list",args,n);                   n = 0;

  XtAddCallback(address_book->search_list, XmNsingleSelectionCallback, 
		address_book_select, address_book);
  XtAddCallback(address_book->search_list, XmNdefaultActionCallback, 
		address_book_select, address_book);

  XtSetArg(args[n], XmNtopAttachment,          XmATTACH_WIDGET);       n ++;
  XtSetArg(args[n], XmNtopWidget,              address_book->comment); n ++;
  XtSetArg(args[n], XmNrightAttachment,        XmATTACH_FORM);         n ++;
  XtSetArg(args[n], XmNleftAttachment,         XmATTACH_POSITION);     n ++;
  XtSetArg(args[n], XmNleftPosition,           50);                    n ++;
  XtSetArg(args[n], XmNbottomAttachment,       XmATTACH_FORM);         n ++;

  XtSetValues(XtParent(address_book->search_list), args, n);           n = 0;

  address_sort(address_book);
  stuff_address_book(address_book);

  XtManageChild(address_book->search_list);
  XtManageChild(address_book->list); 

  XtManageChild(form);
  XtManageChild(address_book->shell);
  XtRealizeWidget(address_book->shell);
  address_book->is_realized = TRUE;

}


void address_book_quit(w,address_book,xp)
     Widget w;
     Address_Book *address_book;
     XtPointer xp;
{
  XtDestroyWidget(address_book->shell);
  address_book->is_realized = FALSE;

}



void address_book_select(w,address_book,cb)
     Widget w;
     Address_Book *address_book;
     XmListCallbackStruct *cb;
{

  char *str;
  Address_Book_Info *info;
  Widget other;

  if(w == address_book->list)
    other = address_book->search_list;
  else
    other = address_book->list;

  XmListDeselectAllItems(other);

  XmStringGetLtoR(cb->item,XmSTRING_DEFAULT_CHARSET,&str);

  if((address_book->current != NULL) && 
     (strcmp(str,address_book->current->name) == STRMATCH) &&
     (address_book->current_list == w)) { 
    address_book_context_switch(address_book,address_book->current,NULL);
    address_book->button_state |= BTN_NOSELECTION;
    address_book->button_state &= ~BTN_SELECTION;
  }
  else {
    for(info = address_book->address_book_info; info ; info = info->prev ) {
      if((strcmp(info->name,str)) == STRMATCH) {
	address_book_context_switch(address_book,address_book->current, info);
	address_book->button_state |= BTN_SELECTION;
	address_book->button_state &= ~BTN_NOSELECTION;
      }
    }
  }

  fs_give((void **) &str);
  address_book->current_list = w;
  check_buttons(address_book_menu, XtNumber(address_book_menu), 
		address_book->button_state);
}


void address_book_context_switch(address_book,old,new)
     Address_Book *address_book;
     Address_Book_Info *old, *new;
{
  
  if(old) {
    if(old->name) 
      fs_give((void **) &old->name);
    old->name = GetTextField(address_book->name);
    if(old->address)
      fs_give((void **) &old->address);
    old->address = GetTextField(address_book->address);
    if(old->comment)
      fs_give((void **) &old->comment);
    old->comment = GetTextField(address_book->comment);
  }
  if(new) {
    XmTextSetString(address_book->name,new->name);
    XmTextSetString(address_book->address,new->address);
    XmTextSetString(address_book->comment,new->comment);
    
  }
  else {
    XmTextSetString(address_book->name,EMPTYSTR);
    XmTextSetString(address_book->address,EMPTYSTR);
    XmTextSetString(address_book->comment,EMPTYSTR);
  }

  address_book->current = new;

}


void stuff_address_book(address_book)
     Address_Book *address_book;
{
  Address_Book_Info *info;
  XmString xstr;
  
  for(info = address_book->address_book_info; info ; info = info->prev) {
    xstr = XmStringCreateSimple(info->name);
    XmListAddItemUnselected(address_book->list,xstr,0);
    XmStringFree(xstr);
  }

}


void address_search(w,address_book,xp)
     Widget w;
     Address_Book *address_book;
     XtPointer xp;
{
  Address_Book_Info *info;
  XmString xstr;
  Boolean found = FALSE;

  /* Search on the current window contents */

  char *name     = GetTextField(address_book->name);
  char *address  = GetTextField(address_book->address);
  char *comment  = GetTextField(address_book->comment);

  /* Nothing to search for. */

  if(*name == NUL_TERM && *address == NUL_TERM && *comment == NUL_TERM)
    return;

  /* Let 'em know we're busy. */

  set_watch_cursors();

  address_book->current = FALSE;
  XmListDeleteAllItems(address_book->search_list);

  /* 
   * Find a match on any field in our internal list, and
   * put it in the search list widget
   */

  for(info = address_book->address_book_info; info; info = info->prev) {
    if((address_searchit(info->name,name))
       || (address_searchit(info->address,address))
       || (address_searchit(info->comment,comment))) {
      found = TRUE;
      xstr = XmStringCreateSimple(info->name);
      XmListAddItemUnselected(address_book->search_list,xstr,0);
      XmStringFree(xstr);
    }
  }

  /* 
   * Popup the first one in the list. This will call our selection
   * callback before it's done, so we don't have to tweak anything.
   */

  if(found) {
    XmListSelectPos(address_book->search_list,1,TRUE);
  }

  reset_cursors();

}


/*
 * This isn't the best search algorithm, but it doesn't have to be.
 * We're looking for any substring match, case insensitive.
 * Empty strings or null strings don't count.
 */

Boolean address_searchit(str,pat)
     char *str;
     char *pat;
{
  register char *ptr;

  if((str == NULL) || (pat == NULL) || (*pat == NUL_TERM))
    return(FALSE);
  for(ptr = str; *ptr; ptr ++ ) 
    if(strncasecmp(ptr, pat, strlen(pat)) == STRMATCH)
      return(TRUE);
  return(FALSE);
}


void address_remove(w,address_book,xp)
     Widget w;
     Address_Book *address_book;
     XtPointer xp;
{
  Address_Book_Info *info = address_book->current;

  if(! info)
    return;

  if(info->prev)
    info->prev->next = info->next;
  if(info->next)
    info->next->prev = info->prev;
  else
    address_book->address_book_info = info->prev;

  if(address_book->current == info)
    address_book->current = NULL;

  free_address_book_info(info);
  
  set_watch_cursors();
  save_defaults();
  address_reset(w,address_book, NULL);
  reset_cursors();
}

void address_add(w,address_book,xp)
     Widget w;
     Address_Book *address_book;
     XtPointer xp;
{
  Address_Book_Info *info;
  Address_Book_Info *curr;

  info = new_address_book_info();

  info->name = GetTextField(address_book->name);

  if(*info->name == NUL_TERM) {
    free_address_book_info(info);
    return;
  }
    
  for(curr = address_book->address_book_info; curr; curr = curr->prev) {
    if(strcasecmp(curr->name,info->name) == STRMATCH) {
      free_address_book_info(info);
      return;
    }
  }

  set_watch_cursors();
  info->address = GetTextField(address_book->address);
  info->comment = GetTextField(address_book->comment);
  info->next = NULL;
  info->prev = address_book->address_book_info;
  if(info->prev)
    info->prev->next = info;
  address_book->address_book_info = info;
  address_sort(address_book);
  save_defaults();
  address_reset(w,address_book, NULL);
  reset_cursors();

}

void address_modify(w,address_book,xp)
     Widget w;
     Address_Book *address_book;
     XtPointer xp;
{
  Address_Book_Info *info;
  char *name = GetTextField(address_book->name);

  info = address_book->current;

  if((! info) || (*name == NUL_TERM))
    return;

  set_watch_cursors();

  if(info->name)
    fs_give((void **) &info->name);
  info->name = name;

  if(info->address)
    fs_give((void **) &info->address);
  info->address = GetTextField(address_book->address);

  if(info->comment)
    fs_give((void **) &info->comment);
  info->comment = GetTextField(address_book->comment);

  address_sort(address_book);
  save_defaults();
  address_reset(w,address_book, NULL);
  reset_cursors();
}


void address_reset(w,address_book,xp)
     Widget w;
     Address_Book *address_book;
     XtPointer xp;
{

  XmListDeleteAllItems(address_book->search_list);
  XmListDeselectAllItems(address_book->search_list);
  XmListDeselectAllItems(address_book->list);
  XmListDeleteAllItems(address_book->list);
  address_book->button_state |= BTN_NOSELECTION;
  address_book->button_state &= ~BTN_SELECTION;


  address_book_context_switch(address_book,address_book->current, NULL);
  stuff_address_book(address_book);
  check_buttons(address_book_menu, XtNumber(address_book_menu),
		address_book->button_state);
}



int compare(adr1,adr2)
     Address_Book_Info **adr1, **adr2;
{

  return(strcmp((*(adr1))->name,(*(adr2))->name));


}


void address_sort(address_book)
     Address_Book *address_book;
{
  Address_Book_Info **sort;
  Address_Book_Info *info;
  int n = 0;
  int total = 0;

  for(info = address_book->address_book_info; info; info = info->prev)
    total ++;

  if(total < 2)
    return;

  sort = (Address_Book_Info **) fs_get(total * sizeof(Address_Book_Info *));

  for(info = address_book->address_book_info; info; info = info->prev) {
    sort[n] = info; 
    n ++;
  }

  qsort((char *) sort,total,sizeof(Address_Book_Info **),compare);

  for(n = 0; n < total; n ++) {
    info = sort[n];
    if(n == 0) {
      address_book->address_book_info = info;
      info->next = NULL;
      info->prev = sort[n + 1];
    }
    else {
      info->next = sort[n - 1];
      if(n == (total - 1))
	info->prev = NULL;
      else
	info->prev = sort[n + 1];
    }
  }

  fs_give((void **) &sort);

}


void address_help(w,address_book,xp)
     Widget w;
     Address_Book *address_book;
     XtPointer xp;
{

  help(address_book->shell, "Address Book Help", "addressbook.help");

}
