/*
 * Copyright (c) 1991,1993  Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <lber.h>
#include <ldap.h>
#ifndef __STDC__
#include <memory.h>
#endif
#include <sys/types.h>
#include <malloc.h>
#include "ud.h"

extern struct entry2 Entry; 
extern int verbose;
extern LDAP *ld;

extern LDAPMessage *find();

#ifdef DEBUG
extern int debug;
#endif

modify(who)
char *who;
{
#ifdef UOFM
	void set_updates();	/* routine to modify noBatchUpdates */
#endif
	void set_pw();		/* routine to modify a password */
	extern char bound_as[]; /* who this user is bound as */
	extern char me[];	/* set if user has used I AM */
	LDAPMessage *mp;	/* returned from find() */
	char *dn;		/* distinguished name */
	char **rdns;		/* for fiddling with the DN */
	char name[BUFSIZ];	/* entry to modify */
	static char ans[RESP_SIZE];	/* for holding user input */
#ifdef UOFM
	static char printed_warning = 0;	/* for use with the */
	extern char * fetch_update_setting();	/* noBatchUpdates attr */
#endif
	int is_a_group;		/* TRUE if it is; FALSE otherwise */

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->modify(%s)\n", who);
#endif
	/*
	 *  Require them to bind first if the are modifying a group.
	 */
	if (!strcmp(bound_as, DEFAULT_BOUND_AS)) {
		if (auth((char *) NULL, (char *) NULL) < 0)
			return;
	}

	/*
	 *  First, decide what entry we are going to modify.  If the
	 *  user has not included a name on the modify command line,
	 *  we will use the person who was last looked up with a find
	 *  command.  If there is no value there either, we don't know
	 *  who to modify.
	 *
	 *  Once we know who to modify, be sure that they exist, and
	 *  parse out their DN.
	 */
	if ((who == NULL) && (Entry.name != NULL))
		who = Entry.name;
	else if (who == NULL) {
		printf("  Modify who? ");
		fflush(stdout);
		fetch_buffer(name, sizeof(name), stdin);
		if (name[0] != '\0')
			who = name;
		else
			return;
	}
	if ((mp = find(who)) == NULL) {
		(void) ldap_msgfree(mp);
		printf("  Could not locate \"%s\" in the Directory.\n", who);
		return;
	}
	dn = ldap_get_dn(ld, ldap_first_entry(ld, mp));
	rdns = ldap_explode_dn(dn, TRUE);
	if (verbose)
		printf("\n  Modifying Directory entry of \"%s\"\n", *rdns);

#ifdef UOFM
	/*
	 *  If verbose mode is turned on and the user has not set a value
	 *  for noBatchUpdates, warn them that what they are about to do
	 *  may be overwritten automatically by that Stinkbug.
	 */
	(void) fetch_update_setting(dn);
	if (verbose && !printed_warning && (ld->ld_errno == LDAP_NO_SUCH_ATTRIBUTE)) {
		printed_warning = 1;
		printf("\n  WARNING!\n");
		printf("  You are about to make a modification to an X.500 entry\n");
		printf("  that has its \"automatic updates\" field set to ON.\n");
		printf("  This means that the entry will be automatically updated\n");
		printf("  each month from official University sources like the\n");
		printf("  Personnel Office.  With \"automatic updates\" set to ON,\n");
		printf("  the following fields will be overwritten each month:\n");
		printf("         Title, home address and phone,\n");
		printf("         business address and phone\n");
		printf("  If you modify any of these fields, you may want to change\n");
		printf("  the \"automatic updates\" field to OFF so that your\n");
		printf("  changes will not be overwritten.  You may change this\n");
		printf("  setting by choosing \"u\" at the \"Modify what?\" prompt\n");
	}
#endif

	/*
	 *  Current values for user 'who' are being held in 'mp'. We
	 *  should parse up that buffer and fill in the Entry structure.
	 *  Once we're done with that, we can find out which fields the
	 *  user would like to modify.
	 */
	parse_answer(mp);
	is_a_group = isgroup();
	(void) ldap_msgfree(mp);
	printf("  You now need to specify what field you'd like to modify.\n\n");
	printf("  Choices are:\n");
	printf("  -----------------------\n");
	print_mod_list(is_a_group);
	printf("\n  Hitting RETURN will abort the process.\n");
	for (;;) {
		printf("\n  Modify what? ");
		fflush(stdout);
		fetch_buffer(ans, sizeof(ans), stdin);
		if (ans[0] == '\0')
			break;
		if (perform_action(ans, dn, is_a_group) == 0)
			break;
	}
	(void) free(dn);
	return;
}

void set_pw(who)
char *who;
{
	char *cp;
	static char new[513];
	extern char * mygetpass();
	extern char password[];
	static char *newpasswd[2];
	LDAPMod mod, *mods[2];

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->set_pw(%s)\n", who);
#endif
	mods[0] = &mod;
	mods[1] = (LDAPMod *) NULL;
	mod.mod_op = LDAP_MOD_REPLACE;
	mod.mod_type = "userPassword";
	mod.mod_values = newpasswd;
	if (verbose) {
		printf("  You indicated that you wanted to change your X.500 password.\n");
		printf("  You will now be prompted for your old, existing X.500 password.\n");
		printf("  Then you will be prompted twice for your new X.500 password.\n");
		printf("  If they match, and your old password is correct, your\n");
		printf("  password will be changed.\n\n");
#ifdef UOFM
		printf("  You cannot use this program to change your Kerberos password.\n\n");
#endif
	}
	cp = mygetpass("  Old password? ");
	if (strcmp(cp, password)) {
		printf("  Incorrect password -- not changed.\n");
		return;
	}
	cp = mygetpass("  New password? ");
	(void) strcpy(new, cp);
	cp = mygetpass("  Enter new password again to confirm: ");
	if (strcmp(cp, new)) {
		printf("  Password mismatch -- not changed.\n");
		return;
	}
	newpasswd[0] = cp;
	if (ldap_modify_s(ld, who, mods)) {
		ldap_perror(ld, "ldap_modify_s");
		return;
	}
	else if (verbose)
		printf("  Password changed.\n");
	ldap_uncache_entry( ld, who );
	(void) strcpy(password, cp);
	return;
}


/* generic routine for changing any field */
change_field(attr, who)
struct attribute attr;		/* attribute to change */
char *who;			/* DN of entry we are changing */
{

#define	IS_MOD(x)	(!strncasecmp(resp, (x), strlen(resp)))
				
	char *get_value();		/* routine to extract values */
	static char buf[4 * RESP_SIZE];	/* for printing things */
	static char resp[RESP_SIZE];	/* for user input */
	char *prompt, *prompt2, *more;
	register int i;				/* for looping thru values */
	static LDAPMod mod;
	static LDAPMod *mods[2] = { &mod };	/* passed to ldap_modify */
	static char *values[MAX_VALUES];	/* passed to ldap_modify */

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->change_field(%x, %s)\n", attr, who);
#endif
	/*
	 *  If there is no current value associated with the attribute,
	 *  then this is the easy case.  Collect one (or more) attributes
	 *  from the user, and then call ldap_modify_s() to write the changes
	 *  to the LDAP server.
	 */
	for (i = 0; i < MAX_VALUES; i++)
		values[i] = NULL;
	if (attr.values == (char **) NULL) {
		printf("\n  No current \"%s\"\n", attr.output_string);
		values[0] = get_value(attr.quipu_name, "Enter a value");
		mod.mod_op = LDAP_MOD_REPLACE;
		mod.mod_type = attr.quipu_name;
		mod.mod_values = values;
		for (i = 1; i < MAX_VALUES; i++) {
			printf("  Do you wish to add an additional value? ");
			fflush(stdout);
			fetch_buffer(resp, sizeof(resp), stdin);
			if ((resp[0] == 'y') || (resp[0] == 'Y'))
				values[i] = get_value(attr.quipu_name, "Enter an additional value");
			else
				break;
		}
#ifdef DEBUG
		if (debug & D_MODIFY) {
			printf("  ld = 0x%x\n", ld);
			printf("  who = [%s]\n", who);
			printf("  mods[0]->mod_op = %1d\n", mods[0]->mod_op);
			printf("  mods[0]->mod_type = %s\n", mods[0]->mod_type);
			for (i = 0; mods[0]->mod_values[i] != NULL; i++)
				printf("  mods[0]->mod_values[%1d] = %s\n", i, mods[0]->mod_values[i]);
		}
#endif
		if (ldap_modify_s(ld, who, mods)) {
			ldap_perror(ld, "ldap_modify_s");
			return;
		}
		else if (verbose)
			printf("  Modification of '%s' complete.\n", attr.output_string);
		ldap_uncache_entry( ld, who );
		for (i--; i > 0; i--)
			(void) free(values[i]);
	}
	/*
	 *  There are values for this attribute already.  In this case,
	 *  we want to allow the user to delete all existing values,
	 *  add additional values to the ones there already, or just
	 *  delete some of the values already present.  DIXIE does not
	 *  handle modifications where the attribute occurs on the LHS
	 *  more than once.  So to delete entries and add entries, we
	 *  need to call ldap_modify() twice.
	 */
	else {
		/*
		 *  If the attribute holds values which are DNs, print them
		 *  in a most pleasant way.
		 */
		sprintf(buf, "%s:  ", attr.output_string);
		if (!strcmp(attr.quipu_name, "owner"))
			(void) print_DN(attr);
		else
			(void) print_values(attr);

		if (verbose) {
			printf("  You may now:\n");
			printf("    Add additional values to the existing ones, OR\n");
			printf("    Clear all values listed above, OR\n");
			printf("    Delete one of the values listed above, OR\n");
			printf("    Replace all of the values above with new ones.\n");
		}
		printf("\n  Add, Clear, Delete, or Replace? ");
		fflush(stdout);
		fetch_buffer(resp, sizeof(resp), stdin);

		/*
		 *  Bail if they just hit the RETURN key.
		 */
		if (resp[0] == '\0') {
			if (verbose)
				printf("\n  No changes made.\n");
			return;
		}

		/*
		 *  If the want to clear the values, just do it.
		 */
		mod.mod_type = attr.quipu_name;
		mod.mod_values = values;
		if (IS_MOD("clear")) {
			mod.mod_op = LDAP_MOD_DELETE;
			mod.mod_values = NULL;
#ifdef DEBUG
			if (debug & D_MODIFY) {
				printf("Clearing attribute '%s'\n", attr.quipu_name);
				printf("who = [%s]\n", who);
				printf("mod = [%d] [%s] [%x]\n", mod.mod_op,
					mod.mod_type, mod.mod_values);
			}
#endif
			if (ldap_modify_s(ld, who, mods)) {
				ldap_perror(ld, "ldap_modify_s");
				return;
			}
			else if (verbose)
				printf("  '%s' has been cleared.\n", attr.output_string);
			ldap_uncache_entry( ld,  who );
			return;
		}

		if (IS_MOD("add")) {
			prompt = "Enter the value you wish to add";
			more = "  Add an additional value? ";
			prompt2 = "Enter another value you wish to add";
			mod.mod_op = LDAP_MOD_ADD;
		}
		else if (IS_MOD("delete")) {
			prompt = "Enter the value you wish to delete";
			more = "  Delete an additional value? ";
			prompt2 = "Enter another value you wish to delete";
			mod.mod_op = LDAP_MOD_DELETE;
		}
		else if (IS_MOD("replace")) {
			prompt = "Enter the new value";
			more = "  Add an additional value? ";
			prompt2 = "Enter another value you wish to add";
			mod.mod_op = LDAP_MOD_REPLACE;
		}
		else {
			printf("  No changes made.\n");
			return;
		}

		values[0] = get_value(attr.quipu_name, prompt);
		for (i = 1; i < MAX_VALUES; i++) {
			printf(more);
			fflush(stdout);
			fetch_buffer(resp, sizeof(resp), stdin);
			if ((resp[0] == 'y') || (resp[0] == 'Y'))
				values[i] = get_value(attr.quipu_name, prompt2);
			else
				break;
		}

		/* if the first value in the value-array is NULL, bail */
		if (values[0] == NULL) {
			if (verbose)
				printf("  No modification made.\n");
			return;
		}
#ifdef DEBUG
		if (debug & D_MODIFY) {
			printf("  ld = 0x%x\n", ld);
			printf("  who = [%s]\n", who);
			printf("  mods[0]->mod_op = %1d\n", mods[0]->mod_op);
			printf("  mods[0]->mod_type = %s\n", mods[0]->mod_type);
			for (i = 0; mods[0]->mod_values[i] != NULL; i++)
				printf("  mods[0]->mod_values[%1d] = %s\n", i, mods[0]->mod_values[i]);
		}
#endif
		if (ldap_modify_s(ld, who, mods)) {
			ldap_perror(ld, "ldap_modify_s");
			return;
		}
		else if (verbose)
			printf("  Modifications to '%s' complete.\n", attr.output_string);
		ldap_uncache_entry( ld, who );
		for (i--; i > 0; i--)
			(void) free(values[i]);
	}
}

/*
 *  These are used to size the buffers we use when collecting values that
 *  can cross more than one line.
 */
#define LINE_SIZE       80
#define MAX_LINES       10

char *get_value(id, prompt)
char *id, *prompt;
{
	char *cp;		/* for the malloc() */
	int count;		/* line # of new value -- if multiline */
	int multiline = 0;	/* 1 if this value is multiline */
	static char line[LINE_SIZE];	/* raw line from user */
	static char buffer[MAX_LINES * LINE_SIZE];	/* holds ALL of the 
							   lines we get */
#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->get_value(%s, %s)\n", id, prompt);
#endif
	/*
	 *  To start with, we have one line of input from the user.
	 *
	 *  Addresses and multiline description can span multiple lines.
	 *  Other attributes may not.
	 */
	count = 1;
	(void) memset(buffer, 0, sizeof(buffer));
#ifdef UOFM
	if (!strcmp(id, "postalAddress") || !strcmp(id, "homePostalAddress") || !strcmp(id, "multiLineDescription"))
#else
	if (!strcmp(id, "postalAddress") || !strcmp(id, "homePostalAddress"))
#endif
		multiline = 1;
	printf("\n  %s:\n", prompt);

	/* fetch lines */
	for (;;) {
		if (multiline)
			printf(" %1d: ", count);
		else
			printf("  > ");
		fflush(stdout);
		fetch_buffer(line, sizeof(line), stdin);

		/*
		 *  Big hack here.
		 *
		 *  If the attribute which we are gathering is a "owner"
		 *  then we should lookup the name.  The user is going to
		 *  either have to change the search base before doing the
		 *  modify, or the person is going to have to be within the
		 *  scope of the current search base, or they will need to
		 *  type in a UFN.
		 */
		if (!strcmp(id, "owner")) {
			LDAPMessage *lmp, *elmp;
			char *tmp;
			
			lmp = find(line);
			if (lmp == (LDAPMessage *) NULL) {
				printf("  Could not find \"%s\" in the Directory\n", line);
				if (verbose) 
					format("Owners of groups must be valid entries in the X.500 Directory.  The name you have typed above could not be found in the X.500 Directory.", 72, 2);
				return(NULL);
			}
			elmp = ldap_first_entry(ld, lmp);
			if (lmp == (LDAPMessage *) NULL) {
				ldap_perror(ld, "ldap_first_entry");
				return(NULL);
			}
			tmp = ldap_get_dn(ld, elmp);
			strcpy(buffer, tmp);
			(void) ldap_msgfree(lmp);
			break;
		}

		if (line[0] == '\0')
			break;

		/*
		 *  Separate lines of multiline attribute values with
		 *  dollar signs.  Copy this line into the buffer we
		 *  use to collect up all of the user-supplied input
		 *  lines.  If this is not a multiline attribute, we
		 *  are done.
		 */
		if (count++ > 1)
			(void) strcat(buffer, "$");
		(void) strcat(buffer, line);
		if (!multiline)
			break;
	}
	if (buffer[0] == '\0')
		return(NULL);
#ifdef DEBUG
	if (debug & D_MODIFY)
		printf("  Value is [%s]\n", buffer);
#endif
	if ((cp = (char *) malloc((unsigned) strlen(buffer) + 1)) == NULL)
		fatal("malloc");
	strcpy(cp, buffer);
	return(cp);
}

#ifdef UOFM

void set_updates(who)
char *who;
{
	char *cp, *s;
	extern char * fetch_update_setting();
	static char response[16];

	static char value[6];
	static char *newsetting[2] = { value, NULL };
	LDAPMod mod, *mods[2];

#ifdef DEBUG
	if (debug & D_TRACE)
		printf("->set_updates(%s)\n", who);
#endif
	mods[0] = &mod;
	mods[1] = (LDAPMod *) NULL;
	mod.mod_op = LDAP_MOD_REPLACE;
	mod.mod_type = "noBatchUpdates";
	mod.mod_values = newsetting;
	/* explain what the implications are */
	if (verbose) {
		printf("\n  By default, updates that are received from the Personnel\n");
		printf("  Office and the Office of the Registrar are applied to all\n");
		printf("  entries in the X.500 database each month.  Sometimes this\n");
		printf("  feature is undesirable.  For example, if you maintain your\n");
		printf("  entry in the X.500 database manually, you may not want to\n");
		printf("  have these updates applied to your entry, possibly overwriting\n");
		printf("  correct information with out-dated information.\n\n");
	}

	/* fetch the current setting */
	if ((cp = fetch_update_setting(who)) == NULL)
		return;
	if (!strcmp(cp, "TRUE"))
		printf("  Automatic updates are currently turned OFF\n");
	else if (!strcmp(cp, "FALSE"))
		printf("  Automatic updates are currently turned ON\n");
	else {
		fprintf(stderr, "  Unknown update flag -> [%s]\n", cp);
		return;
	}

	/* see if they want to change it */
	printf("\n  Change this setting [no]? ");
	fflush(stdout);
	(void) fgets(response, sizeof(response), stdin);
	for (s = response; isspace(*s); s++)
			;
	if ((*s == 'y') || (*s == 'Y')) {
		if (!strcmp(cp, "TRUE"))
			strcpy(value, "FALSE");
		else
			strcpy(value, "TRUE");
		if (ldap_modify_s(ld, who, mods)) {
			ldap_perror(ld, "ldap_modify_s");
			return;
		}
		else if (verbose)
			printf("  Setting has been changed\n");
		ldap_uncache_entry( ld, who );
		return;
	}
	if (verbose)
			printf("  Setting has not been changed\n");
}

#endif


print_mod_list(group)
int group;
{

	if (group == TRUE) {
		printf("   1 ->  Business phone\n");
		printf("   2 ->  Business address\n");
		printf("   3 ->  Fax number\n");
		printf("   4 ->  Associated domain\n");
		printf("   5 ->  Errors-to\n");
		printf("   6 ->  Requests-to\n");
		printf("   7 ->  Member list\n");
		printf("   8 ->  May-join flag\n");
		printf("   9 ->  Owner\n");
		printf("  10 ->  Common names\n");
#ifdef UOFM
		printf("  11 ->  Description\n");
#endif
		printf("   ? ->  Print this list\n\n");
	}
	else {
		printf("   1 ->  Business phone\n");
		printf("   2 ->  Business address\n");
		printf("   3 ->  Fax number\n");
		printf("   4 ->  E-mail address\n");
		printf("   5 ->  Title\n");
		printf("   6 ->  Home phone\n");
		printf("   7 ->  Home address\n");
		printf("   8 ->  X.500 password\n");
#ifdef UOFM
		printf("   9 ->  Description\n");
		printf("  10 ->  Automatic updates\n");
#endif
		printf("   ? ->  Print this list\n\n");
		}
#ifdef UOFM
	printf("  To add nicknames, send mail to x500-nicknames@umich.edu\n");
#endif
}

perform_action(choice, dn, group)
char choice[];
char *dn;
int group;
{
	extern void do_members();

	switch (atoi(choice)) {

	case 0 :
		printf("\n  Choices are:\n");
		printf("  -----------------------\n");
		print_mod_list(group);
		return(1);
		/* NOTREACHED */

	case 1 :
		change_field(Entry.attrs[attr_to_index("telephoneNumber")], dn);
		break;

	case 2 :
		change_field(Entry.attrs[attr_to_index("postalAddress")], dn);
		break;

	case 3 :
		change_field(Entry.attrs[attr_to_index("facsimileTelephoneNumber")], dn);
		break;
		
	case 4 :
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("associatedDomain")], dn);
		else
			change_field(Entry.attrs[attr_to_index("mail")], dn);
		break;

	case 5 :
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("rfc822ErrorsTo")], dn);
		else
			change_field(Entry.attrs[attr_to_index("title")], dn);
		break;

	case 6 :
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("rfc822RequestsTo")], dn);
		else
			change_field(Entry.attrs[attr_to_index("homePhone")], dn);
		break;

	case 7 :
		if (group == TRUE)
			(void) do_members(dn);
		else
			change_field(Entry.attrs[attr_to_index("homePostalAddress")], dn);
		break;

	case 8 :
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("joinable")], dn);
		else
			set_pw(dn);
		break;


	case 9 :
#ifdef UOFM
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("owner")], dn);
		else
			change_field(Entry.attrs[attr_to_index("multiLineDescription")], dn);
		break;
#else
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("owner")], dn);
		else {
			printf("\n  Choices are:\n");
			printf("  -----------------------\n");
			print_mod_list(group);
			return(1);
		}
		break;
#endif

	case 10 :
#ifdef UOFM
		if (group == TRUE) {
			if (verbose) {
				printf("\n");
				format("It is not necessary to use periods or underscores to separate the parts of an alias.  For example, to create the alias \"Big Dogs Fly at Night\", just type these characters as you see them here on the screen, without any extra characters, like periods, separating the parts.", 75, 2);
				printf("\n");
			}
			change_field(Entry.attrs[attr_to_index("cn")], dn);
		}
		else
			set_updates(dn);
		break;
#else
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("cn")], dn);
		else {
			printf("\n  Choices are:\n");
			printf("  -----------------------\n");
			print_mod_list(group);
			return(1);
		}
		break;
#endif

#ifdef UOFM
	case 11 :
		if (group == TRUE)
			change_field(Entry.attrs[attr_to_index("multiLineDescription")], dn);
		else {
			printf("\n  Choices are:\n");
			printf("  -----------------------\n");
			print_mod_list(group);
			return(1);
		}
		break;
#endif
	default :
		printf("\n  Choices are:\n");
		printf("  -----------------------\n");
		print_mod_list(group);
		return(1);
		/* NOTREACHED */
	}
	return(0);
}
