/*
 * query.c: for rcpt500 (X.500 email query responder)
 *
 * 18 June 1992 by Mark C Smith
 * Copyright (c) 1992 The Regents of The University of Michigan
 * All Rights Reserved
 */

#include <stdio.h>
#include <syslog.h>
#include <string.h>
#include <ctype.h>

#include <lber.h>
#include <ldap.h>
#include "rcpt500.h"

extern int dosyslog;
extern int strip;
extern int derefaliases;
extern int sizelimit;
extern char *ldaphost;
extern char *searchbase;
extern char *dapuser;
extern char *filterfile;

static char buf[ MAXSIZE ];
static char *errpreface = "Your query failed: ";


int
query_cmd( msgp, reply )
    struct msginfo	*msgp;
    char		*reply;
{
    LDAP		*ldp;
    LDAPMessage		*ldmsgp;
    char		*s, *dn;
    int			matches, rc;
    LDAPFiltDesc	*lfdp;
    LDAPFiltInfo	*lfi;
    static char	*attrs[] = { "cn", "title", NULL };

    if ( msgp->msg_arg == NULL ) {
	return( help_cmd( msgp, reply ));
    }

    remove_trailing_space( msgp->msg_arg );
    if ( *msgp->msg_arg == '\0' ) {
	return( help_cmd( msgp, reply ));
    }

    if (( lfdp = ldap_init_getfilter( filterfile )) == NULL ) {
	strcat( reply, errpreface );
	strcat( reply, "filter file configuration error.  Try again later." );
	return( 0 );
    }

    /*
     * open connection to LDAP server and bind as dapuser
     */
    if (( ldp = ldap_open( ldaphost, 0 )) == NULL ) {
	strcat( reply, errpreface );
	strcat( reply, "X.500 service unavailable.  Try again later." );
	ldap_getfilter_free( lfdp );
	return( 0 );
    }

    if ( ldap_simple_bind_s( ldp, dapuser, NULL ) != LDAP_SUCCESS ) {
	report_ldap_err( ldp, reply );
	ldap_unbind( ldp );
	ldap_getfilter_free( lfdp );
	return( 0 );
    }

    /*
     * set options for search and build filter
     */
    ldp->ld_deref = derefaliases;
    ldp->ld_sizelimit = sizelimit;

    for ( lfi = ldap_getfirstfilter( lfdp, "rcpt500", msgp->msg_arg );
	    lfi != NULL; lfi = ldap_getnextfilter( lfdp )) {
	if (( rc = ldap_search_s( ldp, searchbase, LDAP_SCOPE_SUBTREE,
		lfi->lfi_filter, attrs, 0, &ldmsgp )) == -1 ) {
	    report_ldap_err( ldp, reply );
	    ldap_unbind( ldp );
	    ldap_getfilter_free( lfdp );
	    return( 0 );
	}

	if (( matches = ldap_count_entries( ldp, ldmsgp )) != 0 ) {
	    break;
	}

	ldap_msgfree( ldmsgp );
    }

    if ( matches == 0 ) {
	sprintf( buf, "No matches were found for '%s'\n", msgp->msg_arg );
	strcat( reply, buf );
	ldap_unbind( ldp );
	ldap_getfilter_free( lfdp );
	return( 0 );
    }

    if ( ldp->ld_errno == LDAP_TIMELIMIT_EXCEEDED
	    || ldp->ld_errno == LDAP_SIZELIMIT_EXCEEDED ) {
	strcat( reply, "(Partial results only - a limit was exceeded)\n" );
    }

    if ( matches == 1 ) {
	sprintf( buf, "One %s match was found for '%s':\n", lfi->lfi_desc,
		msgp->msg_arg );
	strcat( reply, buf );
	dn = ldap_get_dn( ldp, ldap_first_entry( ldp, ldmsgp ));
	ldap_msgfree( ldmsgp );
	if ( do_read( ldp, dn, reply ) < 0 ) {
	    free( dn );
	    report_ldap_err( ldp, reply );
	    ldap_unbind( ldp );
	    ldap_getfilter_free( lfdp );
	    return( 0 );
	}
	free( dn );
	
    } else {
	sprintf( buf, "%d %s matches were found for '%s':\n",
		matches, lfi->lfi_desc, msgp->msg_arg );
	strcat( reply, buf );
	append_entry_list( reply, msgp->msg_arg, ldp, ldmsgp );
	ldap_msgfree( ldmsgp );
    }

    ldap_unbind( ldp );
    ldap_getfilter_free( lfdp );
    return( 0 );
}


append_entry_list( reply, query, ldp, ldmsgp )
    char	*reply;
    char	*query;
    LDAP	*ldp;
    LDAPMessage	*ldmsgp;
{
    LDAPMessage	*e;
    char	*dn, *rdn, *s, **title;
    int		free_rdn = 0;

    for ( e = ldap_first_entry( ldp, ldmsgp ); e != NULL;
		e = ldap_next_entry( ldp, e )) {
	dn = ldap_get_dn( ldp, e );
	if (( s = strchr( dn, ',' )) != NULL ) {
	    *s = '\0';
	}
	if (( s = strchr( dn, '=' )) == NULL ) {
	    rdn = dn;
	} else {
	    rdn = s + 1;
	}

#ifdef UOFM
	/*
	 * if this entry's rdn is an exact match for the thing looked up, we
	 * return the CN that has a digit after it, so that the user is
	 * returned something guaranteed to yield exactly one match if they
	 * pick it from the list and query it
	 */

	if ( strcasecmp( rdn, query ) == 0 ) {
	    char	**cn;
	    int		i;

	    if (( cn = ldap_get_values( ldp, e, "cn" )) != NULL ) {
		for ( i = 0; cn[i] != NULL; i++ ) {
		    if ( isdigit( *( cn[i] + strlen( cn[i] ) - 1 ))) {
			rdn = strdup( cn[i] );
			free_rdn = 1;
			break;
		    }
		}
		ldap_value_free( cn );
	    }
	}
#endif /* UOFM */

	title = ldap_get_values( ldp, e, "title" );
	sprintf( buf, "  %-20s    %s\n", rdn, title ? title[0] : "" );
	strcat( reply, buf );
	if ( title != NULL ) {
	    ldap_value_free( title );
	}
	free( dn );
	if ( free_rdn ) {
	    free( rdn );
	}
    }
}


static
append_attr( reply, label, attr, e, ldp, multiline )
    char	*reply;
    char	*label;
    char	*attr;
    LDAPMessage	*e;
    LDAP	*ldp;
    int		multiline;
{
    char	**val;
    int		i, gotone = 0;

    if ( (val = ldap_get_values( ldp, e, attr )) == NULL ) {
	if ( strcmp( attr, "mail" ) == 0 ) {
	    sprintf( buf, "%-19s\n", label );
	    strcat( reply, buf );
	    sprintf( buf,
		    "                    None registered in this service.\n" );
	    strcat( reply, buf );
	}
	return;
    }

    sprintf( buf, "%-19s\n", label );
    strcat( reply, buf );
    for ( i = 0; val[i] != NULL; i++ ) {
	if ( multiline ) {
	    char	*s, *p;

	    if ( gotone ) {
		sprintf( buf, "%-19s\n", label );
		strcat( reply, buf );
	    }

	    p = s = val[i];
	    while ( s = strchr( s, '$' ) ) {
		*s++ = '\0';
		while ( isspace( *s ) )
			s++;
		sprintf( buf, "                    %s\n", p );
		strcat( reply, buf );
		p = s;
	    }
	    sprintf( buf, "                    %s\n", p );
	    strcat( reply, buf );
	    gotone = 1;
	} else {
	    sprintf( buf, "                    %s\n", val[i] );
	    strcat( reply, buf );
	}
    }

    ldap_value_free( val );
}


int
do_read( ldp, dn, reply )
    LDAP	*ldp;
    char	*dn;
    char	*reply;
{
    LDAPMessage	*ldmp, *entry;
    int		i, rc;
    char	*s, *ufn;
    static char	*attrs[] = { "cn", "postalAddress", "uid", "title",
			     "mail", "telephoneNumber",
			     "facsimileTelephoneNumber", 
#ifdef UOFM
			     "multiLineDescription",
#else /* UOFM */
			     "description",
#endif /* UOFM */
			     NULL };

    if (( rc = ldap_search_s( ldp, dn, LDAP_SCOPE_BASE, "objectclass=*",
	    attrs, 0, &ldmp )) == -1 ) {
	return( -1 );
    }

    if ( ldap_count_entries( ldp, ldmp ) != 1 ) {
	ldap_msgfree( ldmp );
	return( -1 );
    }

    entry = ldap_first_entry( ldp, ldmp );
    dn = ldap_get_dn( ldp, entry );
    ufn = ldap_dn2ufn( dn );
    free( dn );
    for ( i = 0; i < strip; i++ ) {
	if ( (s = strrchr( ufn, ',' )) == NULL )
		break;
	*s = '\0';
    }
    sprintf( buf, "\"%s\"\n", ufn );
    strcat( reply, buf );
    free( ufn );

    append_attr( reply, "  Also known as:", "cn", entry, ldp, 0 );
    append_attr( reply, "  E-mail address:", "mail", entry, ldp, 0 );
    append_attr( reply, "  Fax number:", "facsimileTelephoneNumber",
	    entry, ldp, 0 );
    append_attr( reply, "  Business phone:", "telephoneNumber", entry, ldp, 0 );
    append_attr( reply, "  Business address:", "postalAddress", entry, ldp, 1 );
    append_attr( reply, "  Title:", "title", entry, ldp, 0 );
    append_attr( reply, "  Uniqname:", "uid", entry, ldp, 0 );
#ifdef UOFM
    append_attr( reply, "  Description:", "multiLineDescription",
	    entry, ldp, 1 );
#else /* UOFM */
    append_attr( reply, "  Description:", "description", entry, ldp, 1 );
#endif /* UOFM */
    ldap_msgfree( ldmp );
    return( 0 );
}


report_ldap_err( ldp, reply )
    LDAP	*ldp;
    char	*reply;
{
    strcat( reply, errpreface );
    strcat( reply, ldap_err2string( ldp->ld_errno ));
    strcat( reply, "\n" );
}


remove_trailing_space( s )
    char	*s;
{
    char	*p = s + strlen( s ) - 1;

    while ( isspace( *p ) && p > s ) {
	--p;
    }
    *(++p) = '\0';
}
