/*
 *	Copyright (c) 1990-92 Regents of the University of Michigan.
 *	All rights reserved.
 *
 *	tcp.c -- TCP communication related routines
 */

#include "tcp.h"

#include <Devices.h>
#include <Memory.h>

/*
 * local prototypes
 */
void		bzero( char *buf, unsigned long count );
pascal void setdoneflag( struct hostInfo *hip, char *donep );
pascal void tcp_asynchnotify( StreamPtr tstream, unsigned short event, Ptr userdatap,
								unsigned short term_reason, struct ICMPReport *icmpmsg );
OSErr		kick_mactcp( short *drefnump );


/*
 * open and return an pointer to a TCP stream (return NULL if error)
 * "buf" is a buffer for receives of length "buflen."
 */
tcpstream *
tcpopen( unsigned char * buf, long buflen ) {
	TCPiopb		pb;
	OSErr		err;
	tcpstream *	tsp;
	short		drefnum;

	if ( kick_mactcp( &drefnum ) != noErr ) {
		return ( nil );
	}

	if (nil == (tsp = (tcpstream *)NewPtrClear(sizeof(tcpstream)))) {
		return( nil );
	}
	
	tsp->drefnum = drefnum;
	
	if ( buflen == 0 ) {
		buflen = TCP_BUFSIZ;
	}
	
	if ( buf == NULL && 
		 (nil == ( buf = tsp->tcps_buffer = (unsigned char *)NewPtr( buflen )))) {
		DisposePtr( (Ptr)tsp );
		return( nil );
	}
	bzero( (char *)&pb, sizeof( pb ));	
	pb.csCode = TCPCreate;
	pb.ioCRefNum = tsp->drefnum;
	pb.csParam.create.rcvBuff = (Ptr) buf;
	pb.csParam.create.rcvBuffLen = buflen;
	pb.csParam.create.notifyProc = (TCPNotifyProc)tcp_asynchnotify;
	pb.csParam.create.userDataPtr = (Ptr)tsp;
		
	if (( err = PBControl( (ParmBlkPtr)&pb, 0 )) != noErr || pb.ioResult != noErr ) {
		DisposePtr( (Ptr)tsp->tcps_buffer );
		DisposePtr( (Ptr)tsp );
		return( nil );
	}
	
	tsp->tcps_data = 0;
	tsp->tcps_terminated = tsp->tcps_connected = false;
	tsp->tcps_sptr = pb.tcpStream;
	
	return( tsp );
}

/*
 * connect to remote host at IP address "addr", TCP port "port"
 * return local port assigned, 0 if error
 */
tcp_port tcpconnect( tcpstream * tsp, ip_addr addr, tcp_port port ) {
	TCPiopb		pb;
	OSErr		err;

	if ( tsp->tcps_sptr == (StreamPtr)NULL ) {
		return( 0 );
	}
		
	bzero( (char *)&pb, sizeof( pb ));	
	pb.csCode = TCPActiveOpen;
	pb.ioCRefNum = tsp->drefnum;
	pb.tcpStream = tsp->tcps_sptr;
	pb.csParam.open.remoteHost = addr;
	pb.csParam.open.remotePort = port;
	pb.csParam.open.ulpTimeoutValue = 15;
	pb.csParam.open.validityFlags = timeoutValue;
		
	if (( err = PBControl( (ParmBlkPtr)&pb, 0 )) != noErr || pb.ioResult != noErr ) {
		return( 0 );
	}
	
	tsp->tcps_connected = true;
	return( pb.csParam.open.localPort );
}


/*
 * close and release a TCP stream
 * returns 0 if no error, -1 if any error occurs
 */
short tcpclose( tcpstream * tsp ) {
	TCPiopb		pb;
	short		rc;
	
	if ( tsp->tcps_sptr == (StreamPtr)NULL ) {
		return( -1 );
	}
	
#ifdef notdef
	/*
	 * if connected execute a close
	 */
	if ( tcp->tcps_connected ) {
		bzero( (char *)&pb, sizeof( pb ));
		pb.csCode = TCPClose;
		pb.ioCRefNum = tsp->drefnum;
		pb.tcpStream = sp;
		pb.csParam.close.validityFlags = 0;
		PBControl( (ParmBlkPtr)&pb, 0 );
	}
#endif /* notdef */
		
	bzero( (char *)&pb, sizeof( pb ));
	pb.csCode = TCPRelease;
	pb.ioCRefNum = tsp->drefnum;
	pb.tcpStream = tsp->tcps_sptr;
	pb.csParam.close.validityFlags = 0;
	rc = 0;
	
	if ( PBControl( (ParmBlkPtr)&pb, 0 ) != noErr || pb.ioResult != noErr ) {
		rc = -1;
	}
	
	DisposPtr( (Ptr)tsp->tcps_buffer );
	tsp->tcps_sptr = (StreamPtr)NULL;		/* mark as no longer in use */
		
	return( rc );	
}


/*
 * wait for new data to arrive
 *
 * if "timeout" is NULL, wait until data arrives or connection goes away
 * if "timeout" is a pointer to a zero'ed struct, poll
 * else wait for "timeout->tv_sec + timeout->tv_usec" seconds
 */
short tcpselect( tcpstream * tsp, struct timeval * timeout ) {
	long	ticks, endticks;

	if ( tsp->tcps_sptr == (StreamPtr)NULL ) {
		return( -1 );
	}
	
	if ( timeout != NULL ) {
		endticks = 60 * timeout->tv_sec + ( 60 * timeout->tv_usec ) / 1000000 + TickCount();
	}

	ticks = 0;	
	while (( timeout == NULL || ticks < endticks ) && tsp->tcps_data < 1 && !tsp->tcps_terminated ) {
		Delay( 1L, &ticks );
	} 
	
	if ( tsp->tcps_terminated ) {
		return( -1 );
	}
	
	return ( tsp->tcps_data < 1 ? 0 : 1 );
}


/*
 * read up to "rbuflen" bytes into "rbuf" from a connected TCP stream
 * wait up to "timeout" seconds for data (if timeout == 0, wait "forever")
 */
long
tcpread( tcpstream * tsp, byte timeout, unsigned char * rbuf,
						  unsigned short rbuflen, DialogPtr dlp ) {
	TCPiopb			pb;
	unsigned long	time, end_time;
	
	if ( tsp->tcps_sptr == (StreamPtr)NULL ) {
		return( -1 );
	}
	
	GetDateTime( &time );
	end_time = time + timeout;

	while(( timeout <= 0 || end_time > time ) && tsp->tcps_data < 1 &&
			!tsp->tcps_terminated ) {
		GetDateTime( &time );
	}
	
	if ( tsp->tcps_data < 1 ) {
		return( tsp->tcps_terminated ? -3 : -1 );
	}
		
	bzero( (char *)&pb, sizeof( pb ));	
	pb.csCode = TCPRcv;
	pb.ioCRefNum = tsp->drefnum;
	pb.tcpStream = tsp->tcps_sptr;
	pb.csParam.receive.commandTimeoutValue = timeout;
	pb.csParam.receive.rcvBuff = (char *)rbuf;	
	pb.csParam.receive.rcvBuffLen = rbuflen;
	
	if ( PBControl( (ParmBlkPtr)&pb, 0 ) != noErr || pb.ioResult != noErr ) {
		return( -1 );
	}
	
	if ( --(tsp->tcps_data) < 0 ) {
		tsp->tcps_data = 0;
	}
	
	return( pb.csParam.receive.rcvBuffLen );
#pragma unused (dlp)
}
	

/*
 * send TCP data
 * "sp" is the stream to write to
 * "wbuf" is "wbuflen" bytes of data to send
 * returns < 0 if error, number of bytes written if no error
 */
short
tcpwrite( tcpstream * tsp, unsigned char * wbuf, unsigned short wbuflen ) {
	TCPiopb		pb;
	wdsEntry	wds[ 2 ];
	OSErr 		err;
	
	if ( tsp->tcps_sptr == (StreamPtr)NULL ) {
		return( -1 );
	}
	
	wds[ 0 ].length = wbuflen;
	wds[ 0 ].ptr = (char *)wbuf;
	wds[ 1 ].length = 0;
	
	bzero( (char *)&pb, sizeof( pb ));	
	pb.csCode = TCPSend;
	pb.ioCRefNum = tsp->drefnum;
	pb.tcpStream = tsp->tcps_sptr;
	pb.csParam.send.wdsPtr = (Ptr)wds;
	pb.csParam.send.validityFlags = 0;
	
	if (( err = PBControl( (ParmBlkPtr)&pb, 0 )) != noErr || pb.ioResult != noErr ) {
		return( -1 );
	}
	
	return( wbuflen );
}


pascal void
tcp_asynchnotify( tstream, event, userdatap, term_reason, icmpmsg )
	StreamPtr			tstream;
	unsigned short		event;
	Ptr					userdatap;
	unsigned short		term_reason;	
	struct ICMPReport	*icmpmsg;
{
	tcpstream	*tsp;
	
	tsp = (tcpstream *)userdatap;
	switch( event ) {
		case TCPDataArrival:
			++(tsp->tcps_data);
			break;
		case TCPClosing:
		case TCPTerminate:
		case TCPULPTimeout:
			tsp->tcps_terminated = true;
			break;
		default:
			break;
	}
#pragma unused (tstream, term_reason, icmpmsg)
}


/*
 * bzero -- set "len" bytes starting at "p" to zero
 */
static void
bzero( p, len )
 	char			*p;
	unsigned long	len;
{
	unsigned long	i;
	
	for ( i = 0; i < len; ++i ) {
		*p++ = '\0';
	}
}


pascal void
setdoneflag( hip, donep )
 	struct hostInfo *hip;
	char			*donep;
{
	++(*donep);
#pragma unused (hip)
}


/*
 * return a hostInfo structure for "host" (return != noErr if error)
 */
short
gethostinfobyname( host, hip )
 	char				*host;
	struct hostInfo		*hip;
{
	char					done = 0;
	OSErr					err;
	long					time;
	struct dnr_struct		dnr_global = {nil, 0};

	kick_mactcp( NULL );

	if (( err = OpenResolver( &dnr_global, NULL )) != noErr )
		return( err );

	if (( err = StrToAddr( &dnr_global, host, hip, (ResultProcPtr)setdoneflag, &done ))  == cacheFault ) {
		while( !done ) {
			Delay( 2, &time );			// querying DN servers; wait for reply
		}
		err = hip->rtnCode;
	}

	CloseResolver( &dnr_global );

	if ( err != noErr || (( err == noErr ) && ( hip->addr[ 0 ] == 0 ))) {
		return( err == noErr ? noAnsErr : err );
	}
	
	return( noErr );
}

/*
 * return a hostInfo structure for "addr" (return != noErr if error)
 */
short
gethostinfobyaddr( addr, hip )
 	ip_addr				addr;
	struct hostInfo		*hip;
{
	char					done = 0;
	OSErr					err;
	long					time;
	struct dnr_struct		dnr_global = {nil, 0};
	
	kick_mactcp( NULL );

	if (( err = OpenResolver( &dnr_global, NULL )) != noErr )
		return( err );

	if (( err = AddrToName( &dnr_global, addr, hip, (ResultProcPtr)setdoneflag, &done ))  == cacheFault ) {
		while( !done ) {
			Delay( 2, &time );			// querying DN servers; wait for reply
		}
		err = hip->rtnCode;
	}

	CloseResolver( &dnr_global );

	if ( err != noErr || (( err == noErr ) && ( hip->addr[ 0 ] == 0 ))) {
		return( err == noErr ? noAnsErr : err );
	}
	
	return( noErr );
}


static OSErr kick_mactcp( drefnump )
	short	*drefnump;
{
	short				dref;
	OSErr				err;
	struct IPParamBlock	ipb;

/*
 * we make sure the MacTCP driver is open and issue a "Get My Address" call
 * so that adevs link MacPPP are initialized (MacTCP is dumb)
 */
	if (( err = OpenDriver( "\p.IPP", &dref )) != noErr ) {
		return( err );
	}

	if ( drefnump != NULL ) {
		*drefnump = dref;
	}

	bzero( (char *)&ipb, sizeof( ipb ));
	ipb.csCode = ipctlGetAddr;
	ipb.ioCRefNum = dref;

	err = PBControl( (ParmBlkPtr)&ipb, 0 );

	return( noErr );
}

