/*
 *  Copyright (c) 1990 Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  result.c - wait for an ldap result
 */

#ifndef lint 
static char copyright[] = "@(#) Copyright (c) 1990 Regents of the University of Michigan.\nAll rights reserved.\n";
#endif

#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <string.h>
#ifdef MACOS
#include <stdlib.h>
#include "macos.h"
#else /* MACOS */
#ifdef DOS
#include <time.h>
#include "msdos.h"
#ifdef PCNFS
#include <tklib.h>
#include <tk_errno.h>
#include <bios.h>
#endif /* PCNFS */
#ifdef NCSA
#include "externs.h"
#endif /* NCSA */
#else /* DOS */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef _AIX
#include <sys/select.h>
#endif /* _AIX */
#endif /* DOS */
#endif /* MACOS */
#ifdef __hpux
#define getdtablesize()	_NFILE
#endif

#ifdef NEEDPROTOS
static int ldap_select1( LDAP *ld, struct timeval *timeout );
static int ldap_abandoned( LDAP *ld, int msgid );
static int ldap_mark_abandoned( LDAP *ld, int msgid );
static int wait4msg( LDAP *ld, int msgid, int all, struct timeval *timeout,
	LDAPMessage **result );
#else
static int ldap_select1();
static int ldap_abandoned();
static int ldap_mark_abandoned();
static int wait4msg();
#endif /* NEEDPROTOS */

/*
 * ldap_result - wait for an ldap result response to a message from the
 * ldap server.  If msgid is -1, any message will be accepted, otherwise
 * ldap_result will wait for a response with msgid.  If all is 0 the
 * first message with id msgid will be accepted, otherwise, ldap_result
 * will wait for all responses with id msgid and then return a pointer to
 * the entire list of messages.  This is only useful for search responses,
 * which can be of two message types (zero or more entries, followed by an
 * ldap result).  The type of the first message received is returned.
 * When waiting, any messages that have been abandoned are discarded.
 *
 * Example:
 *	ldap_result( s, msgid, all, timeout, result )
 */

ldap_result( ld, msgid, all, timeout, result )
LDAP		*ld;
int		msgid;
int		all;
struct timeval	*timeout;
LDAPMessage	**result;
{
	LDAPMessage	*lm, *lastlm, *nextlm;

	/*
	 * First, look through the list of responses we have received on
	 * this association and see if the response we're interested in
	 * is there.  If it is, return it.  If not, call wait4msg() to
	 * wait until it arrives or timeout occurs.
	 */

	Debug( LDAP_DEBUG_TRACE, "ldap_result\n", 0, 0, 0 );

	*result = NULLMSG;
	lastlm = NULLMSG;
	for ( lm = ld->ld_responses; lm != NULLMSG; lm = nextlm ) {
		nextlm = lm->lm_next;

		if ( ldap_abandoned( ld, lm->lm_msgid ) ) {
			ldap_mark_abandoned( ld, lm->lm_msgid );

			if ( lastlm == NULLMSG ) {
				ld->ld_responses = lm->lm_next;
			} else {
				lastlm->lm_next = nextlm;
			}

			ldap_msgfree( lm );

			continue;
		}

		if ( msgid == LDAP_RES_ANY || lm->lm_msgid == msgid ) {
			LDAPMessage	*tmp;

			if ( all == 0
			    || (lm->lm_msgtype != LDAP_RES_SEARCH_RESULT
			    && lm->lm_msgtype != LDAP_RES_SEARCH_ENTRY) )
				break;

			for ( tmp = lm; tmp != NULLMSG; tmp = tmp->lm_chain ) {
				if ( tmp->lm_msgtype == LDAP_RES_SEARCH_RESULT )
					break;
			}

			if ( tmp == NULLMSG ) {
				return( wait4msg( ld, msgid, all, timeout,
				    result ) );
			}

			break;
		}
		lastlm = lm;
	}
	if ( lm == NULLMSG ) {
		return( wait4msg( ld, msgid, all, timeout, result ) );
	}

	if ( lastlm == NULLMSG ) {
		ld->ld_responses = (all == 0 && lm->lm_chain != NULLMSG
		    ? lm->lm_chain : lm->lm_next);
	} else {
		lastlm->lm_next = (all == 0 && lm->lm_chain != NULLMSG
		    ? lm->lm_chain : lm->lm_next);
	}
	if ( all == 0 )
		lm->lm_chain = NULLMSG;
	lm->lm_next = NULLMSG;

	*result = lm;
	ld->ld_errno = LDAP_SUCCESS;
	return( lm->lm_msgtype );
}

static wait4msg( ld, msgid, all, timeout, result )
LDAP		*ld;
int		msgid;
int		all;
struct timeval	*timeout;
LDAPMessage	**result;
{
	BerElement	ber;
	LDAPMessage	*new, *l, *prev, *tmp;
	int		tag, rc;
	long		id;
	unsigned long	len;
	int		foundit = 0;

	Debug( LDAP_DEBUG_TRACE, "wait4msg\n", 0, 0, 0 );

	/* return out */
	while ( 1 ) {
		/* hack attack */
		if ( ld->ld_sb.sb_ber.ber_ptr >= ld->ld_sb.sb_ber.ber_end ) {
			rc = ldap_select1( ld, timeout );
			if ( rc == -1 || rc == 0 ) {
				ld->ld_errno = (rc == -1 ? LDAP_SERVER_DOWN :
				    LDAP_TIMEOUT);
				return( rc );
			}
		}

		/* get the next message */
		if ( (tag = ber_get_next( &ld->ld_sb, &len, &ber ))
		    != LDAP_TAG_MESSAGE ) {
			ld->ld_errno = (tag == -1 ? LDAP_SERVER_DOWN :
			    LDAP_LOCAL_ERROR);
			return( -1 );
		}

		/* message id */
		if ( ber_get_int( &ber, &id ) == -1 ) {
			ld->ld_errno = LDAP_DECODING_ERROR;
			return( -1 );
		}

		/* if it's been abandoned, toss it */
		if ( ldap_abandoned( ld, (int)id ) ) {
			free( ber.ber_buf );	/* gack! */
			continue;
		}

		/* the message type */
		if ( (tag = ber_peek_tag( &ber, &len )) == -1 ) {
			ld->ld_errno = LDAP_DECODING_ERROR;
			return( -1 );
		}

		/* make a new ldap message */
		if ( (new = (LDAPMessage *) calloc( 1, sizeof(LDAPMessage) ))
		    == NULL ) {
			ld->ld_errno = LDAP_LOCAL_ERROR;
			return( -1 );
		}
		new->lm_msgid = (int)id;
		new->lm_msgtype = tag;
		new->lm_ber = ber_dup( &ber );

		if ( ld->ld_cache != NULL ) {
			add_result_to_cache( ld, new );
		}

		/* is this the one we're looking for? */
		if ( msgid == LDAP_RES_ANY || id == msgid ) {
			if ( all == 0
			    || (new->lm_msgtype != LDAP_RES_SEARCH_RESULT
			    && new->lm_msgtype != LDAP_RES_SEARCH_ENTRY) ) {
				*result = new;
				ld->ld_errno = LDAP_SUCCESS;
				return( tag );
			} else if ( new->lm_msgtype == LDAP_RES_SEARCH_RESULT) {
				foundit = 1;	/* return the chain later */
			}
		}

		/* 
		 * if not, we must add it to the list of responses.  if
		 * the msgid is already there, it must be part of an existing
		 * search response.
		 */

		prev = NULLMSG;
		for ( l = ld->ld_responses; l != NULLMSG; l = l->lm_next ) {
			if ( l->lm_msgid == new->lm_msgid )
				break;
			prev = l;
		}

		/* not part of an existing search response */
		if ( l == NULLMSG ) {
			if ( foundit ) {
				*result = new;
				ld->ld_errno = LDAP_SUCCESS;
				return( tag );
			}

			new->lm_next = ld->ld_responses;
			ld->ld_responses = new;
			continue;
		}

		/* part of a search response - add it at the end */
		for ( tmp = l; tmp->lm_chain != NULLMSG; tmp = tmp->lm_chain )
			;	/* NULL */
		tmp->lm_chain = new;

		/* return the whole chain if that's what we were looking for */
		if ( foundit ) {
			if ( prev == NULLMSG )
				ld->ld_responses = l->lm_next;
			else
				prev->lm_next = l->lm_next;
			*result = l;
			ld->ld_errno = LDAP_SUCCESS;
			return( tag );
		}
	}
	/* NOT REACHED */
}


#if !defined( MACOS ) && !defined( DOS )
static ldap_select1( ld, timeout )
	LDAP		*ld;
	struct timeval	*timeout;
{
	fd_set		readfds;
	static int	tblsize;

	if ( tblsize == 0 ) {
		tblsize = getdtablesize();
	}

	FD_ZERO( &readfds );
	FD_SET( ld->ld_sb.sb_sd, &readfds );

	return( select( tblsize, &readfds, 0, 0, timeout ) );
}
#endif /* MACOS */


#ifdef MACOS
static int
ldap_select1( ld, timeout )
	LDAP		*ld;
	struct timeval	*timeout;
{
	return( tcpselect( ld->ld_sb.sb_sd, timeout ));
}
#endif /* MACOS */

#ifdef DOS
#ifdef WINSOCK
static ldap_select1( ld, timeout )
    LDAP            *ld;
    struct timeval  *timeout;
{
    fd_set          readfds;
    int             rc;

    FD_ZERO( &readfds );
    FD_SET( ld->ld_sb.sb_sd, &readfds );

    rc = select( 1, &readfds, 0, 0, timeout );
    return( rc == SOCKET_ERROR ? -1 : rc );
}
#endif /* WINSOCK */


#ifdef PCNFS
static int
ldap_select1( ld, timeout )
    LDAP            *ld;
    struct timeval  *timeout;
{
	fd_set	readfds;
	int	res;

	FD_ZERO( &readfds );
	FD_SET( ld->ld_sb.sb_sd, &readfds );

	res = select( FD_SETSIZE, &readfds, NULL, NULL, timeout );
	if ( res == -1 && errno == EINTR) {
		/* We've been CTRL-C'ed at this point.  It'd be nice to
		   carry on but PC-NFS currently won't let us! */
		printf("\n*** CTRL-C ***\n");
		exit(-1);
	}
	return( res );
}
#endif /* PCNFS */

#ifdef NCSA
static int
ldap_select1( ld, timeout )
	LDAP		*ld;
	struct timeval	*timeout;
{
	int rc;
	clock_t	endtime;

	if ( timeout != NULL ) {
		endtime = timeout->tv_sec * CLK_TCK +
			timeout->tv_usec * CLK_TCK / 1000000 + clock();
	}

	do {
		Stask();
		rc = netqlen( ld->ld_sb.sb_sd );
	} while ( rc <= 0 && ( timeout == NULL || clock() < endtime ));

	return( rc > 0 ? 1 : 0 );
}
#endif /* NCSA */
#endif /* DOS */

ldap_msgfree( lm )
LDAPMessage	*lm;
{
	LDAPMessage	*next;
	int		type;

	Debug( LDAP_DEBUG_TRACE, "ldap_msgfree\n", 0, 0, 0 );

	for ( ; lm != NULLMSG; lm = next ) {
		next = lm->lm_chain;
		type = lm->lm_msgtype;
		ber_free( lm->lm_ber, 1 );
		free( (char *) lm );
	}

	return( type );
}

/*
 * ldap_msgdelete - delete a message.  It returns:
 *	0	if the entire message was deleted
 *	-1	if the message was not found, or only part of it was found
 */
ldap_msgdelete( ld, msgid )
LDAP	*ld;
int	msgid;
{
	LDAPMessage	*lm, *prev;

	Debug( LDAP_DEBUG_TRACE, "ldap_msgdelete\n", 0, 0, 0 );

	prev = NULLMSG;
	for ( lm = ld->ld_responses; lm != NULLMSG; lm = lm->lm_next ) {
		if ( lm->lm_msgid == msgid )
			break;
	}

	if ( lm == NULLMSG )
		return( -1 );

	if ( prev == NULLMSG )
		ld->ld_responses = lm->lm_next;
	else
		prev->lm_next = lm->lm_next;

	if ( ldap_msgfree( lm ) == LDAP_RES_SEARCH_ENTRY )
		return( -1 );

	return( 0 );
}


/*
 * return 1 if message msgid is waiting to be abandoned, 0 otherwise
 */
static int
ldap_abandoned( ld, msgid )
LDAP	*ld;
int	msgid;
{
	int	i;

	if ( ld->ld_abandoned == NULL )
		return( 0 );

	for ( i = 0; ld->ld_abandoned[i] != -1; i++ )
		if ( ld->ld_abandoned[i] == msgid )
			return( 1 );

	return( 0 );
}


static int
ldap_mark_abandoned( ld, msgid )
LDAP	*ld;
int	msgid;
{
	int	i;

	if ( ld->ld_abandoned == NULL )
		return( -1 );

	for ( i = 0; ld->ld_abandoned[i] != -1; i++ )
		if ( ld->ld_abandoned[i] == msgid )
			break;

	if ( ld->ld_abandoned[i] == -1 )
		return( -1 );

	for ( ; ld->ld_abandoned[i] != -1; i++ ) {
		ld->ld_abandoned[i] = ld->ld_abandoned[i + 1];
	}

	return( 0 );
}
