/*
 *  Copyright (c) 1990 Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  parse.c
 */

#ifndef lint 
static char copyright[] = "@(#) Copyright (c) 1990 Regents of the University of Michigan.\nAll rights reserved.\n";
#endif

#include "lber.h"
#include "ldap.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#ifdef MACOS
#include <stdlib.h>
#include "macos.h"
#endif /* MACOS */

#ifdef DOS
#include <malloc.h>
#include "msdos.h"
#endif /* DOS */

/*
 * These routines are for handling search entry responses.  A search entry
 * response looks like this:
 *
 *	SearchResponse ::= [APPLICATION 4] SEQUENCE {
 *		objectName	DistinguishedName,
 *		attributes	SEQUENCE OF SEQUENCE {
 *			type		AttributeType,
 *			values		SET OF AttributeValue
 *		}
 *	}
 *
 * Ldap_result() returns a chain of search entry responses, terminated with
 * a search result response.
 */

/* ARGSUSED */
LDAPMessage *ldap_first_entry( ld, chain )
LDAP		*ld;
LDAPMessage	*chain;
{
	return( chain == NULLMSG || chain->lm_msgtype == LDAP_RES_SEARCH_RESULT
	    ? NULLMSG : chain );
}

/* ARGSUSED */
LDAPMessage *ldap_next_entry( ld, entry )
LDAP		*ld;
LDAPMessage	*entry;
{
	if ( entry == NULLMSG || entry->lm_chain == NULLMSG
	    || entry->lm_chain->lm_msgtype == LDAP_RES_SEARCH_RESULT )
		return( NULLMSG );

	return( entry->lm_chain );
}

LDAPMessage *
ldap_delete_result_entry( list, e )
    LDAPMessage	**list;
    LDAPMessage	*e;
{
	LDAPMessage	*tmp, *prev = NULL;

	for ( tmp = *list; tmp != NULL && tmp != e; tmp = tmp->lm_chain )
		prev = tmp;

	if ( tmp == NULL )
		return( NULL );

	if ( prev == NULL )
		*list = tmp->lm_chain;
	else
		prev->lm_chain = tmp->lm_chain;
	tmp->lm_chain = NULL;

	return( tmp );
}

void
ldap_add_result_entry( list, e )
    LDAPMessage	**list;
    LDAPMessage	*e;
{
	e->lm_chain = *list;
	*list = e;
}

char *ldap_get_dn( ld, entry )
LDAP		*ld;
LDAPMessage	*entry;
{
	char		*dn;
	BerElement	tmp;

	Debug( LDAP_DEBUG_TRACE, "ldap_get_dn\n", 0, 0, 0 );

	if ( entry == NULL ) {
		ld->ld_errno = LDAP_PARAM_ERROR;
		return( NULL );
	}

	tmp = *entry->lm_ber;	/* struct copy */
	if ( ber_scanf( &tmp, "{{a", &dn ) == -1 ) {
		ld->ld_errno = LDAP_DECODING_ERROR;
		return( NULL );
	}

	return( dn );
}

char *ldap_dn2ufn( dn )
char	*dn;
{
	char	*p, *ufn, *r;
	int	state;

	Debug( LDAP_DEBUG_TRACE, "ldap_dn2ufn\n", 0, 0, 0 );

	if ( (p = strchr( dn, '=' )) == NULL )
		return( NULL );

	ufn = strdup( ++p );

#define INQUOTE		1
#define OUTQUOTE	2
	state = OUTQUOTE;
	for ( p = ufn, r = ufn; *p; p++ ) {
		switch ( *p ) {
		case '"':
			if ( state == INQUOTE )
				state = OUTQUOTE;
			else
				state = INQUOTE;
			*r++ = *p;
			break;
		case ';':
		case ',':
			if ( state == OUTQUOTE )
				*r++ = ',';
			else
				*r++ = *p;
			break;
		case '=':
			if ( state == INQUOTE )
				*r++ = *p;
			else {
				char	*rsave = r;

				*r-- = '\0';
				while ( !isspace( *r ) && *r != ';'
				    && *r != ',' && r > ufn )
					r--;
				r++;

				if ( strcmp( r, "c" )
				    && strcmp( r, "o" )
				    && strcmp( r, "ou" )
				    && strcmp( r, "cn" ) ) {
					r = rsave;
					*r++ = '=';
				}
			}
			break;
		default:
			*r++ = *p;
			break;
		}
	}
	*r = '\0';

	return( ufn );
}

char *ldap_first_attribute( ld, entry, ber )
LDAP		*ld;
LDAPMessage	*entry;
BerElement	**ber;
{
	long	len;

	Debug( LDAP_DEBUG_TRACE, "ldap_first_attribute\n", 0, 0, 0 );

	if ( (*ber = ber_alloc()) == NULLBER ) {
		ld->ld_errno = LDAP_LOCAL_ERROR;
		return( NULL );
	}

	**ber = *entry->lm_ber;

	/* 
	 * Skip past the sequence, dn, sequence of sequence, snarf the
	 * attribute type, and skip the set of values, leaving us
	 * positioned right before the next attribute type/value sequence.
	 */

	len = LDAP_MAX_ATTR_LEN;
	if ( ber_scanf( *ber, "{{x{{sx}", ld->ld_attrbuffer, &len ) == -1 ) {
		ld->ld_errno = LDAP_DECODING_ERROR;
		ber_free( *ber, 0 );
		return( NULL );
	}

	return( ld->ld_attrbuffer );
}

/* ARGSUSED */
char *ldap_next_attribute( ld, entry, ber )
LDAP		*ld;
LDAPMessage	*entry;
BerElement	*ber;
{
	long	len;

	Debug( LDAP_DEBUG_TRACE, "ldap_next_attribute\n", 0, 0, 0 );

	/* skip sequence, snarf attribute type, skip values */
	len = LDAP_MAX_ATTR_LEN;
	if ( ber_scanf( ber, "{sx}", ld->ld_attrbuffer, &len ) == -1 ) {
		ld->ld_errno = LDAP_DECODING_ERROR;
		ber_free( ber, 0 );
		return( NULL );
	}

	return( ld->ld_attrbuffer );
}

char **ldap_get_values( ld, entry, target )
LDAP		*ld;
LDAPMessage	*entry;
char		*target;
{
	BerElement	ber;
	char		attr[LDAP_MAX_ATTR_LEN];
	int		found = 0;
	long		len;
	char		**vals;

	Debug( LDAP_DEBUG_TRACE, "ldap_get_values\n", 0, 0, 0 );

	ber = *entry->lm_ber;

	/* skip sequence, dn, sequence of, and snag the first attr */
	len = sizeof(attr);
	if ( ber_scanf( &ber, "{{x{{s", attr, &len ) == -1 ) {
		ld->ld_errno = LDAP_DECODING_ERROR;
		return( NULL );
	}

	if ( strcasecmp( target, attr ) == 0 )
		found = 1;

	/* break out on success, return out on error */
	while ( ! found ) {
		len = sizeof(attr);
		if ( ber_scanf( &ber, "x}{s", attr, &len ) == -1 ) {
			ld->ld_errno = LDAP_DECODING_ERROR;
			return( NULL );
		}

		if ( strcasecmp( target, attr ) == 0 )
			break;
	}

	/* 
	 * if we get this far, we've found the attribute and are sitting
	 * just before the set of values.
	 */

	if ( ber_scanf( &ber, "[v]", &vals ) == -1 ) {
		ld->ld_errno = LDAP_DECODING_ERROR;
		return( NULL );
	}

	return( vals );
}

struct berval **ldap_get_values_len( ld, entry, target )
LDAP		*ld;
LDAPMessage	*entry;
char		*target;
{
	BerElement	ber;
	char		attr[LDAP_MAX_ATTR_LEN];
	int		found = 0;
	long		len;
	struct berval	**vals;

	Debug( LDAP_DEBUG_TRACE, "ldap_get_values_len\n", 0, 0, 0 );

	ber = *entry->lm_ber;

	/* skip sequence, dn, sequence of, and snag the first attr */
	len = sizeof(attr);
	if ( ber_scanf( &ber, "{{x{{s", attr, &len ) == -1 ) {
		ld->ld_errno = LDAP_DECODING_ERROR;
		return( NULL );
	}

	if ( strcasecmp( target, attr ) == 0 )
		found = 1;

	/* break out on success, return out on error */
	while ( ! found ) {
		len = sizeof(attr);
		if ( ber_scanf( &ber, "x}{s", attr, &len ) == -1 ) {
			ld->ld_errno = LDAP_DECODING_ERROR;
			return( NULL );
		}

		if ( strcasecmp( target, attr ) == 0 )
			break;
	}

	/* 
	 * if we get this far, we've found the attribute and are sitting
	 * just before the set of values.
	 */

	if ( ber_scanf( &ber, "[V]", &vals ) == -1 ) {
		ld->ld_errno = LDAP_DECODING_ERROR;
		return( NULL );
	}

	return( vals );
}

/* ARGSUSED */
ldap_count_entries( ld, chain )
LDAP		*ld;
LDAPMessage	*chain;
{
	int	i;

	for ( i = 0; chain != NULL && chain->lm_msgtype
	    != LDAP_RES_SEARCH_RESULT; chain = chain->lm_chain )
		i++;

	return( i );
}

ldap_count_values( vals )
    char	**vals;
{
	int	i;

	if ( vals == NULL )
		return( 0 );

	for ( i = 0; vals[i] != NULL; i++ )
		;	/* NULL */

	return( i );
}

void
ldap_value_free( vals )
char	**vals;
{
	int	i;

	if ( vals == NULL )
		return;
	for ( i = 0; vals[i] != NULL; i++ )
		free( vals[i] );
	free( (char *) vals );
}

char **ldap_explode_dn( dn, notypes )
char	*dn;
int	notypes;
{
	char	savechar, *p, *q, *rdnstart, **rdns = NULL;
	int	state, count = 0, endquote;

	Debug( LDAP_DEBUG_TRACE, "ldap_explode_dn\n", 0, 0, 0 );
	
	rdnstart = dn;
	p = dn-1;
	state = OUTQUOTE;

	do {
		++p;
		savechar = *p;
		switch ( *p ) {
		case '"':
			if ( state == INQUOTE )
				state = OUTQUOTE;
			else
				state = INQUOTE;
			break;
		case ';':
		case ',':
		case '\0':
			if ( state == OUTQUOTE ) {
				++count;
				if ( rdns == NULL ) {
					if (( rdns = (char **)malloc( 8
						 * sizeof( char *))) == NULL )
						return( NULL );
				} else if ( count >= 8 ) {
					if (( rdns = (char **)realloc( rdns,
						(count+1) * sizeof( char *)))
						== NULL )
						return( NULL );
				}
				rdns[ count ] = NULL;
				*p = '\0';
				endquote = 0;
				if ( notypes ) {
					if (( q = strchr( rdnstart, '=' )) != NULL )
						rdnstart = ++q;
					if ( *rdnstart == '"' )
						++rdnstart;
					
					if ( *(p-1) == '"' ) {
						*(p-1) = '\0';
						endquote = 1;
					}
				}
				rdns[ count-1 ] = strdup( rdnstart );
				*p = savechar;
				if ( endquote )
					*(p-1) = '"';

				rdnstart = *p ? p + 1 : p;
				while ( isspace( *rdnstart ))
					++rdnstart;
			}
			break;
		}
	} while ( *p );

	return( rdns );
}

#if defined( ultrix ) || defined( NeXT )

char *strdup( s )
char	*s;
{
	char	*p;

	if ( (p = (char *) malloc( strlen( s ) + 1 )) == NULL )
		return( NULL );

	strcpy( p, s );

	return( p );
}

#endif /* ultrix */
