/*
 *  Copyright (c) 1993 Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  kbind.c
 */

#ifndef lint 
static char copyright[] = "@(#) Copyright (c) 1993 Regents of the University of Michigan.\nAll rights reserved.\n";
#endif

#include "lber.h"
#include "ldap.h"

#ifdef KERBEROS

#include <stdio.h>
#include <string.h>
#ifndef MACOS
#include <krb.h>
#include <sys/types.h>
#include <sys/time.h>
#endif /* !MACOS */


/*
 * ldap_kerberos_bind1 - initiate a bind to the ldap server using
 * kerberos authentication.  The dn is supplied.  It is assumed the user
 * already has a valid ticket granting ticket, so the passwd parameter is
 * ignored.  A future version may handle getting the tgt.  The msgid of the
 * request is returned on success (suitable for passing to ldap_result()),
 * -1 is returned if there's trouble.
 *
 * Example:
 *	ldap_kerberos_bind1( ld, "cn=manager, o=university of michigan, c=us",
 *	    "secret" )
 */

ldap_kerberos_bind1( ld, dn, passwd )
LDAP	*ld;
char	*dn;
char	*passwd;
{
	BerElement	*ber;
	char		*cred;
	int		credlen;
	char		*get_kerberosv4_credentials();

	/*
	 * The bind request looks like this:
	 *	BindRequest ::= SEQUENCE {
	 *		version		INTEGER,
	 *		name		DistinguishedName,
	 *		authentication	CHOICE {
	 *			krbv42ldap	[1] OCTET STRING
	 *			krbv42dsa	[2] OCTET STRING
	 *		}
	 *	}
	 * all wrapped up in an LDAPMessage sequence.
	 */

	Debug( LDAP_DEBUG_TRACE, "ldap_kerberos_bind1\n", 0, 0, 0 );

	if ( dn == NULL )
		dn = "";
	if ( passwd == NULL )
		passwd = "";

	if ( (cred = get_kerberosv4_credentials( ld, dn, passwd, "ldapserver",
	    &credlen )) == NULL ) {
		return( -1 );	/* ld_errno should already be set */
	}

	/* create a message to send */
	if ( (ber = ber_alloc()) == NULLBER ) {
		ld->ld_errno = LDAP_LOCAL_ERROR;
		return( -1 );
	}

	/* fill it in */
	if ( ber_printf( ber, "{it{{ist{o}}}}", ++ld->ld_msgid, LDAP_REQ_BIND,
	    ld->ld_version, dn, LDAP_AUTH_KRBV41, cred, credlen ) == -1 ) {
		ber_free( ber, 1 );
		ld->ld_errno = LDAP_ENCODING_ERROR;
		return( -1 );
	}

	/* send the message */
	if ( ber_flush( &ld->ld_sb, ber, 1 ) == -1 ) {
		ld->ld_errno = LDAP_SERVER_DOWN;
		return( -1 );
	}

	if ( ld->ld_cache != NULL ) {
		ldap_flush_cache( ld );
	}

	return( ld->ld_msgid );
}

ldap_kerberos_bind1_s( ld, dn, passwd )
LDAP	*ld;
char	*dn;
char	*passwd;
{
	int		msgid;
	LDAPMessage	*res;

	Debug( LDAP_DEBUG_TRACE, "ldap_kerberos_bind1_s\n", 0, 0, 0 );

	/* initiate the bind */
	if ( (msgid = ldap_kerberos_bind1( ld, dn, passwd )) == -1 )
		return( ld->ld_errno );

	/* wait for a result */
	if ( ldap_result( ld, ld->ld_msgid, 1, (struct timeval *) 0, &res )
	    == -1 ) {
		return( ld->ld_errno );	/* ldap_result sets ld_errno */
	}

	return( ldap_result2error( ld, res, 1 ) );
}

/*
 * ldap_kerberos_bind2 - initiate a bind to the X.500 server using
 * kerberos authentication.  The dn is supplied.  It is assumed the user
 * already has a valid ticket granting ticket, so the passwd parameter is
 * ignored.  A future version may handle getting the tgt.  The msgid of the
 * request is returned on success (suitable for passing to ldap_result()),
 * -1 is returned if there's trouble.
 *
 * Example:
 *	ldap_kerberos_bind2( ld, "cn=manager, o=university of michigan, c=us",
 *	    "secret" )
 */

ldap_kerberos_bind2( ld, dn, passwd )
LDAP	*ld;
char	*dn;
char	*passwd;
{
	BerElement	*ber;
	char		*cred;
	int		credlen;
	char		*get_kerberosv4_credentials();

	Debug( LDAP_DEBUG_TRACE, "ldap_kerberos_bind2\n", 0, 0, 0 );

	if ( dn == NULL )
		dn = "";
	if ( passwd == NULL )
		passwd = "";

	if ( (cred = get_kerberosv4_credentials( ld, dn, passwd, "x500dsa",
	    &credlen )) == NULL ) {
		return( -1 );	/* ld_errno should already be set */
	}

	/* create a message to send */
	if ( (ber = ber_alloc()) == NULLBER ) {
		ld->ld_errno = LDAP_LOCAL_ERROR;
		return( -1 );
	}

	/* fill it in */
	if ( ber_printf( ber, "{it{{ist{o}}}}", ++ld->ld_msgid, LDAP_REQ_BIND,
	    ld->ld_version, dn, LDAP_AUTH_KRBV42, cred, credlen ) == -1 ) {
		ber_free( ber, 1 );
		ld->ld_errno = LDAP_ENCODING_ERROR;
		return( -1 );
	}

	/* send the message */
	if ( ber_flush( &ld->ld_sb, ber, 1 ) == -1 ) {
		ld->ld_errno = LDAP_SERVER_DOWN;
		return( -1 );
	}

	return( ld->ld_msgid );
}

/* synchronous bind to DSA using kerberos */
ldap_kerberos_bind2_s( ld, dn, passwd )
LDAP	*ld;
char	*dn;
char	*passwd;
{
	int		msgid;
	LDAPMessage	*res;

	Debug( LDAP_DEBUG_TRACE, "ldap_kerberos_bind2_s\n", 0, 0, 0 );

	/* initiate the bind */
	if ( (msgid = ldap_kerberos_bind2( ld, dn, passwd )) == -1 )
		return( ld->ld_errno );

	/* wait for a result */
	if ( ldap_result( ld, ld->ld_msgid, 1, (struct timeval *) 0, &res )
	    == -1 ) {
		return( ld->ld_errno );	/* ldap_result sets ld_errno */
	}

	return( ldap_result2error( ld, res, 1 ) );
}

/* synchronous bind to ldap and DSA using kerberos */
ldap_kerberos_bind_s( ld, dn, passwd )
LDAP	*ld;
char	*dn;
char	*passwd;
{
	int	err;

	Debug( LDAP_DEBUG_TRACE, "ldap_kerberos_bind_s\n", 0, 0, 0 );

	if ( (err = ldap_kerberos_bind1_s( ld, dn, passwd )) != LDAP_SUCCESS )
		return( err );

	return( ldap_kerberos_bind2_s( ld, dn, passwd ) );
}


#ifndef AUTHMAN
/*
 * get_kerberosv4_credentials - obtain kerberos v4 credentials for ldap.
 * The dn and password of the entry to which to bind are supplied.  Right
 * now, the passwd is ignored as this routine assumes the user already has
 * a valid tgt.  A future version may get the tgt using the passwd.
 */

/* ARGSUSED */
char *get_kerberosv4_credentials( ld, who, passwd, service, len )
LDAP	*ld;
char	*who;
char	*passwd;
char	*service;
int	*len;
{
	static KTEXT_ST	k;
	KTEXT		ktxt = &k;
	int		err;
	char		realm[REALM_SZ];

	Debug( LDAP_DEBUG_TRACE, "get_kerberosv4_credentials\n", 0, 0, 0 );

	if ( (err = krb_get_tf_realm( tkt_string(), realm )) != KSUCCESS ) {
#ifndef NO_USERINTERFACE
		fprintf( stderr, "krb_get_tf_realm failed (%s)\n",
		    krb_err_txt[err] );
#endif /* NO_USERINTERFACE */
		ld->ld_errno = LDAP_INVALID_CREDENTIALS;
		return( NULL );
	}

	if ( (err = krb_mk_req( ktxt, service, ld->ld_host, realm, 0 ))
	    != KSUCCESS ) {
#ifndef NO_USERINTERFACE
		fprintf( stderr, "krb_mk_req failed (%s)\n", krb_err_txt[err] );
#endif /* NO_USERINTERFACE */
		ld->ld_errno = LDAP_INVALID_CREDENTIALS;
		return( NULL );
	}

	*len = ktxt->length;
	return( (char *) ktxt->dat );
}

#endif /* !AUTHMAN */
#endif /* KERBEROS */
