/* decode.c - ber input decoding routines */
/*
 * Copyright (c) 1990 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

#include <stdio.h>
#ifdef MACOS
#include <stdlib.h>
#include <stdarg.h>
#include "macos.h"
#else /* MACOS */
#ifdef NeXT
#include <stdlib.h>
#else /* NeXT */
#include <malloc.h>
#endif /* NeXT */
#include <varargs.h>
#include <sys/types.h>
#include <netinet/in.h>
#ifdef PCNFS
#include <tklib.h>
#endif /* PCNFS */
#endif /* MACOS */

#ifdef DOS
#include "msdos.h"
#endif /* DOS */

#include <string.h>
#include "lber.h"

#ifdef LDAP_DEBUG
int	lber_debug;
#endif

#ifdef NEEDPROTOS
static int ber_getnint( BerElement *ber, long *num, int len );
#endif /* NEEDPROTOS */

ber_skip_tag( ber, len )
BerElement	*ber;
unsigned long	*len;
{
	unsigned char	tag;
	unsigned char	lc;
	int		noctets, diff;
	unsigned long	netlen;

	/*
	 * Any ber element looks like this: tag length contents.
	 * Assuming everything's ok, we return the tag byte (we
	 * can assume a single byte), and return the length in len.
	 *
	 * Assumptions:
	 *	1) small tags (less than 128)
	 *	2) definite lengths
	 *	3) primitive encodings used whenever possible
	 */

	/*
	 * First, we read the tag.
	 */

	if ( ber_read( ber, (char *) &tag, 1 ) != 1 )
		return( -1 );

	/*
	 * Next, read the length.  The first byte contains the length of
	 * the length.  If bit 8 is set, the length is the long form,
	 * otherwise it's the short form.  We don't allow a length that's
	 * greater than what we can hold in an unsigned long.
	 */

	*len = netlen = 0;
	if ( ber_read( ber, (char *) &lc, 1 ) != 1 )
		return( -1 );
	if ( lc & 0x80 ) {
		noctets = (lc & 0x7f);
		if ( noctets > sizeof(unsigned long) )
			return( -1 );
		diff = sizeof(unsigned long) - noctets;
		if ( ber_read( ber, (char *) &netlen + diff, noctets )
		    != noctets )
			return( -1 );
		*len = ntohl( netlen );
	} else {
		*len = lc;
	}

	return( tag );
}

ber_peek_tag( ber, len )
BerElement	*ber;
unsigned long	*len;
{
	char	*save;
	int	tag;

	save = ber->ber_ptr;
	tag = ber_skip_tag( ber, len );
	ber->ber_ptr = save;

	return( tag );
}

static ber_getnint( ber, num, len )
BerElement	*ber;
long		*num;
int		len;
{
	int	diff, sign, i;
	long	netnum;

	/*
	 * The tag and length have already been stripped off.  We should
	 * be sitting right before len bytes of 2's complement integer,
	 * ready to be read straight into an int.  We may have to sign
	 * extend after we read it in.
	 */

	if ( len > sizeof(long) )
		return( -1 );

	netnum = 0;
	diff = sizeof(long) - len;
	/* read into the low-order bytes of netnum */
	if ( ber_read( ber, ((char *) &netnum) + diff, len ) != len )
		return( -1 );

	/* sign extend if necessary */
	sign = ((0x80 << ((len - 1) * 8)) & netnum);
	if ( sign && len < sizeof(long) ) {
		for ( i = sizeof(long) - 1; i > len - 1; i-- ) {
			netnum |= (0xffL << (i * 8));
		}
	}
	*num = ntohl( netnum );

	return( len );
}

ber_get_int( ber, num )
BerElement	*ber;
long		*num;
{
	int		tag;
	unsigned long	len;

	if ( (tag = ber_skip_tag( ber, &len )) == -1 )
		return( -1 );

	if ( ber_getnint( ber, num, (int)len ) != len )
		return( -1 );
	else
		return( tag );
}

ber_get_stringb( ber, buf, len )
BerElement	*ber;
char		*buf;
unsigned long	*len;
{
	unsigned long	datalen;
	int		tag;

	if ( (tag = ber_skip_tag( ber, &datalen )) == -1 )
		return( -1 );
	if ( datalen > (*len - 1) )
		return( -1 );
	*len = datalen;

	if ( ber_read( ber, buf, datalen ) != datalen )
		return( -1 );

	buf[datalen] = '\0';

	return( tag );
}

ber_get_stringa( ber, buf )
BerElement	*ber;
char		**buf;
{
	unsigned long	datalen;
	int		tag;

	if ( (tag = ber_skip_tag( ber, &datalen )) == -1 )
		return( -1 );

	if ( (*buf = (char *) malloc( (size_t)datalen + 1 )) == NULL )
		return( -1 );

	if ( ber_read( ber, *buf, datalen ) != datalen )
		return( -1 );
	(*buf)[datalen] = '\0';

	return( tag );
}

ber_get_stringal( ber, bv )
BerElement	*ber;
struct berval	**bv;
{
	int		tag;
	unsigned long	len;

	if ( (*bv = (struct berval *) malloc( sizeof(struct berval) )) == NULL )
		return( -1 );

	if ( (tag = ber_skip_tag( ber, &len )) == -1 )
		return( -1 );

	if ( ((*bv)->bv_val = (char *) malloc( (size_t)len + 1 )) == NULL )
		return( -1 );

	if ( ber_read( ber, (*bv)->bv_val, len ) != len )
		return( -1 );
	((*bv)->bv_val)[len] = '\0';
	(*bv)->bv_len = len;

	return( tag );
}

ber_get_bitstringa( ber, buf, blen )
BerElement	*ber;
char		**buf;
unsigned long	*blen;
{
	unsigned long	datalen;
	int		tag;
	unsigned char	unusedbits;

	if ( (tag = ber_skip_tag( ber, &datalen )) == -1 )
		return( -1 );
	--datalen;

	if ( (*buf = (char *) malloc( (size_t)datalen )) == NULL )
		return( -1 );

	if ( ber_read( ber, (char *)&unusedbits, 1 ) != 1 )
		return( -1 );

	if ( ber_read( ber, *buf, datalen ) != datalen )
		return( -1 );

	*blen = datalen * 8 - unusedbits;
	return( tag );
}

ber_get_null( ber )
BerElement	*ber;
{
	int		tag;
	unsigned long	len;

	if ( (tag = ber_skip_tag( ber, &len )) == -1 )
		return( -1 );

	if ( len != 0 )
		return( -1 );

	return( tag );
}

ber_get_boolean( ber, bool )
BerElement	*ber;
int		*bool;
{
	long	longbool;
	int	rc;

	rc = ber_get_int( ber, &longbool );
	*bool = longbool;

	return( rc );
}

ber_first_element( ber, len, last )
BerElement	*ber;
unsigned long	*len;
char		**last;
{
	/* skip the sequence header, use the len to mark where to stop */
	if ( ber_skip_tag( ber, len ) == -1 ) {
		return( -1 );
	}

	*last = ber->ber_ptr + *len;

	if ( *last == ber->ber_ptr ) {
		return( -1 );
	}

	return( ber_peek_tag( ber, len ) );
}

ber_next_element( ber, len, last )
BerElement	*ber;
unsigned long	*len;
char		*last;
{
	if ( ber->ber_ptr == last ) {
		return( -1 );
	}

	return( ber_peek_tag( ber, len ) );
}

/* VARARGS */
ber_scanf(
#ifdef MACOS
	BerElement *ber, char *fmt, ... )
#else /* MACOS */
	va_alist )
va_dcl
#endif /* MACOS */
{
	va_list		ap;
#ifndef MACOS
	BerElement	*ber;
	char		*fmt;
#endif /* MACOS */
	char		*last;
	char		*s, **ss, ***sss;
	struct berval 	***bv;
	int		*i, j;
	long		*l;
	int		rc, tag;
	unsigned long	len;

#ifdef MACOS
	va_start( ap, fmt );
#else /* MACOS */
	va_start( ap );
	ber = va_arg( ap, BerElement * );
	fmt = va_arg( ap, char * );
#endif /* MACOS */

#ifdef LDAP_DEBUG
	if ( lber_debug & 64 ) {
		fprintf( stderr, "ber_scanf fmt (%s) ber:\n", fmt );
		ber_dump( ber, 1 );
	}
#endif

	for ( rc = 0; *fmt && rc != -1; fmt++ ) {
		switch ( *fmt ) {
		case 'a':	/* octet string - allocate storage as needed */
			ss = va_arg( ap, char ** );
			rc = ber_get_stringa( ber, ss );
			break;

		case 'b':	/* boolean */
			i = va_arg( ap, int * );
			rc = ber_get_boolean( ber, i );
			break;

		case 'e':	/* enumerated */
		case 'i':	/* int */
			l = va_arg( ap, long * );
			rc = ber_get_int( ber, l );
			break;

		case 'l':	/* length of next item */
			l = va_arg( ap, long * );
			rc = ber_peek_tag( ber, (unsigned long *)l );
			break;

		case 'n':	/* null */
			rc = ber_get_null( ber );
			break;

		case 's':	/* octet string - in a buffer */
			s = va_arg( ap, char * );
			l = va_arg( ap, long * );
			rc = ber_get_stringb( ber, s, (unsigned long *)l );
			break;

		case 'B':	/* bit string - allocate storage as needed */
			ss = va_arg( ap, char ** );
			l = va_arg( ap, long * ); /* for length, in bits */
			rc = ber_get_bitstringa( ber, ss, (unsigned long *)l );
			break;

		case 't':	/* tag of next item */
			i = va_arg( ap, int * );
			*i = rc = ber_peek_tag( ber, &len );
			break;

		case 'v':	/* sequence of strings */
			sss = va_arg( ap, char *** );
			*sss = NULL;
			j = 0;
			for ( tag = ber_first_element( ber, &len, &last );
			    tag != -1 && rc != -1;
			    tag = ber_next_element( ber, &len, last ) ) {
				if ( *sss == NULL ) {
					*sss = (char **) malloc(
					    2 * sizeof(char *) );
				} else {
					*sss = (char **) realloc( *sss,
					    (j + 2) * sizeof(char *) );
				}
				rc = ber_get_stringa( ber, &((*sss)[j]) );
				j++;
			}
			if ( j > 0 )
				(*sss)[j] = NULL;
			break;

		case 'V':	/* sequence of strings + lengths */
			bv = va_arg( ap, struct berval *** );
			*bv = NULL;
			j = 0;
			for ( tag = ber_first_element( ber, &len, &last );
			    tag != -1 && rc != -1;
			    tag = ber_next_element( ber, &len, last ) ) {
				if ( *bv == NULL ) {
					*bv = (struct berval **) malloc(
					    2 * sizeof(struct berval *) );
				} else {
					*bv = (struct berval **) realloc( *bv,
					    (j + 2) * sizeof(struct berval *) );
				}
				rc = ber_get_stringal( ber, &((*bv)[j]) );
				j++;
			}
			if ( j > 0 )
				(*bv)[j] = NULL;
			break;

		case 'x':	/* skip the next element - whatever it is */
			if ( (rc = ber_skip_tag( ber, &len )) == -1 )
				break;
			ber->ber_ptr += len;
			break;

		case '{':	/* begin sequence */
		case '[':	/* begin set */
			if ( *(fmt + 1) != 'v' && *(fmt + 1) != 'V' )
				rc = ber_skip_tag( ber, &len );
			break;

		case '}':	/* end sequence */
		case ']':	/* end set */
			break;

		default:
#ifndef NO_USERINTERFACE
			fprintf( stderr, "unknown fmt %c\n", *fmt );
#endif /* NO_USERINTERFACE */
			rc = -1;
			break;
		}
	}

	va_end( ap );

	return( rc );
}

ber_bvfree( bv )
struct berval	*bv;
{
	if ( bv->bv_val != NULL )
		free( bv->bv_val );
	free( (char *) bv );
}

ber_bvecfree( bv )
struct berval	**bv;
{
	int	i;

	for ( i = 0; bv[i] != NULL; i++ )
		ber_bvfree( bv[i] );
	free( (char *) bv );
}
