/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*****************************************************************************
*
* File:		iso_quad.c
*
* Description:	contains the element definition routines for isoparametric
*		plane stress / plane strain elements with only four nodes
*		(faster, simpler, etc.)
*
******************************************************************************/

# include <stdio.h>
# include <math.h>
# include "allocate.h"
# include "element.h"
# include "misc.h"

# define PLANESTRESS 1
# define PLANESTRAIN 2

# define FORCE	 1
# define NOFORCE 0

unsigned LocalQuadShapeFunctions   ( );
Vector   GlobalQuadShapeFunctions  ( );
Matrix   IsoQuadLocalB ( );
Vector	 IsoQuadEquivNodalForces ( );
int      QuadElementStiffness ( );
int  	 QuadElementStress    ( );

struct definition quad_PlaneStrainDefinition = {
   quad_PlaneStrainType, Planar, 4, 4, 2, {0, 1, 2, 0, 0, 0, 0}
};

struct definition quad_PlaneStressDefinition = {
   quad_PlaneStressType, Planar, 4, 4, 2, {0, 1, 2, 0, 0, 0, 0}
};

int quad_PlaneStrainEltStiffness (element)
   Element	element;
{
   return QuadElementStiffness (element, PLANESTRAIN);
}

int quad_PlaneStrainEltStress (element)
   Element	element;
{
   return QuadElementStress (element, PLANESTRAIN);
}

int quad_PlaneStressEltStress (element)
   Element	element;
{
   return QuadElementStress (element, PLANESTRESS);
}

int quad_PlaneStressEltStiffness (element)
   Element	element;
{
   return QuadElementStiffness (element, PLANESTRESS);
}

int QuadElementStiffness (element, type)
   Element	element;
   unsigned	type;
{
   unsigned		numnodes;
   unsigned		i,j;
   int			ninteg;
   Matrix		B;
   Matrix		D;
   Vector		jac;
   Vector		equiv;
   int			count;
   static Vector	weights;
   static Matrix	tempK;
   static Matrix	N, dNdxi, dNde,
                        dNdx, dNdy = NullMatrix;
   static Matrix	Bt, temp;

   if (dNdy == NullMatrix) {
  
      N     = CreateMatrix (4,4);
      dNdxi = CreateMatrix (4,4);
      dNde  = CreateMatrix (4,4);
      dNdx  = CreateMatrix (4,4);
      dNdy  = CreateMatrix (4,4);
      weights = CreateVector (4);
      tempK = CreateMatrix (8,8);
      Bt    = CreateMatrix (8,3);
      temp  = CreateMatrix (8,3);
     
      if (N == NullMatrix || dNdxi == NullMatrix || dNde == NullMatrix ||
          dNdx == NullMatrix || dNdy == NullMatrix || 
          weights == NullMatrix || tempK == NullMatrix)
  
          Fatal ("allocation error in element %d stiffness creation",element -> number);
   }

   if (element -> material -> E == 0) {
      error ("isoparametric element %d has 0.0 for Young's modulus (E)",element -> number);
      return 1;
   }
   if (element -> material -> nu == 0) {
      error ("isoparametric element %d has 0.0 for Poisson's ratio (nu)",element -> number);
      return 1;
   }
   if (element -> material -> t == 0) {
      error ("isoparametric element %d has 0.0 for thickness (t)",element -> number);
      return 1;
   }
  
   numnodes = LocalQuadShapeFunctions (element, 
                                       N, dNdxi, dNde, weights, NOFORCE);  

   ninteg = 4;

   jac = GlobalQuadShapeFunctions (element,dNdxi,dNde,dNdx,dNdy,
                                   ninteg,numnodes);

   if (type == PLANESTRESS)
      D = PlaneStressD (element);
   else if (type == PLANESTRAIN)
      D = PlaneStrainD (element);

   if (D == NullMatrix)
      return 1;
   
   for (i = 1 ; i <= ninteg ; i++) {
      if (jac -> vdata[i] <= 0.0) {
         error ("det |J| for elt %d is <= 0, check elt distortion",element -> number);
         return 1;
      }
   } 

   element -> K = CreateMatrix (8,8);
   if (element -> K == NullMatrix)
      Fatal ("allocation error creating element %d stiffness matrix",element -> number);

   if (numnodes == 3) {
      MatrixRows (element -> K) = 6;
      MatrixCols (element -> K) = 6;
   }

   ZeroMatrix (element -> K);

	/*
	 * set-up so that multiplications work right
	 */

   MatrixRows (tempK) = 2*numnodes;
   MatrixCols (tempK) = 2*numnodes;

   for (i = 1 ; i <= ninteg ; i++) {
      B = IsoQuadLocalB (element, numnodes, dNdx, dNdy, i);
      if (B == NullMatrix)
         return 1;

      MatrixRows (Bt) = MatrixRows (temp) = MatrixCols (B);
      Bt = TransposeMatrix (Bt, B);
      if (Bt == NullMatrix) {
         error ("could not transpose a strain-displ for element %d",element -> number);
         return 1;
      }

      temp = MultiplyMatrices (temp, Bt, D);
      tempK = MultiplyMatrices (tempK, temp, B);
      ScaleMatrix (tempK, tempK, weights -> vdata [i]*jac -> vdata [i], 0.0);
      AddMatrices (element -> K, element -> K, tempK);
   }
   ScaleMatrix (element -> K, element -> K, element -> material -> t, 0.0);

	/*
	 * clean out the 7 & 8th rows and columns if this was a triangular
	 * element so nothing extra gets assembled into the global
	 * stiffness routines
	 */

   if (numnodes == 3) {
      for (i = 1 ; i <= 8 ; i++) 
         for (j = 7 ; j <= 8 ; j++) 
            MatrixData (element -> K) [i][j] = 0.0;

      for (i = 7 ; i <= 8 ; i++) 
         for (j = 1 ; j <= 6 ; j++) 
            MatrixData (element -> K) [i][j] = 0.0;
   }

   if (element -> numdistributed > 0) {
      equiv = IsoQuadEquivNodalForces (element, &count);
      if (equiv == NullMatrix)
         return count;

       for (i = 1; i <= numnodes ; i++) {
          element -> node[i] -> eq_force[1] += VectorData (equiv) [2*i - 1];
          element -> node[i] -> eq_force[2] += VectorData (equiv) [2*i];
       }
   }

   return 0;
} 

int QuadElementStress (element, type)
   Element	element;
   unsigned	 type;
{
   static Vector	stress = NullMatrix,
			d;
   static Matrix	temp;
   static Vector	weights;
   static Matrix	N, dNdxi, dNde,
                        dNdx, dNdy = NullMatrix;
   unsigned		numnodes;
   int			ninteg;
   Matrix		D,
			B;
   double		diameter;
   double		sigma1,
			sigma_x,
			sigma_y,
			tau_xy;
   Vector		jac;
   unsigned		i,j;
   double		x,y;

   if (dNdy == NullMatrix) {
  
      N     = CreateMatrix (4,4);
      dNdxi = CreateMatrix (4,4);
      dNde  = CreateMatrix (4,4);
      dNdx  = CreateMatrix (4,4);
      dNdy  = CreateMatrix (4,4);
      weights = CreateVector (4);
     
      if (N == NullMatrix || dNdxi == NullMatrix || dNde == NullMatrix ||
          dNdx == NullMatrix || dNdy == NullMatrix)
  
          Fatal ("allocation error in element %d stiffness creation",element -> number);
   }
   
   if (stress == NullMatrix) {
      stress = CreateVector (3);
      d = CreateVector (8);
      temp = CreateMatrix (3,8);

      if (stress == NullMatrix || d == NullMatrix || temp == NullMatrix)
         Fatal ("allocation error in element %d stresses", element -> number);
   }

   ninteg = 4;

   if (type == PLANESTRAIN)
      D = PlaneStrainD (element);
   else if (type == PLANESTRESS)
      D = PlaneStressD (element);

   if (D == NullMatrix)
      return 1;

   if (element -> number == 1)
      numnodes = LocalQuadShapeFunctions (element, N, 
                                          dNdxi, dNde, weights, FORCE);  
   else
      numnodes = LocalQuadShapeFunctions (element, N, 
                                          dNdxi, dNde, weights, NOFORCE);  

   ninteg = 4;
   
   jac = GlobalQuadShapeFunctions (element,dNdxi,dNde,dNdx,dNdy,
                                   ninteg,numnodes);

   for (i = 1 ; i <= numnodes ; i++) {
      VectorData (d) [2*i - 1] = element -> node[i] -> dx[1];
      VectorData (d) [2*i] = element -> node[i] -> dx[2];
   }

   VectorSize (d) = numnodes*2;
   MatrixCols (temp) = numnodes*2;

   element -> stress = Allocate (double, ninteg*6);

   if (element -> stress == NULL)
      Fatal ("allocation error creating element %d stress vector",element -> number);
   
   UnitOffset (element -> stress);
   element -> numstresses = ninteg*6;

   for (i = 1 ; i <= ninteg ; i++) {
      B = IsoQuadLocalB (element, numnodes, dNdx, dNdy, i);
      if (B == NullMatrix)
         return 1;

      x = y = 0.0;
      for (j = 1 ; j <= numnodes ; j++) {
         x += N -> mdata[j][i]*element -> node[j] -> x;
         y += N -> mdata[j][i]*element -> node[j] -> y;
      }
   
      temp = MultiplyMatrices (temp, D, B);  

      stress = MultiplyMatrices (stress, temp, d);
    
      sigma_x = VectorData (stress) [1];
      sigma_y = VectorData (stress) [2];
      tau_xy = VectorData (stress) [3];

      diameter = sqrt((sigma_x-sigma_y)*(sigma_x-sigma_y)/4 + tau_xy*tau_xy); 

      sigma1 = (sigma_x + sigma_y)/2 + diameter;

      element -> stress [6*(i - 1) + 1] = x;
      element -> stress [6*(i - 1) + 2] = y; 
      element -> stress [6*(i - 1) + 3] = sigma_x;
      element -> stress [6*(i - 1) + 4] = sigma_y;
      element -> stress [6*(i - 1) + 5] = tau_xy;
      element -> stress [6*(i - 1) + 6] = sigma1;
   }

   return 0;
} 

Matrix IsoQuadLocalB (element, numnodes, dNdx, dNdy, point)
   Element	element;
   unsigned	numnodes;
   Matrix	dNdx, dNdy;
   unsigned	point;
{
   unsigned		i;
   static Matrix	B = NullMatrix;

   if (B == NullMatrix) {
      B = CreateMatrix (3,8);
      if (B == NullMatrix)
         Fatal ("allocation error creating elt %d B matrix",element -> number);
   }

   for (i = 1 ; i <= numnodes ; i++) {
      B -> mdata [1][2*i - 1] = dNdx -> mdata [i][point];
      B -> mdata [1][2*i]     = 0.0;
      B -> mdata [2][2*i - 1] = 0.0;
      B -> mdata [2][2*i]     = dNdy -> mdata [i][point];
      B -> mdata [3][2*i - 1] = dNdy -> mdata [i][point];
      B -> mdata [3][2*i]     = dNdx -> mdata [i][point];
   }

   MatrixCols (B) = numnodes*2;

   return B;
}
    
Vector GlobalQuadShapeFunctions (element, dNdxi, dNde, dNdx, dNdy, 
                                 ninteg,nodes)
   Element	element;
   int		ninteg;
   Matrix	dNdxi, dNde,
		dNdx, dNdy;
   unsigned	nodes;
{
   unsigned		i,j;
   static Vector	jac,
			dxdxi, dxde,
			dydxi, dyde = NullMatrix;

   if (dyde == NullMatrix) {

      dxdxi = CreateVector (4);
      dxde  = CreateVector (4);
      dydxi = CreateVector (4);
      dyde  = CreateVector (4);
      jac = CreateVector (4);
    
       if (dxdxi == NullMatrix || dxde == NullMatrix || 
           dydxi == NullMatrix || dyde == NullMatrix || jac == NullMatrix) 

          Fatal ("allocation error computing elt %d shape functions",element -> number);
   }
 
   for (i = 1 ; i <= 4 ; i++) {
      dxdxi -> vdata [i] = 0.0; 
      dxde -> vdata [i] = 0.0; 
      dydxi -> vdata [i] = 0.0; 
      dyde -> vdata [i] = 0.0; 
      jac -> vdata [i] = 0.0;
   }

   for (i = 1 ; i <= ninteg ; i++) {
      for (j = 1 ; j <= nodes ; j++) {

         dxdxi -> vdata [i] += dNdxi -> mdata [j][i]*
                                   (element -> node[j] -> x);

         dxde -> vdata [i] += dNde -> mdata [j][i]*
                                   (element -> node[j] -> x);

         dydxi -> vdata [i] += dNdxi -> mdata [j][i]*
                                   (element -> node[j] -> y);
     
         dyde -> vdata [i] += dNde -> mdata [j][i]*
                                   (element -> node[j] -> y);
      }

      jac -> vdata [i] = dxdxi -> vdata[i] * dyde -> vdata[i] -
                         dxde -> vdata[i] * dydxi -> vdata[i];

      for (j = 1 ; j <= nodes ; j++) {
         dNdx -> mdata[j][i] = (dNdxi -> mdata [j][i]*dyde -> vdata[i] -
                               dNde -> mdata[j][i]*dydxi -> vdata[i])/
                               jac -> vdata[i];

         dNdy -> mdata[j][i] = -(dNdxi -> mdata[j][i]*dxde -> vdata[i] -
                                dNde -> mdata[j][i]*dxdxi -> vdata[i])/
                                jac -> vdata[i]; 
      }
   }

   return jac;
}

/*****************************************************************************
*
* Function:	LocalQuadShapeFunctions
*
* Description:  calculates the shape functions and the derivatives (w/ respect
*		to xi, eta coordinates) of the shape function for a four to 
*		nine node plane stress / plane strain element	
*
* Note:		The approach looks rather brutish, but it seems much clearer
*		to me this way and we have to do each individual computation 
*		anyways, whether we put ourselves in a loop and use index
*		notation or not ...
*
******************************************************************************/

unsigned LocalQuadShapeFunctions (element, N, dNdx, dNde, weights, force_init)
   Element	element;
   Matrix	N,
		dNdx,
		dNde;
   unsigned	force_init;
   Vector	weights;
{
   unsigned		i,j,k;
   int			ninteg;
   double		eta,xi;
   double		Nt[5];
   double		de[5],dx[5];
   double		*gauss_points;
   double		*gauss_wts;
   unsigned		numnodes;
   static unsigned	prev_nodes = 0;
   

   if (element -> node[3] -> number == element -> node[4] -> number) 
      numnodes = 3;
   else   
      numnodes = 4;   

   if (numnodes == prev_nodes && !force_init)
      return numnodes;

   ninteg = 2;
   GaussPoints (ninteg, &gauss_points, &gauss_wts);
   
   for (i = 0 ; i < ninteg ; i++) {
      xi  = gauss_points [i];
      for (j = 0 ; j < ninteg ; j++) {

         eta = gauss_points [j];
         Nt [1] = 0.25*(1 - eta)*(1 - xi);
         dx [1] = 0.25*(-1 + eta);
         de [1] = 0.25*(-1 + xi);
         Nt [2] = 0.25*(1 - eta)*(1 + xi);
         dx [2] = 0.25*(1 - eta);
         de [2] = 0.25*(-1 - xi);
         Nt [3] = 0.25*(1 + eta)*(1 + xi);
         dx [3] = 0.25*(1 + eta);
         de [3] = 0.25*(1 + xi);
         Nt [4] = 0.25*(1 + eta)*(1 - xi);
         dx [4] = 0.25*(-1 - eta);
         de [4] = 0.25*(1 - xi);

         if (numnodes == 3) {
             Nt [3] += Nt [4];
             dx [3] += dx [4];
             de [3] += de [4];
         }

         for (k = 1 ; k <= 4 ; k++) {
            N -> mdata [k][i*ninteg + j+1] = Nt [k];
            dNdx -> mdata [k][i*ninteg + j+1] = dx [k];
            dNde -> mdata [k][i*ninteg + j+1] = de [k];
         }

         weights -> vdata [i*ninteg + j+1] = gauss_wts [j];
      }
   }

   prev_nodes = numnodes;
   return numnodes;
}

Vector IsoQuadEquivNodalForces (element, err_count)
   Element	element;
   int		*err_count;
{
   double		L;
   double		wa,wb;
   double		force1,
			force2;
   int			count;
   double		xc1,xc2,
			yc1,yc2;
   double		thick;
   unsigned		node_a,
			node_b;
   unsigned		i,j;
   static Vector 	equiv = NullMatrix;
 
   if (equiv == NullMatrix) {
      equiv = CreateVector (8);

      if (equiv == NullMatrix)
         Fatal ("error creating equivalent force vector for element %d",element -> number);
   }

   count = 0;
 
   if (element -> numdistributed > 2) {
      error ("quad element %d can have at most two distributed loads",element -> number);
      count++;
   }

   thick = element -> material -> t;

   for (i = 1 ; i <= 8 ; i++)
      VectorData (equiv) [i] = 0.0;

   for (i = 1 ; i <= element -> numdistributed ; i++) {

      if (element -> distributed[i] -> nvalues != 2) {
         error ("load %s does not have 2 nodal values (element %d)",
                 element -> distributed[i] -> name,element -> number);
         count++;
      }

      if (element -> distributed[i] -> direction != DirX &&
         element -> distributed[i] -> direction != DirY) {
          error ("invalid direction specified for load %s (element %d)",
                 element -> distributed[i] -> name,element -> number);
          count++;
      }

      node_a = element -> distributed[i] -> value[1].node;
      node_b = element -> distributed[i] -> value[2].node;

      if (node_a < 1 || node_a > 4 || node_b < 1 || node_b > 4) {
         error ("incorrect node numbering for load %s (element %d)", 
                element -> distributed[i] -> name,element -> number);
         count++;
      }

      if (node_a == node_b) {
         error ("incorrect node numbering for load %s (element %d)", 
                element -> distributed[i] -> name,element -> number);
         count++;
      }

      xc1 = element -> node[node_a] -> x;
      xc2 = element -> node[node_b] -> x;
      yc1 = element -> node[node_a] -> y;
      yc2 = element -> node[node_b] -> y;

      L = sqrt ((xc1 - xc2)*(xc1 - xc2) + (yc1 - yc2)*(yc1 - yc2));

      if (L <= TINY) {
         error ("length of side of element %d is zero to machine precision",element -> number);
         count ++;
      } 

	/* 
	 * Thats all the error checking, bail out if we've had any
	 */

      if (count) {
         *err_count = count;
         return NullMatrix;
      }

      wa = element -> distributed[i] -> value[1].magnitude;
      wb = element -> distributed[i] -> value[2].magnitude;

      if (wa == wb) 		          /* uniform distributed load */
         force1 = force2 = wa*L*thick/2.0;
      else if (fabs(wa) > fabs(wb)) {     /* load sloping node1 to node2 */
         force2 = wb*L*thick/2.0 + (wa - wb)*L*thick/6.0;
         force1 = wb*L*thick/2.0 + (wa - wb)*L*thick/3.0; 
      }
      else if (fabs(wa) < fabs(wb)) {     /* load sloping node2 to node1 */
         force2 = wa*L*thick/2.0 + (wb - wa)*L*thick/6.0;
         force1 = wa*L*thick/2.0 + (wb - wa)*L*thick/3.0; 
      } 

      if (element -> distributed[i] -> direction == DirX) {
         VectorData (equiv) [2*node_a - 1] += force1;
         VectorData (equiv) [2*node_b - 1] += force2;
      }
      else {
         VectorData (equiv) [2*node_a] += force1;
         VectorData (equiv) [2*node_b] += force2;
      } 
   }

	/*
	 * Now that we know all is okay, allocate some memory if we
	 * haven't already done so for some other element
	 */

   for (i = 1 ; i <= element -> definition -> numnodes ; i++) {
      if (element -> node[i] -> eq_force == NULL) {
         element -> node[i] -> eq_force = Allocate (double, 6);

         if (element -> node[i] == NULL)
            Fatal ("unable to allocate memory");

         UnitOffset (element -> node[i] -> eq_force);
         for (j = 1 ; j <=6 ; j++)
            element -> node[i] -> eq_force[j] = 0.0;
      }
   }

   *err_count = 0;
   return equiv; 
}
