/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/************************************************************************
 * File:	beam.c							*
 *									*
 * Description:	This file contains the definition structure and		*
 *		stiffness function for a beam element.			*
 *									*
 * History:	V1.0 by Jason Gobat and Darren C. Atkinson		*
 ************************************************************************/

# include <math.h>
# include "allocate.h"
# include "element.h"
# include "misc.h"


struct definition beamDefinition = {
    beamType, Linear, 2, 2, 3, {0, 1, 2, 6, 0, 0, 0}
};

Matrix BeamLocalK           ( );
Matrix BeamTransformMatrix  ( );
Vector BeamEquivNodalForces ( );

int beamEltStiffness (element)
   Element 	element;
{
   Matrix		T,
			ke;
   Vector		equiv;
   int      		count;
   static Matrix 	Tt,
			temp = NullMatrix;

   if (temp == NullMatrix) {
      Tt = CreateMatrix (6,6);
      temp = CreateMatrix (6,6);

      if (Tt==NullMatrix || temp==NullMatrix)
         Fatal ("allocation error creating element %d stiffness matrix",element -> number);
   }

   if (element -> material -> A == 0) {
      error ("beam element %d has 0.0 for cross-sectional area (A)",element -> number);
      return 1;
   }
   if (element -> material -> E == 0) {
      error ("beam element %d has 0.0 for Young's modulus (E)", element -> number);
      return 1;
   }
   if (element -> material -> Ix == 0) {
      error ("beam element %d has 0.0 for Ixx moment of inertia (Ix)", element -> number);
      return 1;
   }

   ke = BeamLocalK (element);
   if (ke == NullMatrix)
      return 1;

   T = BeamTransformMatrix (element);
   if (T == NullMatrix)
      return 1;

   Tt = TransposeMatrix (Tt,T);

   element -> K = CreateMatrix (6,6);
   if (element -> K == NullMatrix) 
      Fatal ("allocation error creating element %d stiffness matrix",element -> number);

   temp = MultiplyMatrices (temp,Tt,ke); 
   element -> K = MultiplyMatrices (element -> K,temp,T);

   if (element -> numdistributed > 0) {
      equiv = BeamEquivNodalForces (element, &count);
      if (equiv == NullMatrix)
         return count;

      element -> node[1] -> eq_force[1] += VectorData (equiv) [1];
      element -> node[1] -> eq_force[2] += VectorData (equiv) [2];
      element -> node[1] -> eq_force[6] += VectorData (equiv) [3];
      element -> node[2] -> eq_force[1] += VectorData (equiv) [4];
      element -> node[2] -> eq_force[2] += VectorData (equiv) [5];
      element -> node[2] -> eq_force[6] += VectorData (equiv) [6];
   }
 
   return 0;
}

int beamEltStress (element)
   Element	element;
{
   unsigned		i;
   int			count;
   static Vector	f,
			dlocal,
			d = NullMatrix;
   Matrix		T,
			ke;
   Vector		equiv;
   static Vector	eq_local;

   if (d == NullMatrix) {
      d = CreateVector (6);
      f = CreateVector (6);
      dlocal = CreateVector (6);
      equiv = CreateVector (6);
      eq_local = CreateVector (6);

      if (d == NullMatrix || f == NullMatrix || dlocal == NullMatrix ||
          equiv == NullMatrix || eq_local == NullMatrix)

         Fatal ("allocation error in element %d stresses",element -> number);
   } 

   VectorData (d) [1] = element -> node[1] -> dx[1];
   VectorData (d) [2] = element -> node[1] -> dx[2]; 
   VectorData (d) [3] = element -> node[1] -> dx[6]; 
   VectorData (d) [4] = element -> node[2] -> dx[1];
   VectorData (d) [5] = element -> node[2] -> dx[2];
   VectorData (d) [6] = element -> node[2] -> dx[6]; 

   T = BeamTransformMatrix (element);
   if (T == NullMatrix)
      return 1;

   ke = BeamLocalK (element);
   if (ke == NullMatrix)
      return 1;

   dlocal = MultiplyMatrices (dlocal, T, d);
   f = MultiplyMatrices (f, ke, dlocal);

   if (element -> numdistributed > 0) {
      equiv = BeamEquivNodalForces (element, &count);
      if (equiv == NullMatrix)
         return count;

      eq_local = MultiplyMatrices (eq_local, T, equiv);

      for (i = 1 ; i <= 6 ; i++)
         VectorData (f) [i] -= VectorData (eq_local) [i];
   } 

   element -> stress = Allocate (double, 6);
   if (element -> stress == NULL)
      Fatal ("allocation error creating element %d stress vector",element -> number);

   element -> numstresses = 6;
   UnitOffset (element -> stress);

   for (i = 1; i< 6 ; i++)
      element -> stress [i] = VectorData (f) [i];

   return 0;          
} 

	/*
	 * These are the essentially private functions
	 */

Matrix BeamLocalK (element)
   Element	element;
{
   double		L,
			L3,
			L2;
   double		EI,
			AEonL;
   static Matrix	ke = NullMatrix;

   if (ke == NullMatrix) {
      ke = CreateMatrix (6,6);
      
      if (ke == NullMatrix)
         Fatal ("allocation error in local element stiffness %d", element -> number);
   }

   ZeroMatrix (ke);

   L = ElementLength (element, 2); 

   if (L <= TINY) {
      error ("length of element %d is zero to machine precision",element -> number);
      return NullMatrix;
   } 

   L2 = L*L;
   L3 = L2*L;

   EI = element -> material -> E * element -> material -> Ix;
   AEonL = (element -> material -> E * element -> material -> A)/L;

   MatrixData (ke) [1][1] = AEonL;
   MatrixData (ke) [1][4] = -AEonL;
   MatrixData (ke) [2][2] = 12*EI/L3;
   MatrixData (ke) [2][3] = 6*EI/L2;
   MatrixData (ke) [2][5] = -12*EI/L3;
   MatrixData (ke) [2][6] = 6*EI/L2;
   MatrixData (ke) [3][3] = 4*EI/L;
   MatrixData (ke) [3][5] = -6*EI/L2;
   MatrixData (ke) [3][6] = 2*EI/L;
   MatrixData (ke) [4][4] = AEonL;
   MatrixData (ke) [5][5] = 12*EI/L3;
   MatrixData (ke) [5][6] = -6*EI/L2;
   MatrixData (ke) [6][6] = 4*EI/L;

   MirrorMatrix (ke,ke);

   return ke;
}

Matrix BeamTransformMatrix (element)
   Element	element;
{
   double		cx,cy,
			L;
   static Matrix	T = NullMatrix;

   if (T == NullMatrix) {
      T = CreateMatrix (6,6);

      if (T == NullMatrix)
         Fatal ("allocation error in element %d transform matrix", element -> number);
   }

   ZeroMatrix (T);

   L = ElementLength (element, 2);

   if (L <= TINY) {
      error ("length of element %d is zero to machine precision",element -> number);
      return NullMatrix;
   } 

   cx = (element -> node[2] -> x - element -> node[1] -> x)/L;
   cy = (element -> node[2] -> y - element -> node[1] -> y)/L;

   MatrixData (T) [1][1] = cx;
   MatrixData (T) [1][2] = cy;
   MatrixData (T) [2][1] = -cy;
   MatrixData (T) [2][2] = cx;
   MatrixData (T) [3][3] = 1;
   MatrixData (T) [4][4] = cx;
   MatrixData (T) [4][5] = cy;
   MatrixData (T) [5][4] = -cy;
   MatrixData (T) [5][5] = cx;
   MatrixData (T) [6][6] = 1;

   return T;
}

Vector BeamEquivNodalForces (element, err_count)
   Element	element;
   int		*err_count;
{
   double		L;
   double		wa,wb;
   double		force1,
			force2,
			moment1,
			moment2;
   int			count;
   unsigned		i,j;
   Matrix		T;
   static Matrix	Tt;
   static Vector 	equiv = NullMatrix;
   static Vector	result;
 
   if (equiv == NullMatrix) {
      equiv = CreateVector (6);
      result = CreateVector (6);
      Tt = CreateMatrix (6,6);

      if (equiv == NullMatrix || result == NullMatrix || Tt == NullMatrix)
         Fatal ("error creating equivalent force vector for element %d",element -> number);
   }

   count = 0;
 
   if (element -> numdistributed != 1) {
      error ("beam elt %d can only have one distributed load", element -> number);
      count++;
   }
 
   if (element -> distributed[1] -> nvalues != 2) {
      error ("beam elt %d does not have 2 nodal values for a distributed load",
              element -> number);
      count++;
   }

   L = ElementLength (element, 2);

   if (L <= TINY) {
      error ("length of element %d is zero to machine precision",element -> number);
      count ++;
   } 

   if (element -> distributed[1] -> direction != Perpendicular &&
       element -> distributed[1] -> direction != Parallel) {
       error ("invalid direction specified for beam elt %d distributed load",
              element -> number);
       count++;
   }

   for (i = 1 ; i <= element -> distributed[1] -> nvalues ; i++) {
      if (element -> distributed[1] -> value[i].node < 1 || 
          element -> distributed[1] -> value[i].node > 2) {

          error ("incorrect node numbering for beam elt %d distributed load",
                  element -> number);
          count++;
      }
   }

   if (element -> distributed[1] -> value[1].node == 
       element -> distributed[1] -> value[2].node) {

       error ("incorrect node numbering for elt %d distributed load", element -> number);
       count++;
   }

	/* 
	 * Thats all the error checking, bail out if we've had any
	 */

   if (count) {
      *err_count = count;
      return NullMatrix;
   }

   if (element -> distributed[1] -> value[1].node == 1) {
      wa = element -> distributed[1] -> value[1].magnitude;
      wb = element -> distributed[1] -> value[2].magnitude;
   }
   else if (element -> distributed[1] -> value[1].node == 2) {
      wb = element -> distributed[1] -> value[1].magnitude;
      wa = element -> distributed[1] -> value[2].magnitude;
   }

   if (element -> distributed[1] -> direction == Perpendicular) {
      if (wa == wb) {		     /* uniform distributed load    */
         force1 = force2 = -wa*L/2.0;
         moment1 = -wa*L*L/12.0;
         moment2 = -moment1;
      } 
      else if (fabs(wa) > fabs(wb)) {     /* load sloping node1 to node2 */
         force1 = -wb*L/2.0 - 7.0*(wa - wb)*L/20.0; 
         force2 = -wb*L/2.0 - 3.0*(wa - wb)*L/20.0;
         moment1 = -wb*L*L/12.0 - (wa - wb)*L*L/20.0;
         moment2 = wb*L*L/12.0 + (wa - wb)*L*L/30.0;
      }
      else if (fabs(wa) < fabs(wb)) {     /* load sloping node2 to node1 */
         force1 = -wa*L/2.0 - 3.0*(wb - wa)*L/20.0; 
         force2 = -wa*L/2.0 - 7.0*(wb - wa)*L/20.0;
         moment1 = -wa*L*L/12.0 - (wb - wa)*L*L/30.0;
         moment2 = wa*L*L/12.0 + (wb - wa)*L*L/20.0;
      } 
      VectorData (equiv) [1] = 0.0;
      VectorData (equiv) [2] = force1;
      VectorData (equiv) [3] = moment1;
      VectorData (equiv) [4] = 0.0;
      VectorData (equiv) [5] = force2;
      VectorData (equiv) [6] = moment2;
   } else {
      if (wa == wb) 
         force1 = force2 = wa*L/2;
      else if (fabs (wa) > fabs (wb)) {
         force1 = wb*L/2 + (wa - wb)*L/3;
         force2 = wb*L/2 + (wa - wb)*L/6;
      }
      else if (fabs (wb) > fabs (wa)) {
         force1 = wa*L/2 + (wb - wa)*L/6;
         force2 = wa*L/2 + (wb - wa)*L/3;
      }
      VectorData (equiv) [1] = force1;
      VectorData (equiv) [2] = 0.0;
      VectorData (equiv) [3] = 0.0;
      VectorData (equiv) [4] = force2;
      VectorData (equiv) [5] = 0.0;
      VectorData (equiv) [6] = 0.0;
   }

	/*
	 * Now that we know all is okay, allocate some memory if we
	 * haven't already done so for some other element
	 */

   for (i = 1 ; i <= element -> definition -> numnodes ; i++) {
      if (element -> node[i] -> eq_force == NULL) {
         element -> node[i] -> eq_force = Allocate (double, 6);

         if (element -> node[i] == NULL)
            Fatal ("unable to allocate memory");

         UnitOffset (element -> node[i] -> eq_force);
         for (j = 1 ; j <= 6 ; j++)
            element -> node[i] -> eq_force[j] = 0.0;
      }
   }

   T = BeamTransformMatrix (element);
   Tt = TransposeMatrix (Tt, T);
   result = MultiplyMatrices (result, Tt, equiv);

   *err_count = 0;
   return result; 
}
