/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/************************************************************************
*
* File:		panel.c
*
* Description:	contains all of the functions that constitute callbacks
*		and action procedures from the Velvet main control
*		panel widgets.
*
* History:	by Jason I. Gobat and Darren C. Atkinson
*
*************************************************************************/

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <unistd.h>
# include <X11/Xlib.h>
# include <X11/Intrinsic.h>
# include <X11/StringDefs.h>
# include <X11/Shell.h>
# include <X11/Xaw/Simple.h>
# include <X11/Xaw/AsciiText.h>
# include <X11/Xaw/Viewport.h>
# include <X11/Xaw/SmeBSB.h>
# include "fe.h"
# include "Drawing.h"
# include "forms.h"
# include "dialog.h"
# include "procedures.h"
# include "problem.h"
# include "globals.h"
# include "util.h"

# include "element.h"

# define FONTSIZE 14
# define sgn(x) (x < 0 ? -1 : x > 0)


extern Panel   panel [ ];

void GetConfiguration ()
{
    String			result [8];
    extern EntryRecord		config_entries;
    static String		suggestion [5];
    static unsigned		flag = 1;
    unsigned			status, i;
 
    if (flag) {
       for (i = 0 ; i < 5 ; i++) 
          suggestion [i] = (String) XtMalloc (sizeof(char)*80);

       flag = 0;

       sprintf (suggestion [0], "%s", elt_color);
       sprintf (suggestion [1], "%s", node_color);
       sprintf (suggestion [2], "%s", label_font);
       sprintf (suggestion [3], "%s", tool_color);
       sprintf (suggestion [4], "%s", text_font);
       FillTextBuffers (config_form, config_entries, suggestion);
    }

    SetShellTitle (config_form -> shellwidget, "Canvas Configuration");
    status = GetEntryFormValues (config_form, config_entries, result);

    if (!status) return;

    if (strlen (result [0]) > 0) 
       elt_color = result [0];
    if (strlen (result [1]) > 0)
       node_color = result [1];
    if (strlen (result [2]) > 0)
       label_font = result [2];
    if (strlen (result [3]) > 0)
       tool_color = result [3];
    if (strlen (result [4]) > 0)
       text_font = result [4]; 

    changeflag = True;
} 

void GetCanvasParameters ()
{
    String			result [6];
    extern EntryRecord  	canvas_entries;
    Arg				arglist[8];
    Cardinal			count; 
    static String		suggestion [6];
    static unsigned		flag = 1;
    unsigned			status,i;
    float			newxmax,newxmin,
				newymax,newymin,
				newgrid,newsnap;

    if (flag) {
       for (i = 0 ; i < 6 ; i++) 
          suggestion [i] = (String) XtMalloc (sizeof(char)*10);
       flag = 0;
    }

    sprintf (suggestion [0],"%7.5f",xmin);
    sprintf (suggestion [1],"%7.5f",xmax);
    sprintf (suggestion [2],"%7.5f",ymin);
    sprintf (suggestion [3],"%7.5f",ymax);
    sprintf (suggestion [4],"%7.5f",grid_size);
    sprintf (suggestion [5],"%7.5f",snap_size); 
    FillTextBuffers (canvas_form, canvas_entries, suggestion);

    SetShellTitle (canvas_form -> shellwidget, "Canvas Parameters");
    status = GetEntryFormValues (canvas_form, canvas_entries, result);

    if (!status) return;

    count = 0;
    changeflag = True;
   
    if (strlen (result [0]) != 0) 
       newxmin  = atof (result [0]);
    if (strlen (result [1]) != 0) 
       newxmax  = atof (result [1]);
    if (strlen (result [2]) != 0) 
       newymin  = atof (result [2]);
    if (strlen (result [3]) != 0) 
       newymax  = atof (result [3]);

    if (newxmin >= newxmax) {
       error ("Range error in x limits.");
       return;
    } else {
       xmax = newxmax;
       xmin = newxmin;
       XtSetArg (arglist [count], XtNxMax, Float2Arg(xmax)); count++;
       XtSetArg (arglist [count], XtNxMin, Float2Arg(xmin)); count++; 
    }

    if (newymin >= newymax) {
       error ("Range error in y limits.");
       return;
    } else { 
       ymax = newymax;
       ymin = newymin;
       XtSetArg (arglist [count], XtNyMin, Float2Arg(ymin)); count++;
       XtSetArg (arglist [count], XtNyMax, Float2Arg(ymax)); count++;
    }

    if (strlen (result [4]) != 0) {
       newgrid = atof (result [4]);
       if (!grid_status && newgrid != grid_size) ToggleGridStatus (); 
       grid_size = newgrid;
       XtSetArg (arglist [count], XtNgridSize, Float2Arg(grid_size)); count++;
    }

    if (strlen (result [5]) != 0) {
       newsnap = atof (result [5]);
       if (!snap_status && newsnap != snap_size) ToggleSnapStatus (); 
       snap_size = newsnap;
       XtSetArg (arglist [count], XtNsnapSize, Float2Arg(snap_size)); count++;
    }

    XtSetValues (drawing, arglist, count);
}

void ToggleSnapStatus ()
{
    Arg            arglist[1];

    if (snap_status) 
       XtSetArg (arglist[0], XtNrightBitmap, NULL);
    else
       XtSetArg (arglist[0], XtNrightBitmap, checkmark);

    XtSetValues (XtNameToWidget (panel [7].menu, "snap") , arglist, 1);

    snap_status = !snap_status;

    XtSetArg (arglist [0], XtNsnap, snap_status);
    XtSetValues (drawing, arglist, 1);
    changeflag = True;
}

void ToggleNumberStatus ()
{
   Arg		arglist [1];

   if (number_status)
      XtSetArg (arglist [0], XtNrightBitmap, NULL);
   else
      XtSetArg (arglist [0], XtNrightBitmap, checkmark);

   XtSetValues (XtNameToWidget (panel [7].menu, "numbering"), arglist, 1);

   number_status  = !number_status;
   SetNodeNumbering (number_status ? True : False);
   SetElementNumbering (number_status ? True : False);
   changeflag = True;
}

void ToggleGridStatus ()
{
    Arg            arglist[1];

    if (grid_status) 
       XtSetArg (arglist[0], XtNrightBitmap, NULL);
    else
       XtSetArg (arglist[0], XtNrightBitmap, checkmark);

    XtSetValues (XtNameToWidget (panel [7].menu, "grid"), arglist, 1);
    
    grid_status = !grid_status;

    XtSetArg (arglist [0], XtNgrid, grid_status);
    XtSetValues (drawing, arglist, 1);
    changeflag = True;
}

void ZoomAll ()
{
   float	xscale,
		yscale,
		scale;
   Dimension	height,
		width;
   Arg		arglist [4];
   Cardinal	count;

   count = 0;
   XtSetArg (arglist [count], XtNheight, &height); count++;
   XtSetArg (arglist [count], XtNwidth, &width); count++;
   XtGetValues (viewport, arglist, count);

   xscale = (float) width / (xmax - xmin);
   yscale = (float) height / (ymax - ymin);

   if (xscale < yscale) scale = xscale;
   else scale = yscale;

   count = 0;
   XtSetArg (arglist [count], XtNxScale, Float2Arg(scale)); count++;
   XtSetArg (arglist [count], XtNyScale, Float2Arg(scale)); count++;
   XtSetValues (drawing, arglist, count);
   changeflag = True;
}

void ZoomStart ()
{
   Cardinal	count;
   Arg		arglist [4];

   SetEditMode ( );

   count = 0;
   XtSetArg (arglist [count], XtNcursorName, "left_ptr"); count++;
   XtSetValues (drawing, arglist, count);

   ChangeStatusLine ("Select first corner:", True);

   XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
   XtAddCallback (drawing, XtNbuttonCallback, ZoomSelect, NULL);
   XtOverrideTranslations (entry, 
        XtParseTranslationTable ("<Key>Return: ZoomAP()"));

   if (DW_SetForeground (drawing, tool_color) == False)
	(void) DW_SetForeground (drawing, "black");
}

void ZoomAP ()
{
   static unsigned	corner_number = 0;
   static float		xl,xr,
			yb,yt;
   Figure		box;
   char*		status;

   if (corner_number == 0) {
      status = GetTextCoordinates (&xl,&yb,NULL); 
      if (status == NULL) {
         corner_number++;
         ChangeStatusLine ("Select second corner:", True);
      }
   } 
   else if (corner_number == 1) {
      status = GetTextCoordinates (&xr,&yt,NULL);
      if (status == NULL) {
         DW_SetInteractive (drawing, True);
         box = DW_DrawRectangle (drawing, True, xl, yb, xr - xl, yt - yb);
         DW_SetInteractive (drawing, False);
         ZoomEnd (box);
         DW_RemoveFigure (drawing, box);
         corner_number = 0;
      }
   }
}

void ZoomSelect (w, clientData, callData)
   Widget	w;
   XtPointer	clientData;
   XtPointer	callData;
{
   static Point		corner;
   static Figure	box; 
   DrawingReport	*report;
   FigureAttributes	attributes;

   report = (DrawingReport *) callData;

   switch (report -> event -> type) {
   case ButtonPress:
      ChangeStatusLine ("- Select second corner -", False);
      corner = report -> unsnapped;
      DW_SetInteractive (w, True);
      box = DW_DrawRectangle (w, True, corner.x, corner.y, 0.0, 0.0);
      break;

   case MotionNotify:
      attributes.width  = report -> unsnapped.x - corner.x;
      attributes.height = report -> unsnapped.y - corner.y;
      DW_SetAttributes (w, box, DW_FigureSize, &attributes);
      break;

   case ButtonRelease:
      DW_SetInteractive (drawing, False);
      ZoomEnd (box);
      DW_RemoveFigure (drawing, box);
      break;
   }
} 

void ZoomEnd (box)
   Figure	box;
{
   float		xscale;
   float		yscale;
   float		scale;
   float		xr,xl,
			yt,yb;
   Dimension		pos_x,
			pos_y;
   Dimension		height,
			width;
   Arg			arglist [4];
   Cardinal		count;
   FigureAttributes	attributes;
   XRectangle		rect;
   
   count = 0;
   XtSetArg (arglist [count], XtNheight, &height); count++;
   XtSetArg (arglist [count], XtNwidth, &width); count++;
   XtGetValues (viewport, arglist, count);

   DW_GetAttributes (drawing, box, &attributes);

   if (attributes.width > 0) {
      xl = attributes.x;
      xr = attributes.width + xl;
   } else {
      xl = attributes.x + attributes.width;
      xr = attributes.x;
   }

   if (attributes.height > 0) {
      yb = attributes.y;
      yt = attributes.height + yb;
   } else {
      yb = attributes.y + attributes.height;
      yt = attributes.y;
   } 
  
   if (xr > xl) 
      xscale = (float) width / (xr - xl);
   else
      return;

   if (yt > yb)
      yscale = (float) height / (yt - yb);
   else
      return;

   if (xscale < yscale) scale = xscale;
   else scale = yscale;

   count = 0;
   XtSetArg (arglist [count], XtNxScale, Float2Arg(scale)); count++;
   XtSetArg (arglist [count], XtNyScale, Float2Arg(scale)); count++;
   XtSetValues (drawing, arglist, count); 

   DW_ClipBox (box, &rect);
   pos_x = rect.x;
   pos_y = rect.y;

   SetNormalMode ();

   XawViewportSetCoordinates (viewport, pos_x, pos_y);
   changeflag = True;
}

void ParseEntryLine ()
{
   extern PanelId     last_command;
   Arg		      arglist [1];
   String             result;
   unsigned	      i;
   XtPointer	      execute;
   extern TextCommand commands [ ];

   XtSetArg (arglist[0], XtNstring, &result);
   XtGetValues (entry, arglist, 1);

   if (strcmp (result, "") == 0) {
      if (last_command != -1)
         execute = &last_command;
   }
   else {
      execute = NULL;

      i = 0;
      while (commands [i].id > -1) {
         if (!strcmp (commands [i].name, result)) {
            execute = &commands [i].id;
            last_command = commands [i].id;
            break;
         }
         i ++;
      }
   }

   XtSetArg (arglist[0], XtNstring, "");
   XtSetValues (entry, arglist, 1);

   if (execute != NULL) 
      PanelCallback (NULL, execute, NULL);
   else
      XBell (XtDisplay (toplevel), 20);
}

void ActionToPanelConverter (w, event, params, num_params)
   Widget	w;
   XEvent	*event;
   String	*params;
   Cardinal	*num_params;
{
   XtPointer	execute;
   unsigned	i,j;

   i = 0;
   execute = NULL;
   while (strcmp (panel [i].name, "abort")) {
      for (j = 0 ; j < panel [i].numentries ; j++) {

         if (!strcmp (panel [i].menuentry [j].name, *params)) {
             execute = &panel [i].menuentry [j].id;
             break;
         }
      }

      if (execute != NULL) break;
      else i++;
   }

   if (execute != NULL)
      PanelCallback (NULL, execute, NULL);
}  

void QuitEdit ( )
{
   if (edit_mode == True) 
      SetNormalMode ();
}

void AbortEdit ( )
{
   if (edit_mode == True) 
      SetNormalMode ();
}

void SetNormalMode ()
{
   unsigned	i;
   Arg		arglist [2];
   Cardinal	count;

   ChangeStatusLine ("Command:", True);

   XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
   XtRemoveAllCallbacks (drawing, XtNmotionCallback);

   XtSetArg (arglist [0], XtNcursorName, "crosshair");
   XtSetValues (drawing, arglist, 1);

   XtOverrideTranslations (entry, 
        XtParseTranslationTable ("<Key>Return: ParseEntryLine()"));

   count = 0;
   XtSetArg (arglist[count], XtNeditType, XawtextEdit); count++;
   XtSetArg (arglist [count], XtNstring, ""); count++;
   XtSetValues (entry, arglist, 1);

   XawTextDisplayCaret (entry, True);

   XtAddCallback (drawing, XtNbuttonCallback, SelectCallback, NULL);

   AssignQuitAbort (QuitEdit, "QuitEdit", AbortEdit, "AbortEdit");

   if (sensitive_menus == True) {
      XtSetArg (arglist [0], XtNsensitive, True);
      for (i = 0 ; i < 7 ; i++)
         XtSetValues (panel [i].button, arglist, 1); 

      for (i = 0 ; i < 5 ; i++)
         XtSetValues (panel [7].menuentry [i].widget, arglist, 1);

      XtSetValues (panel [8].button, arglist, 1);

      XtSetArg (arglist [0], XtNsensitive, False);
      XtSetValues (quitbutton, arglist, 1);
      XtSetValues (abortbutton, arglist, 1);
   }

   edit_mode = False;
}

void AssignQuitAbort (quitCB, quitAP, abortCB, abortAP)
   XtCallbackProc	quitCB;
   XtCallbackProc	abortCB;
   String		quitAP;
   String		abortAP;
{
   char		cmd1 [256];
   char		cmd2 [256];

   XtRemoveAllCallbacks (quitbutton, XtNcallback);
   XtRemoveAllCallbacks (abortbutton, XtNcallback);

   XtAddCallback (quitbutton, XtNcallback, quitCB, NULL);
   XtAddCallback (abortbutton, XtNcallback, abortCB, NULL);

   sprintf (cmd1, "<Key>Escape: %s()", quitAP);
   sprintf (cmd2, "Ctrl<Key>c: %s()", abortAP);

   XtOverrideTranslations (entry,
      XtParseTranslationTable (cmd1));
   XtOverrideTranslations (entry,
      XtParseTranslationTable (cmd2));
}
                                

void SetEditMode ()
{
   Arg		arglist [1];
   unsigned	i;

   edit_mode = True;

   if (sensitive_menus == True) {
      XtSetArg (arglist [0], XtNsensitive, False);
      for (i = 0 ; i < 7 ; i++)
         XtSetValues (panel [i].button, arglist, 1); 

      for (i = 0 ; i < 5 ; i++)
         XtSetValues (panel [7].menuentry [i].widget, arglist, 1);

      XtSetValues (panel [8].button, arglist, 1);

      XtSetArg (arglist [0], XtNsensitive, True);
      XtSetValues (quitbutton, arglist, 1);
      XtSetValues (abortbutton, arglist, 1);
   }
}

 
void ChangeStatusLine (new_label, allow_input)
   String 	new_label;
   Boolean	allow_input;
{
   Arg		arglist [1];
   Cardinal	count;

   if (!allow_input) {
      count = 0;
      XtSetArg (arglist[count], XtNeditType, XawtextRead); count++;
      XtSetValues (entry, arglist, count);

      XawTextDisplayCaret (entry, False);
   }

   XtUnmanageChild (status);
   XtUnmanageChild (entry);

   count = 0;
   XtSetArg (arglist [count], XtNlabel, new_label); count++;
   XtSetValues (status, arglist, count);

   XtManageChild (status);
   XtManageChild (entry);
}

void SetShellTitle (shell, title)
   Widget	shell;
   String	title;
{
   Arg		arglist [1];

   XtSetArg (arglist [0], XtNtitle, title);
   XtSetValues (shell, arglist, 1);
   XtSetValues (error_dialog -> shellwidget, arglist, 1);
}

void UpdateTitle ()
{
   Arg	arglist [1];

   if (filename [0])
      XtSetArg (arglist [0], XtNtitle, filename);
   else
      XtSetArg (arglist [0], XtNtitle, "UnNamed");

   XtSetValues (toplevel, arglist, 1);
}

void SaveWidgetXWD ()
{
   String	xwd_file;
   FILE		*output;
   unsigned	selected;

   SetShellTitle (file_dialog -> shellwidget,"XWD Dump");

   selected = PopupDialog (file_dialog,"Save XWD as:","",&xwd_file);
   if (selected != Okay || xwd_file [0] == 0) 
      return;

   output = fopen (xwd_file, "w");
   if (output == NULL) {
      error ("could not open file for writing");
      return;
   }
 
   if (DumpWidget (drawing, output)) {
      fclose (output);
      unlink (xwd_file);
      return;
   }

   fclose (output); 

   return;
}

void DoNothing ()
{
}
