/*
    This file is part of the FElt finite element analysis package.
    Copyright (C) 1993 Jason I. Gobat and Darren C. Atkinson

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <X11/Intrinsic.h>
# include <X11/StringDefs.h>
# include <X11/Xaw/Simple.h>
# include "fe.h"
# include "forms.h"
# include "dialog.h"
# include "Drawing.h"
# include "procedures.h"
# include "problem.h"
# include "globals.h"
# include "vfe.h"
# include "util.h"
# include "objects.h"
# include "error.h"


void GetNodeInformation (node)
   Node		node;
{
   Arg          arglist [1];
   static char	information [1024];


   displayed_node = node;

   sprintf (information,"Number:        %d\n\
Location:      %g  %g  %g\n\
Force:         %s\n\
Constraint:    %s\n\
Displacements: % 11.5g  % 11.5g  % 11.5g\n\
               % 11.5g  % 11.5g  % 11.5g", node -> number, 
                            node -> x, node -> y, node -> z,
                            (node -> force == NULL ? "(none)" :
                                node -> force -> name),
                            (node -> constraint == NULL ? "(none)" : 
                                node -> constraint -> name),
                            node -> dx[1], node -> dx[2], node -> dx[3],
                            node -> dx[4], node -> dx[5], node -> dx[6]);

   XtSetArg (arglist [0], XtNstring, information);
   XtSetValues (nodewin, arglist, 1);

   XtPopup (nodeShell, XtGrabNone);
}

static Boolean first_time;

void EditAddNode ()
{
   SetShellTitle (error_dialog -> shellwidget, "Add Node");

   if (active_constraint_number == -1) {
	error ("No active constraint defined.");
	return;
   }

   first_time = True;
   SetEditMode ( );
   ChangeStatusLine ("Nodal coordinates:", True);

   XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
   XtAddCallback (drawing, XtNbuttonCallback, AddNodeCB, NULL);

   XtOverrideTranslations (entry, 
           XtParseTranslationTable ("<Key>Return: AddNodeAP()"));
}

void AddNodeCB (w, client_data, call_data)
   Widget	w;
   XtPointer	client_data,
		call_data;
{
   DrawingReport 	*report;

   report = (DrawingReport *) call_data;

   if (report -> event -> type != ButtonPress)
	return;

   if (report -> event -> xbutton.button == 1)
      DoAddNode (report -> snapped.x, report -> snapped.y, 0.0);

   if (report -> event -> xbutton.button == 3)
      QuitEdit ( );
}

void AddNodeAP ()
{
   char *status;
   float x,y,z = 0;

   status = GetTextCoordinates (&x, &y, NULL);
   if (status == NULL)
      DoAddNode (x, y, z);
}

void DoAddNode (x, y, z)
   float	x,y,z;
{
   char     message [40];
   Node     node;
   unsigned max;

   node = (Node) TreeMaximum (problem.node_tree);
   max = node != NULL ? node -> number : 0;

   node = CreateNode (max + 1);
   node -> constraint = active_constraint;
   node -> x = x;
   node -> y = y;
   node -> z = z;

   DrawNode (node, first_time);

   sprintf (message, "Added node %d.  Nodal coordinates:", node -> number);
   ChangeStatusLine (message, True);

   if (XtIsRealized (nodeShell) == True)
	GetNodeInformation (node);

   first_time = False;
   changeflag = True;
}


static Node             moved_node;
static Figure           moved_figure;
static Figure           ghost_figure;
static FigureAttributes attr;


static int MoveNode (item)
    Item item;
{
    unsigned i;
    unsigned numnodes;
    Element  element = (Element) item;


    numnodes = element -> definition -> numnodes;

    for (i = 1; i <= numnodes; i ++)
	if (element -> node [i] == moved_node) {
	    DW_GetAttributes (drawing, ((Drawn) element -> aux) -> figure,
			      &attr);
	    attr.points [i - 1].x = moved_node -> x;
	    attr.points [i - 1].y = moved_node -> y;
	    if (numnodes >= 2 && i == 1) {
		attr.points [numnodes].x = moved_node -> x;
		attr.points [numnodes].y = moved_node -> y;
	    }
	    DW_SetAttributes (drawing, ((Drawn) element -> aux) -> figure,
			      DW_FigurePoints, &attr);
	}

    return 0;
}


static void DoWalkNode (node)
    Node node;
{
    Drawn drawn;


    moved_node = node;
    attr.x = node -> x;
    attr.y = node -> y;
    drawn = (Drawn) node -> aux;

    DW_SetAutoRedraw (drawing, False);
    DW_SetAttributes (drawing, drawn -> figure, DW_FigureLocation, &attr);
    DW_SetAttributes (drawing, drawn -> label, DW_FigureLocation, &attr);
    (void) TreeSetIterator (problem.element_tree, MoveNode);
    (void) TreeIterate (problem.element_tree);
    DW_SetAutoRedraw (drawing, True);

    if (XtIsRealized (nodeShell))
	GetNodeInformation (node);

    changeflag = True;
}


void EditNodeInfo (node)
    Node node;
{
    int			i, status;
    String		result [6];
    extern EntryRecord	node_entries;
    static String	suggestion [6];
    static int		flag = 0;
    struct force        force;
    struct constraint   constraint;
    struct node		dummy;
    Item		newforce;
    Item		newconstraint;
    Item		found;
    double		x;
    double		y;
    double		z;


    if (!flag) {
	flag = 1;
	for (i = 0; i < 6; i ++)
	    suggestion [i] = (String) XtMalloc (sizeof (char) * 80);
    }

    sprintf (suggestion [0], "%d", node -> number);
    strcpy (suggestion [1], node -> force ? node -> force -> name : "");
    strcpy (suggestion [2], node -> constraint ? node -> constraint->name : "");
    sprintf (suggestion [3], "%g", node -> x);
    sprintf (suggestion [4], "%g", node -> y);
    sprintf (suggestion [5], "%g", node -> z);
    FillTextBuffers (node_form, node_entries, suggestion);

    SetShellTitle (node_form -> shellwidget, "Edit Node");
    status = GetEntryFormValues (node_form, node_entries, result);

    if (status) {
	if (strlen (result [0]) > 0) {
	    dummy.number = atoi (result [0]);
	    if (node -> number != dummy.number) {
		found = TreeSearch (problem.node_tree, &dummy);
		if (found == NULL) {
		    error ("Node %d does not exist.", dummy.number);
		    return;
		}

		node = (Node) found;
	    }

	    if (strlen (result [1]) == 0)
		newforce = NULL;
	    else {
		force.name = result [1];
		newforce = TreeSearch (problem.force_tree, &force);
		if (newforce == NULL) {
		    error ("Force %s does not exist.", force.name);
		    return;
		}
	    }

	    if (strlen (result [2]) == 0)
		newconstraint = NULL;
	    else {
		constraint.name = result [2];
		newconstraint = TreeSearch (problem.constraint_tree, &constraint);
		if (newconstraint == NULL) {
		    error ("Constraint %s does not exist.", constraint.name);
		    return;
		}
	    }

	    x = node -> x;
	    y = node -> y;
	    z = node -> z;

	    node -> force = (Force) newforce;
	    node -> constraint = (Constraint) newconstraint;
	    node -> x = atof (result [3]);
	    node -> y = atof (result [4]);
	    node -> z = atof (result [5]);

	    if (x != node -> x || y != node -> y || z != node -> z)
		DoWalkNode (node);

	    changeflag = True;
	} else
	    error ("No node number specified.");
    }
}


void DeleteNodeGroup (figures, nfigures)
    Figure  *figures;
    unsigned nfigures;
{
    unsigned         i;
    Figure           fig;
    Drawn            drawn;
    Node             node;
    unsigned         numleft;
    Boolean	     firsttime;
    Boolean	     newinfo;


    numleft = 0;
    firsttime = True;
    newinfo = False;

    for (i = 0; i < nfigures; i ++) {
	fig = figures [i];
	DW_GetAttributes (drawing, fig, &attr);

	if (attr.user_data == NULL || attr.type == TextFigure)
	    continue;

	node = (Node) attr.user_data;
	drawn = (Drawn) node -> aux;
	if (drawn -> type != DrawnNode)
	    continue;

	if (drawn -> ref_count) {
	    numleft ++;
	    continue;
	}

	if (firsttime == True) {
	    firsttime = False;
	    DW_SetAutoRedraw (drawing, False);
	}

	if (node == displayed_node)
	    newinfo = True;

	DW_RemoveFigure (drawing, drawn -> figure);
	DW_RemoveFigure (drawing, drawn -> label);
	(void) TreeDelete (problem.node_tree, node);
	DestroyNode (node);
    }


    if (newinfo && XtIsRealized (nodeShell)) {
	node = TreeMinimum (problem.node_tree);
	if (node != NULL)
	    GetNodeInformation (node);
	else {
	    XtPopdown (nodeShell);
	    XtUnrealizeWidget (nodeShell);
	}
    }


    if (firsttime == False) {
	DW_SetAutoRedraw (drawing, True);
	changeflag = True;
    }

    if (numleft)
	error ("Warning: elements still reference %d node%s.", numleft,
						numleft > 1 ? "s" : "");

    XtFree ((char *) figures);
}


void DoDeleteNode (node)
    Node node;
{
    static char message [80];
    Drawn drawn = (Drawn) node -> aux;
    Node newnode;


    if (drawn -> ref_count) {
	error ("Node %d is still referenced by %d element%s.", node -> number,
		drawn -> ref_count, drawn -> ref_count > 1 ? "s" : "");
	return;
    }

    if (drawn -> figure != NULL) {
	DW_SetAutoRedraw (drawing, False);
	DW_RemoveFigure (drawing, drawn -> figure);
	if (drawn -> label != NULL)
	    DW_RemoveFigure (drawing, drawn -> label);
	DW_SetAutoRedraw (drawing, True);
    }

    sprintf (message, "Node %d deleted.  Select node:", node -> number);

    if (node == displayed_node && XtIsRealized (nodeShell)) {
	newnode = TreePredecessor (problem.node_tree, node);
	if (newnode == NULL)
	    newnode = TreeSuccessor (problem.node_tree, node);
	if (newnode == NULL) {
	    XtPopdown (nodeShell);
	    XtUnrealizeWidget (nodeShell);
	} else
	    GetNodeInformation (newnode);
    }

    (void) TreeDelete (problem.node_tree, node);

    DestroyNode (node);
    ChangeStatusLine (message, True);
    changeflag = True;
}


void DeleteNodeCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Drawn            drawn;
    Node             node;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 2) {
	DeleteGroup (call_data, DeleteNodeGroup);
	return;
    }

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    node = (Node) attributes.user_data;
    drawn = (Drawn) node -> aux;
    if (drawn -> type != DrawnNode)
	return;

    DoDeleteNode (node);
}


void DeleteNodeAP ( )
{
    char       *status;
    struct node dummy;
    Item        found;


    if ((status = GetTextNumber (&dummy.number)) != NULL) {
	if (!strcmp (status, "w"))
	    DeleteGroup (NULL, DeleteNodeGroup);
	return;
    }

    found = TreeSearch (problem.node_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Node %d does not exist.", dummy.number);
	return;
    }

    DoDeleteNode (found);
}


void EditDeleteNode ( )
{
    Arg		arglist [1];

    SetEditMode ( );
    SetShellTitle (error_dialog -> shellwidget, "Delete Node");
    ChangeStatusLine ("Select node:", True);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, DeleteNodeCB, NULL);

    XtSetArg (arglist [0], XtNcursorName, "dotbox");
    XtSetValues (drawing, arglist, 1);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: DeleteNodeAP()"));
}


void EditNodeCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure           figure;
    Drawn            drawn;
    Node             node;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    node = (Node) attributes.user_data;
    drawn = (Drawn) node -> aux;
    if (drawn -> type != DrawnNode)
	return;

    EditNodeInfo (node);
}


void EditNodeAP ( )
{
    char       *status;
    struct node dummy;
    Item        found;


    if ((status = GetTextNumber (&dummy.number)) != NULL)
	return;

    found = TreeSearch (problem.node_tree, (Item) &dummy);
    if (found == NULL) {
	error ("Node %d does not exist.", dummy.number);
	return;
    }

    EditNodeInfo (found);
}


void EditNodeNumber ( )
{
    SetEditMode ( );
    SetShellTitle (error_dialog -> shellwidget, "Edit Node");
    ChangeStatusLine ("Select node:", True);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, EditNodeCB, NULL);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: EditNodeAP()"));
}


int DrawNode (node, first_time)
    Node    node;
    Boolean first_time;
{
    Figure		fig;
    Figure		label;
    char		buffer [10];
    FigureAttributes 	attr;
    Item		found;
    Drawn               drawn;


    found = TreeInsert (problem.node_tree, (Item) node);
    if (found != (Item) node)
	return 1;


    if (first_time == True) {
	if (DW_SetFont (drawing, label_font) == False)
	    (void) DW_SetFont (drawing, "fixed");

	if (DW_SetForeground (drawing, node_color) == False)
	    (void) DW_SetForeground (drawing, "black");
    }

    if (node -> z == 0) {

	fig = DW_FillArc (drawing, False, node -> x, node -> y,
                        6.0, 6.0, 0.0, 360.0);

	if (number_status == True) {
	    sprintf (buffer, " %d", node -> number);
	    label = DW_DrawText (drawing, True, node -> x, node -> y, buffer);
	} else
	    label = NULL;

    } else
	fig = label = NULL;

    if (node -> aux == NULL) {
	drawn = (Drawn) XtNew (struct drawn);
	node -> aux = (char *) drawn;
	drawn -> type = DrawnNode;
	drawn -> ref_count = 0;
    } else
	drawn = (Drawn) node -> aux;

    drawn -> figure = fig;
    drawn -> label = label;


    attr.user_data = (char *) node;

    if (fig != NULL)
	DW_SetAttributes (drawing, fig, DW_FigureUserData, &attr);
    if (label != NULL)
        DW_SetAttributes (drawing, label, DW_FigureUserData, &attr);

    return 0;
}


void QuitMoveNode ( )
{
    DW_RemoveFigure (drawing, ghost_figure);
    DW_SetInteractive (drawing, False);
    SetNormalMode ( );
}


void WalkNodeCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport *report;
    Node           node;


    report = (DrawingReport *) call_data;
    node = (Node) client_data;

    if (report -> event -> type == ButtonPress) {
	if (report -> event -> xbutton.button == 3)
	    QuitMoveNode ( );

	if (report -> event -> xbutton.button > 2)
	    return;

	node -> x = report -> snapped.x;
	node -> y = report -> snapped.y;
	DW_RemoveFigure (drawing, ghost_figure);
	DW_SetInteractive (drawing, False);
	DoWalkNode (node);
	SetNormalMode ( );

    } else if (report -> event -> type == MotionNotify) {
	attr.x = report -> snapped.x;
	attr.y = report -> snapped.y;
	DW_SetAttributes (drawing, ghost_figure, DW_FigureLocation, &attr);
    }
}


void WalkNodeAP ( )
{
    char *status;
    float x, y;

    status = GetTextCoordinates (&x, &y, NULL);
    if (status != NULL)
	return;

   moved_node -> x = x;
   moved_node -> y = y;
   QuitMoveNode ( );
   DoWalkNode (moved_node);
}


void DoMoveNode (node, motion)
    Node    node;
    Boolean motion;
{
    static char buffer [80];


    SetEditMode ( );
    sprintf (buffer, "Nodal coordinates for node %d:", node -> number);
    ChangeStatusLine (buffer, True);

    AssignQuitAbort (QuitMoveNode, "QuitMoveNode", QuitMoveNode,"QuitMoveNode");

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, WalkNodeCB, node);

    if (motion == True) {
	XtRemoveAllCallbacks (drawing, XtNmotionCallback);
	XtAddCallback (drawing, XtNmotionCallback, WalkNodeCB, node);
	DW_SetInteractive (drawing, True);
	ghost_figure = DW_FillArc (drawing, False, node -> x, node -> y,
					6.0, 6.0, 0.0, 360.0);
    } else {
	moved_figure = NULL;
	ghost_figure = NULL;
    }

    moved_node = node;
    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: WalkNodeAP()"));
}


void MoveNodeCB (w, client_data, call_data)
    Widget    w;
    XtPointer client_data;
    XtPointer call_data;
{
    DrawingReport   *report;
    FigureAttributes attributes;
    Figure	     figure;
    Node	     node;
    Drawn	     drawn;


    report = (DrawingReport *) call_data;

    if (report -> event -> type != ButtonPress)
	return;

    if (report -> event -> xbutton.button == 3)
	QuitEdit ( );

    if (report -> event -> xbutton.button != 1)
	return;

    figure = DW_FindFigure (w, report -> unsnapped.x, report -> unsnapped.y);

    if (figure == NULL)
	return;

    DW_GetAttributes (w, figure, &attributes);
    if (attributes.user_data == NULL)
	return;

    node = (Node) attributes.user_data;
    drawn = (Drawn) node -> aux;
    if (drawn -> type != DrawnNode)
	return;

    moved_figure = figure;
    DoMoveNode (node, False);
}


void MoveNodeAP ( )
{
    char       *status;
    struct node dummy;
    Item        found;


    if ((status = GetTextNumber (&dummy.number)) != NULL)
	return;

    found = TreeSearch (problem.node_tree, &dummy);
    if (found == NULL) {
	error ("Node %d does not exist.", dummy.number);
	return;
    }

    DoMoveNode (found, False);
}


void MoveNodeNumber ( )
{
    SetEditMode ( );
    SetShellTitle (error_dialog -> shellwidget, "Move Node");
    ChangeStatusLine ("Select node: ", True);

    XtRemoveAllCallbacks (drawing, XtNbuttonCallback);
    XtAddCallback (drawing, XtNbuttonCallback, MoveNodeCB, NULL);

    XtOverrideTranslations (entry,
	XtParseTranslationTable ("<Key>Return: MoveNodeAP()"));
}
